------------------------------------------------------
-- Status of clean program currently being executed
----------------------------------------------------

require("Util")

CLEANPROG_STATE_IDLE   = 0 -- not running
CLEANPROG_STATE_RUN    = 1 -- running
CLEANPROG_STATE_DONE   = 2 -- done
CLEANPROG_STATE_CANCEL = 3 -- canceled
CLEANPROG_STATE_PAUSED = 4 -- paused

-- agreed cleaning step names we have to do something
local M_CLEAN_STEP_SOAP_COOLING       = "Soap Cooling"
local M_CLEAN_STEP_COOLING_OPEN_WAIT  = "Cooling Open Wait"
local M_CLEAN_STEP_COOLING_OPEN       = "Cooling Open"
local M_CLEAN_STEP_COOLING_CLOSE_WAIT = "Cooling Close Wait"
local M_CLEAN_STEP_PLACE_DETERGENT    = "Place Detergent"
local M_CLEAN_STEP_WAIT_START_LATER   = "Wait"
local M_CLEAN_STEP_COOLING            = "Cooling"

--[[ Active clean program class ]]--

-- active clean program  singleton
ActiveCleanProg = {}
ActiveCleanProg.__index = ActiveCleanProg

ActiveCleanProg.name = "Cleaning"
ActiveCleanProg.state = CLEANPROG_STATE_IDLE
ActiveCleanProg.total_time = 0             -- total duration in sec
ActiveCleanProg.total_passed = 0           -- total passed in sec
ActiveCleanProg.step = 0                   -- step idx
ActiveCleanProg.stepno = -1                -- stepno (not unique)
ActiveCleanProg.stepdesc = nil             -- step description
ActiveCleanProg.manually_paused = false    -- paused due to user action?
ActiveCleanProg.timestamp = 0              -- scheduled time
ActiveCleanProg.target_temp = 0            -- target temperature


--
-- set the name of clean program
--
function ActiveCleanProg:set_name(iName)
 self.name = iName
end

--
-- start the clean program
--
function ActiveCleanProg:start(iTotalTime)
  self.state = CLEANPROG_STATE_RUN
  self.total_time = iTotalTime
  self.total_passed = 0
  self.step = 0
  self.stepno = -1
  self.stepdesc = nil
  self.manually_paused = false

  gre.send_event_data("clean_start", "4u1 total_time 1s0 name", {
    total_time = self.total_time,
    name = self.name
  })
end

--
-- check if initial cleaning is active (cool to get temperature < 100°C, before placing detergent)
--
function ActiveCleanProg:is_soap_cooling()
  return self.stepdesc == M_CLEAN_STEP_SOAP_COOLING or
         self.stepdesc == M_CLEAN_STEP_COOLING_OPEN_WAIT or
         self.stepdesc == M_CLEAN_STEP_COOLING_OPEN or
         self.stepdesc == M_CLEAN_STEP_COOLING_CLOSE_WAIT
end

--
-- check if waiting for soap to be placed
--
function ActiveCleanProg:is_waiting_for_detergent()
  return self.stepdesc == M_CLEAN_STEP_PLACE_DETERGENT
end

--
-- check if the cleaning program is scheduled to start later
--
function ActiveCleanProg:is_scheduled()
  return self.stepdesc == M_CLEAN_STEP_WAIT_START_LATER
end

--
-- check if still cooling down before actual clean program start
--
function ActiveCleanProg:is_cooling()
  return self.stepdesc == M_CLEAN_STEP_COOLING
end

--
-- get the timestamp at which the cleaning program is scheduled (0 for instant)
--
function ActiveCleanProg:scheduled_time()
  return self.timestamp
end

--
-- set the timestamp at which the cleaning program is scheduled (0 for instant)
--
function ActiveCleanProg:set_scheduled_time(timestamp)
  self.timestamp = timestamp
end

--
-- is the recipe paused because of user action? (pause button)
--
function ActiveCleanProg:is_manually_paused()
  return self.state == CLEANPROG_STATE_PAUSED and self.manually_paused
end

--
--  reset
--
function ActiveCleanProg:manually_paused_reset()
  self.manually_paused = false
end

--
-- action on passed time update
--
function ActiveCleanProg:passedtime(iTotalPassed)
  self.total_passed = iTotalPassed
  self.state = CLEANPROG_STATE_RUN

  gre.send_event_data("clean_progress",  "4u1 total_passed 4u1 total_time", {
    total_passed = self.total_passed,
    total_time = self.total_time
  })
end

--
-- action when a new step is sheduled
--
function ActiveCleanProg:newstep(iStepNo, iStepDesc)
  self.step = self.step + 1
  self.stepno = iStepNo
  self.stepdesc = iStepDesc

  gre.send_event_data("clean_step",  "1u1 step 1s0 stepdesc", {
    step = self.step,
    stepdesc = self.stepdesc
  })
end

--
-- action when clean program is done
--
function ActiveCleanProg:done()
  -- when a cancel has occur an io_clean_done is send after the io_clean_cancel
  -- suppress this
  if self.state ~= CLEANPROG_STATE_CANCEL then
    self.state = CLEANPROG_STATE_DONE
    gre.send_event_data("clean_finished",  "1u1 canceled", { canceled = 0 })
  end
end

--
-- action when clean program is canceled
--
function ActiveCleanProg:cancel()
  self.state = CLEANPROG_STATE_CANCEL
  gre.send_event_data("clean_finished",  "1u1 canceled", { canceled = 1 })
end

--
-- manually pause the cleaning program
--
function ActiveCleanProg:pause()
  self.manually_paused = true
  Event:pause_recipe()
end

--
-- action when clean program is paused
--
function ActiveCleanProg:on_paused()
  self.state = CLEANPROG_STATE_PAUSED
  gre.send_event("clean_paused")
end

--
-- continue after pause
--
function ActiveCleanProg:on_continue()
  self.manually_paused = false
  -- set state to RUN and refresh
  ActiveCleanProg:passedtime(self.total_passed)
end


--
-- are we currently cleaning?
--
function ActiveCleanProg:is_cleaning()
  return (self.state == CLEANPROG_STATE_RUN) or (self.state == CLEANPROG_STATE_PAUSED)
end

--
-- get the clean target temperature
--
function ActiveCleanProg:get_target_temp()
  return self.target_temp
end

--[[ Event handlers ]]--

--
-- handle io_clean_cancel event
--
--- @param gre#context mapargs
function CB_OnCleanCancel(mapargs)
   ActiveCleanProg:cancel()
end

--
-- handle io_clean_done event
--
--- @param gre#context mapargs
function CB_OnCleanDone(mapargs)
  ActiveCleanProg:done()
end

--
-- handle io_clean_info event
--
--- @param gre#context mapargs
function CB_OnCleanInfo(mapargs)
  local totaltime = mapargs.context_event_data.totaltime
  ActiveCleanProg:start(totaltime)
end

--
--  handle io_clean_step event
--
--- @param gre#context mapargs
function CB_OnCleanStep(mapargs)
  local stepno = mapargs.context_event_data.stepno
  local stepdesc = mapargs.context_event_data.stepdesc
  ActiveCleanProg:newstep(stepno, stepdesc)
end

--
--  handle io_clean_target_temp event
--
--- @param gre#context mapargs
function CB_OnCleanTargetTemp(mapargs)
  local temp = mapargs.context_event_data.temp
  ActiveCleanProg.target_temp = temp
end

--
--  handle io_clean_passed event
--
-- @param gre#context mapargs
function CB_OnCleanPassed(mapargs)
  local totalpassed = mapargs.context_event_data.totalpassed
  ActiveCleanProg:passedtime(totalpassed)
end

--
-- handle io_clean_paused event
--
function CB_OnCleanPaused(mapargs)
  ActiveCleanProg:on_paused()
end

--
-- handle "io_clean_resume" event
-- resuming a clean program stopped due to power loss
--
-- @param gre#context mapargs
function CB_OnCleanResume(mapargs)
  local name = mapargs.context_event_data.name
  local scheduled_timestamp = mapargs.context_event_data.timestamp

  ActiveCleanProg:set_name(name)
  ActiveCleanProg.state = CLEANPROG_STATE_RUN
  ActiveCleanProg.manually_paused = false
  ActiveCleanProg.timestamp = scheduled_timestamp

  gre.send_event_data("clean_resume", "1s0 name", { name = name } )
  gre.send_event("cleaning_screen")
end
