------------------------------------------
-- Status of recipe currently being cooked
------------------------------------------

require("Util")

local RECIPE_STATE_START     = 0 -- recipe hasn't started
local RECIPE_STATE_PREHEAT   = 1 -- oven is preheating
local RECIPE_STATE_LOADING   = 2 -- waiting for user to put chicken in the oven
local RECIPE_STATE_RECIPE    = 3 -- recipe is cooking, new step started
local RECIPE_STATE_CLIMASAFE = 4 -- clima-safe mode entered
local RECIPE_STATE_KEEPWARM  = 5 -- keeping warm
local RECIPE_STATE_DONE      = 6 -- recipe is done
local RECIPE_STATE_IDLE      = 7 -- recipe not active (anymore)

local RECIPE_STATE_SCHEDULED = 8 -- recipe is scheduled to start on a set time.

--[[

State transition table (rows: from state, colums: to state)

            +-------+---------+---------+--------+-----------+----------+------+------+
            | START | PREHEAT | LOADING | RECIPE | CLIMASAFE | KEEPWARM | DONE | IDLE |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| SCHEDULED |       |    V    |         |   V    |           |          |      |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| START     |       |    V    |         |   V    |           |          |      |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| PREHEAT   |       |         |    V    |        |           |          |      |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| LOADING   |       |         |         |   V    |           |          |      |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| RECIPE    |       |         |         |   V    |     V     |    V     |  V   |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| CLIMASAFE |       |         |         |        |           |    V     |  V   |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| KEEPWARM  |       |         |         |        |           |          |  V   |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| DONE      |       |         |         |   V    |           |          |      |  V   |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+
| IDLE      |       |         |         |        |           |          |      |      |
+-----------+-------+---------+---------+--------+-----------+----------+------+------+

--]]

--[[ Active recipe class ]]--

-- active recipe singleton
ActiveRecipe = {}
ActiveRecipe.__index = ActiveRecipe

ActiveRecipe.name = nil                 -- name of the currently active recipe
ActiveRecipe.recipe = nil               -- currently active recipe
ActiveRecipe.state = RECIPE_STATE_DONE  -- active recipe state
ActiveRecipe.total_time = 0             -- total recipe duration (minutes, including autocorrect)
ActiveRecipe.active_step = 0            -- active recipe step
ActiveRecipe.cook_correction = 0        -- cook correction
ActiveRecipe.extra_time = 0             -- current added time
ActiveRecipe.extra_time_accu = 0        -- accumulated added time
ActiveRecipe.seconds_passed = 0         -- total seconds passed in the prev steps
ActiveRecipe.step_passed = 0            -- seconds passed in the current step
ActiveRecipe.paused = false             -- is the recipe paused?
ActiveRecipe.canceled = false           -- is canceled
ActiveRecipe.manually_paused = false    -- is manually paused by user?
ActiveRecipe.reference_cook = false     -- reference cook (calibrating)?
ActiveRecipe.heatup = false             -- heatup recipe
ActiveRecipe.timestamp = 0              -- scheduled time (starts recipe on set time)
--
-- calculate the total time the recipe will take in minutes
--
local function calculateStepsTime(steps)
  local total_time = 0
  for _, step in ipairs(steps) do
    total_time = total_time + step.time
  end
  return total_time
end

--
-- set the currently active recipe
--
function ActiveRecipe:set_recipe(recipe, name, reference_cook)
  self.name = name
  self.recipe = table_copy(recipe)
  self.total_time = calculateStepsTime(self.recipe.steps)
  self.active_step = 1
  self.cook_correction = 0
  self.extra_time = 0
  self.extra_time_accu = 0
  self.state = RECIPE_STATE_START
  self.seconds_passed = 0
  self.step_passed = 0
  self.paused = false
  self.canceled = false
  self.manually_paused = false
  self.reference_cook = reference_cook
  self.heatup = false
end

--
-- get the timestamp at which the recipe is scheduled (0 for instant start)
--
function ActiveRecipe:scheduled_time()
  return self.timestamp
end

-- set the timestamp at which the recipe is scheduled (0 for instant start)
--
function ActiveRecipe:set_scheduled_time(timestamp)
  if timestamp > 0 then
    self.state = RECIPE_STATE_SCHEDULED
  end
  self.timestamp = timestamp
end

--
-- is a recipe waiting to start on a scheduled time ?
--
function ActiveRecipe:is_scheduled()
  if ActiveRecipe:scheduled_time() == 0 then
    return false
  else
    return self.state == RECIPE_STATE_SCHEDULED
  end
end

--
-- as calculateStepsTime calculate the total time the recipe will take in minutes
-- but up to given stepIdx ( helper prep_resume() )
--
local function calculateStepsTimeUpTo(steps, maxIdx)
  local total_time = 0
  for idx, step in ipairs(steps) do
    if idx >= maxIdx then
       break
    else
      total_time = total_time + step.time
    end
  end
--  print("calculateStepsTimeUpTo " .. total_time)
  return total_time
end

--
--  after called set_recipe()
--  prepare to resume after powerdown
--
function ActiveRecipe:prep_resume(state, stepIdx, step_passed)
 if state == RECIPE_STATE_RECIPE then
   self.seconds_passed = 60*calculateStepsTimeUpTo(self.recipe.steps, stepIdx)
 end
 if state == RECIPE_STATE_KEEPWARM then
   self.seconds_passed = 60*calculateStepsTime(self.recipe.steps)
 end

end

--
-- manually pause the recipe
--
function ActiveRecipe:pause()
  self.manually_paused = true
  Event:pause_recipe()
end

--
-- is the recipe currently paused?
--
function ActiveRecipe:is_paused()
  return self.recipe and self.paused
end

--
-- is the recipe paused because of user action? (pause button)
--
function ActiveRecipe:is_manually_paused()
  return self.recipe and self.paused and self.manually_paused
end

--
-- is the active recipe being calibrated?
--
function ActiveRecipe:is_reference_cook()
  return self.recipe and self.reference_cook
end

--
--  reset
--
function ActiveRecipe:manually_paused_reset()
  self.manually_paused = false
end

--
-- get the currently active recipe
--
function ActiveRecipe:get_recipe()
  return self.recipe
end

--
-- get the total recipe time
--
function ActiveRecipe:get_time()
  return self.total_time
end

--
-- get the active step number
--
function ActiveRecipe:get_active_step()
  return self.active_step
end

--
-- are we currently preheating?
--
function ActiveRecipe:is_preheating()
  return self.state == RECIPE_STATE_PREHEAT
end

--
-- is the active recipe in the loading state?
--
function ActiveRecipe:is_loading()
  return self.state == RECIPE_STATE_LOADING
end

--
-- are we currently holding?
--
function ActiveRecipe:is_holding()
  return self.state == RECIPE_STATE_KEEPWARM
end


--
-- are we currently cooking?
--
function ActiveRecipe:is_cooking()
  return (self.state >= RECIPE_STATE_PREHEAT) and (self.state <= RECIPE_STATE_KEEPWARM)
end

--
-- done cooking?
--
function ActiveRecipe:is_done()
  return self.state == RECIPE_STATE_DONE
end

--
-- request extra time to be added
--
function ActiveRecipe:add_time(seconds)
  Event:live_set("ADD_TIME", seconds)
end

--
-- cancel active recipe timer
--
function ActiveRecipe:cancel()
  self.canceled = true
  self.state = RECIPE_STATE_IDLE
  self.recipe = nil
  self.timestamp = 0
  gre.send_event("recipe_cancel")
end

--
-- passed time handler
--
function ActiveRecipe:on_passedtime(time)
  self.step_passed = time

  gre.send_event_data("recipe_step_progress", "2u1 total_passed 2u1 step_passed", {
    total_passed = self.seconds_passed + self.step_passed,
    step_passed = self.step_passed
  })
end
--
-- recipe state changed to preheating
--
local function OnRecipeStatePreheat(self)
  self.active_step = 1
  gre.send_event("recipe_preheat")
end

--
-- recipe state changed to waiting for oven loading
--
local function OnRecipeStateLoading(self)
  gre.send_event("recipe_preheat_done")
end

--
-- recipe state changed to recipe (or new step)
--
local function OnRecipeStateRecipe(self, step)

  -- only update when a new step -- not when step is rerun due to post run with add time
  if self.active_step < step then
    self.seconds_passed = self.seconds_passed + self.step_passed
    self.active_step = step
    self.step_passed = 0
  end

  gre.send_event_data("recipe_step", "1u1 step", { step = self.active_step })
end

--
-- recipe state changed to clima-safe
--
local function OnRecipeStateClimasafe(self)
  gre.send_event("recipe_climasafe")
end

--
-- recipe state changed to keep warm
--
local function OnRecipeStateKeepwarm(self)
  self.active_step = #self.recipe.steps
  gre.send_event("recipe_holding")
end

--
-- recipe state changed to done
--
local function OnRecipeStateDone(self)
  if self.reference_cook and self.recipe and not self.canceled then
    Recipes:get(self.name).calibrated = true
  end
  gre.send_event("recipe_done")
end

--
-- recipe state changed to idle
--
local function OnRecipeStateIdle(self)
  self.recipe = nil
  gre.send_event("recipe_idle")
end

--
-- recipe state changed to wait for start.
--
local function OnRecipeStateScheduled(self)
  print("OnRecipeStateScheduled()")
end

-- mapping of states to handlers
local gStateHandlerTable = {
  [RECIPE_STATE_PREHEAT  ] = OnRecipeStatePreheat,
  [RECIPE_STATE_LOADING  ] = OnRecipeStateLoading,
  [RECIPE_STATE_RECIPE   ] = OnRecipeStateRecipe,
  [RECIPE_STATE_CLIMASAFE] = OnRecipeStateClimasafe,
  [RECIPE_STATE_KEEPWARM ] = OnRecipeStateKeepwarm,
  [RECIPE_STATE_DONE     ] = OnRecipeStateDone,
  [RECIPE_STATE_IDLE     ] = OnRecipeStateIdle,
  [RECIPE_STATE_SCHEDULED] = OnRecipeStateScheduled
}

--
-- state change handler
--
function ActiveRecipe:on_state_change(new_state, step)
  self.state = new_state
  local handler_callback = assert(gStateHandlerTable[new_state], "Invalid recipe state.")
  handler_callback(self, step)
end

--
-- cook correction handler
--
function ActiveRecipe:on_cookcorrect(delta)
  self.cook_correction = delta
  gre.send_event_data("recipe_cookcorrect", "2s1 delta", { delta = self.cook_correction })
end

--
-- pause event handler
--
function ActiveRecipe:on_pause()
  self.paused = true
end

--
-- continue event handler
--
function ActiveRecipe:on_continue()
  self.paused = false
  self.manually_paused = false
end

--
-- extra time was added
--
function ActiveRecipe:on_time_added(current, accu)
  self.extra_time = current
  self.extra_time_accu = accu
  gre.send_event_data("recipe_timeadded", "2s1 extra_time", { extra_time = self.extra_time })
end

--
-- turn on heatup flag to indicate a heatup recipe
--
function ActiveRecipe:set_heatup()
   self.heatup = true
end

--
-- is a heatup recipe?
--
function ActiveRecipe:is_heatup()
   return self.heatup
end

--
-- set state to idle
--
function ActiveRecipe:set_idle()
   ActiveRecipe:on_state_change(RECIPE_STATE_IDLE, 0)
end


--[[ Event handlers ]]--

--
-- active recipe state change handler
--
--- @param gre#context mapargs
function CB_OnRecipeStateChange(mapargs)
  ActiveRecipe:on_state_change(mapargs.context_event_data.state, mapargs.context_event_data.step)
end

--
-- passed time handler
--
--- @param gre#context mapargs
function CB_OnPassedTime(mapargs)
  ActiveRecipe:on_passedtime(mapargs.context_event_data.time)
end

--
-- cook correction handler
--
--- @param gre#context mapargs
function CB_OnCookCorrect(mapargs)
  ActiveRecipe:on_cookcorrect(mapargs.context_event_data.delta)
end

--
-- io_paused event handler
--
function CB_OnRecipePause()
  ActiveRecipe:on_pause()
end

--
-- cancel recipe event handler
--
function CB_OnRecipeCancel()
  ActiveRecipe:cancel()
end

--
-- ADD_TIME live variable changed
--
function CB_OnRecipeTimeAdded(mapargs)
  if mapargs.context_event_data.id == "ADD_TIME" then
    local added_time = Settings:get("ADD_TIME")

    -- ADD_TIME_ACCU is always refreshed before ADD_TIME
    local added_time_accu = Settings:get("ADD_TIME_ACCU")
    ActiveRecipe:on_time_added(added_time, added_time_accu)
  end
end

--
-- handle "io_recipe_resume" event
--
-- @param gre#context mapargs
function CB_OnRecipeResume(mapargs)
  local name = mapargs.context_event_data.name
  local state = mapargs.context_event_data.state
  local stepIdx = mapargs.context_event_data.step
  local step_passed = mapargs.context_event_data.time

  local recipe = Recipes:get(name)
  if recipe  ~= nil then
    ActiveRecipe:set_recipe(recipe, name, false)
    ActiveRecipe:prep_resume(state, stepIdx, step_passed)
    gre.send_event("resume_recipe") -- CBCOOK_OnScreenShow
  else
    Event:action("CANCEL_ACTION")
  end
end

-- recipe used to heatup
-- note that the hold flags is set to disable extra time dialog...
local gHeatUpRecipe = {
    calibrated = false,
    icon = nil,
    steps = {
      { name = 'Heating Up', temp = 175, preheat = true, hold = true, time = 0 }
    }
}

--
-- handle "io_heatup_start" event
--
-- @param gre#context mapargs
function CB_OnHeatUp_Start(mapargs)
  local heatuptemp = mapargs.context_event_data.heatuptemp

  ActiveRecipe:set_recipe(gHeatUpRecipe, i18n:get("HEATUP_PROGRAM"), false)
  ActiveRecipe:set_heatup()

  gre.send_event("resume_recipe") -- CBCOOK_OnScreenShow

  ActiveRecipe:on_state_change(RECIPE_STATE_PREHEAT, 1)
  gre.send_event("cook_screen") -- as we are started from settings menu item

end

--
-- handle "io_heatup_done" event
--
-- @param gre#context mapargs
function CB_OnHeatUp_Done(mapargs)
  ActiveRecipe:on_state_change(RECIPE_STATE_DONE, 1)
  gre.send_event_data("heatup_done", "2u1 heatuptime 1s0 varname", {
                                    heatuptime = mapargs.context_event_data.heatuptime,
                                    varname = mapargs.context_event_data.varname
    })
end
