------------------------------------------------
-- implementation of the cleaning program screen
------------------------------------------------

require("events")
require("ActiveCleanProg")
require("Settings")
require("Temperature")
require("Time")
require("CleanTotalTimes")

local BUBBLES_CONTROL = "Cleaning_Layer.CL_Bubbles_Img.bubblesImg"
local BUBBLES_IMAGE = "images/bellenloop-%d.png"

local gFrame = 1
local gNumFrames = 10
local gCleaningType = ""
local gSwapTimer = nil
local gTemperature = Temperature.new{temp = 0}

-- table mapping our type to controller clean mode names
local gType2NameTable = {
    [1] = "CLEAN_DAILY",
    [2] = "CLEAN_QUICK",
    [3] = "CLEAN_DEEP",
    [4] = "CLEAN_RINSE"
  }

--
-- return cleaning name fro given type
--
local function getCleanName(type)
  local name = gType2NameTable[type]
  return name and name or ""
end
--
-- set percentage done on the progress bar
--
local function setProgress(percentage)
  local data = {}

  if percentage == 0 then
    data["Cleaning_Layer.status_bar.grd_hidden"] = true
  else
    local x_pos = gre.get_value("Cleaning_Layer.status_bar.Body_Img.grd_x")
    local width = (percentage * (244 - (2 * 8))) / 100

    data["Cleaning_Layer.status_bar.grd_hidden"] = false
    data["Cleaning_Layer.status_bar.Body_Img.grd_width"] = width
    data["Cleaning_Layer.status_bar.Right_Img.grd_x"] = x_pos + width
  end

  gre.set_data(data)
end

--
-- set the next frame of the bubbles animation, called from a timer
--
function CBCleaning_NextFrame(mapargs)
  local data = {}

  data[BUBBLES_CONTROL] = string.format(BUBBLES_IMAGE, gFrame)
  gre.set_data(data)

  gFrame = gFrame + 1
  if gFrame > gNumFrames then
    gFrame = 1
  end
end

--
-- start/stop bubble animation depending on state
--
local function CheckBubblesAnimation()
  local animation_is_active = gre.get_value("CLEANING_Screen.bubbles_animation_active") == 1
  local animation_should_be_active = ActiveCleanProg.state == CLEANPROG_STATE_RUN
                                     and not ActiveCleanProg:is_soap_cooling()
                                     and not ActiveCleanProg:is_waiting_for_detergent()
                                     and not ActiveCleanProg:is_scheduled()
                                     and not ActiveCleanProg:is_cooling()

  if animation_is_active ~= animation_should_be_active then
    gre.send_event(animation_should_be_active and "start_bubbles_animation" or "stop_bubbles_animation")
  end
end

--
-- handler for updating the cleaning program progress
--
function CBCleaning_OnCleanProgProgress(mapargs)
  local percentage_done = (mapargs.context_event_data.total_passed * 100) /  mapargs.context_event_data.total_time
  local _, _, _, h, min, _ = utc_time(mapargs.context_event_data.total_time - mapargs.context_event_data.total_passed)
  local finish_time = format_time((ActiveCleanProg:scheduled_time() or now()) + (h * 3600) + (min * 60))
  setProgress(percentage_done)

  local data = {}

  data["Cleaning_Layer.remaining_time_group.Remaining_Text.text"] = string.format("%02d:%02d", h, min)
  data["Cleaning_Layer.finish_time_group.Finish_Text.text"] = finish_time

  if gre.get_value("Cleaning_Layer.CL_Play_Btn.grd_hidden") == 0 then
    data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = false
    data["Cleaning_Layer.CL_Play_Btn.grd_hidden"] = true
  end

  gre.set_data(data)

  CheckBubblesAnimation()
end

--
-- handle cleaning progam step change
--
function CBCleaning_OnCleanStep(mapargs)
  local data = {}

  local target_temp = Temperature.new{temp = ActiveCleanProg:get_target_temp()/10.0}

  data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = ActiveCleanProg.state == CLEANPROG_STATE_PAUSED or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling()
  data["Cleaning_Layer.CL_Play_Btn.grd_hidden"] = ActiveCleanProg.state ~= CLEANPROG_STATE_PAUSED or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling()
  data["Cleaning_Layer.CL_Cancel_Btn.grd_hidden"] = not (ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling())
  data["Cleaning_Layer.cooldown_img.grd_hidden"] = not (ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling())
  data["Cleaning_Layer.CL_Bubbles_Img.grd_hidden"] = ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling() or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_waiting_for_detergent()
  data["Cleaning_Layer.icons_group_place_soap.grd_hidden"] = not ActiveCleanProg:is_soap_cooling()
  data["Cleaning_Layer.icons_group_dont_open.grd_hidden"] = ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_waiting_for_detergent()
  data["Cleaning_Layer.delayedstart_ctrl.grd_hidden"] = not ActiveCleanProg:is_scheduled()
  data["Cleaning_Layer.temperature_ctrl.grd_hidden"] = not (ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling())
  data["Cleaning_Layer.target_temp_ctrl.text"] = target_temp:to_string()

  if ActiveCleanProg:is_scheduled() then
    local timestamp, ampm = format_time(ActiveCleanProg:scheduled_time())
    if ampm then timestamp = string.format("%s %s", timestamp, ampm) end
    data["Cleaning_Layer.delayedstart_ctrl.text"] = timestamp
  end

  gre.set_data(data)

  CheckBubblesAnimation()
end

--
-- cleaning paused
--
function CBCleaning_OnCleanPaused()
  gre.send_event("stop_bubbles_animation")

  local data = {}
  data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = true
  data["Cleaning_Layer.CL_Play_Btn.grd_hidden"] = false
  gre.set_data(data)
end

--
-- swap between time remaining and finishes at
--
function CBCleaning_SwapRemainingArrival(mapargs)
  local bRemainingTimeActive = not mapargs.context_event_data.force_remaining and gre.get_value("Cleaning_Layer.remaining_time_group.grd_hidden") == 0

  local data = {}
  data["Cleaning_Layer.remaining_time_group.grd_hidden"] = bRemainingTimeActive
  data["Cleaning_Layer.finish_time_group.grd_hidden"] = not bRemainingTimeActive
  gre.set_data(data)

  if gSwapTimer == nil then
    gSwapTimer = gre.timer_set_timeout(function()
      gSwapTimer = nil
      CBCleaning_SwapRemainingArrival({ context_event_data = { force_remaining = true } })
    end, 5000)
  end
end

--
-- prepare cleaning program selection dialog
--
function Cleaning_ShowProgramSelectionDialog()
  CBCleaning_StartTime_Prepare()

  local bDailyCleanEnabled = Settings:get_def("CLEAN_DAILY_ENABLED", 1) == 1
  local bQuickCleanEnabled = Settings:get_def("CLEAN_QUICK_ENABLED", 1) == 1
  local bDeepCleanEnabled = Settings:get_def("CLEAN_DEEP_ENABLED", 1) > 0 -- backwards compatible
  local bRinseCleanEnabled = false -- obsolate

  local cell_height = gre.get_table_cell_attrs("StartCleaning_Overlay.cleanselection_table", 1, 1, "height").height

  local data = {}
  local index = 0
  local type = 0

  data["CLEANING_Screen.StartCleaning_Overlay.grd_hidden"] = false
  data["CLEANING_Screen.DimFilter_Layer.grd_hidden"] = false
  data["CLEANING_Screen.DoorOpen_Layer.grd_hidden"] = true
  data["CLEANING_Screen.DoorOpen_Cleaning_Layer.grd_hidden"] = true

  if bDailyCleanEnabled then
    index = index + 1
    type = 1
    data["StartCleaning_Overlay.cleanselection_table.text."..index..".1"] = gre.get_value("Cleaning_Layer.icons_group_standard.Recipe_Text.text")
    data["StartCleaning_Overlay.cleanselection_table.detergent_image."..index..".1"] = image_path("icon-clean-cartridge-small.png")
    data["StartCleaning_Overlay.cleanselection_table.totaltime."..index..".1"] =  CleaningTotalTimes:get(getCleanName(type))
    data["StartCleaning_Overlay.cleanselection_table.type."..index..".1"] = type
  end
  if bQuickCleanEnabled then
    index = index + 1
    type = 2
    data["StartCleaning_Overlay.cleanselection_table.text."..index..".1"] = gre.get_value("Cleaning_Layer.icons_group_quick.Recipe_Text.text")
    data["StartCleaning_Overlay.cleanselection_table.detergent_image."..index..".1"] = image_path("icon-clean-sachet-small.png")
    data["StartCleaning_Overlay.cleanselection_table.totaltime."..index..".1"] = CleaningTotalTimes:get(getCleanName(type))
    data["StartCleaning_Overlay.cleanselection_table.type."..index..".1"] = type
  end
  if bDeepCleanEnabled then
    index = index + 1
    type = 3
    data["StartCleaning_Overlay.cleanselection_table.text."..index..".1"] = gre.get_value("Cleaning_Layer.icons_group_deep.Recipe_Text.text")
    data["StartCleaning_Overlay.cleanselection_table.detergent_image."..index..".1"] = image_path("icon-clean-tablet-small.png")
    data["StartCleaning_Overlay.cleanselection_table.totaltime."..index..".1"] = CleaningTotalTimes:get(getCleanName(type))
    data["StartCleaning_Overlay.cleanselection_table.type."..index..".1"] = type
  end
  if bRinseCleanEnabled then
    index = index + 1
    type = 4
    data["StartCleaning_Overlay.cleanselection_table.text."..index..".1"] = gre.get_value("Cleaning_Layer.icons_group_rinse.Recipe_Text.text")
    data["StartCleaning_Overlay.cleanselection_table.detergent_image."..index..".1"] = image_path("icon-clean-x262-y740.png")
    data["StartCleaning_Overlay.cleanselection_table.totaltime."..index..".1"] = CleaningTotalTimes:get(getCleanName(type))
    data["StartCleaning_Overlay.cleanselection_table.type."..index..".1"] = type
  end
  local table_height = index * cell_height
  data["StartCleaning_Overlay.cleanselection_table.grd_y"] = (800 - table_height) / 2
  data["StartCleaning_Overlay.cleanselection_table.grd_height"] = table_height

  gre.set_data(data)
  gre.set_table_attrs("StartCleaning_Overlay.cleanselection_table", {rows=index})
end

--
-- cleaning finished
--
function CBCleaning_OnCleanFinished(mapargs)
  gre.send_event("stop_bubbles_animation")

  local canceled = mapargs.context_event_data.canceled == 1
  if canceled then
    Cleaning_ShowProgramSelectionDialog()
  else
    local data = {}
    data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = true
    data["Cleaning_Layer.finish_img.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_dont_open.grd_hidden"] = true
    gre.set_data(data)
  end
  Control_SetButtons(true, true, true, true, true, true, true)
end

--
-- display the active/paused cleaning program
--
local function Cleaning_ShowCleaningProgram(mapargs)
  local data = {}

  data["Cleaning_Layer.delayedstart_ctrl.grd_hidden"] = true
  data["Cleaning_Layer.cooldown_img.grd_hidden"] = true
  data["Cleaning_Layer.finish_img.grd_hidden"] = true
  data["Cleaning_Layer.remaining_time_group.grd_hidden"] = false
  data["Cleaning_Layer.finish_time_group.grd_hidden"] = true
  data["CLEANING_Screen.StartCleaning_Overlay.grd_hidden"] = true
  data["CLEANING_Screen.DimFilter_Layer.grd_hidden"] = true
  data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = true
  data["Cleaning_Layer.CL_Play_Btn.grd_hidden"] = true
  data["Cleaning_Layer.CL_Cancel_Btn.grd_hidden"] = true
  data["Cleaning_Layer.icons_group_place_soap.grd_hidden"] = true
  data["Cleaning_Layer.icons_group_dont_open.grd_hidden"] = true
  data["Cleaning_Layer.temperature_ctrl.grd_hidden"] = true
  data["Cleaning_Layer.target_temp_ctrl.grd_hidden"] = true

  if ActiveCleanProg.name == "CLEAN_DAILY" then
    data["Cleaning_Layer.icons_group_standard.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_quick.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_deep.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_rinse.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_generic.grd_hidden"] = true
    data["Cleaning_Layer.status_bar.Left_Img.image_left"] = "images/statusbar-kop-links-x118-y633.png"
    data["Cleaning_Layer.status_bar.Body_Img.image_body"] = "images/statusbar-bodypixel-y633.png"
    data["Cleaning_Layer.status_bar.Right_Img.image_right"] = "images/statusbar-kop-rechts-y633.png"
  elseif ActiveCleanProg.name == "CLEAN_QUICK" then
    data["Cleaning_Layer.icons_group_standard.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_quick.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_deep.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_rinse.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_generic.grd_hidden"] = true
    data["Cleaning_Layer.status_bar.Left_Img.image_left"] = "images/statusbar-quick-kop-links-x118-y633.png"
    data["Cleaning_Layer.status_bar.Body_Img.image_body"] = "images/statusbar-quick-bodypixel-y633.png"
    data["Cleaning_Layer.status_bar.Right_Img.image_right"] = "images/statusbar-quick-kop-rechts-y633.png"
  elseif ActiveCleanProg.name == "CLEAN_DEEP" then
    data["Cleaning_Layer.icons_group_standard.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_quick.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_deep.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_rinse.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_generic.grd_hidden"] = true
    data["Cleaning_Layer.status_bar.Left_Img.image_left"] = "images/statusbar-kop-links-x118-y633.png"
    data["Cleaning_Layer.status_bar.Body_Img.image_body"] = "images/statusbar-bodypixel-y633.png"
    data["Cleaning_Layer.status_bar.Right_Img.image_right"] = "images/statusbar-kop-rechts-y633.png"
   elseif ActiveCleanProg.name == "CLEAN_RINSE" then
    data["Cleaning_Layer.icons_group_standard.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_quick.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_deep.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_rinse.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_generic.grd_hidden"] = true
    data["Cleaning_Layer.status_bar.Left_Img.image_left"] = "images/statusbar-kop-links-x118-y633.png"
    data["Cleaning_Layer.status_bar.Body_Img.image_body"] = "images/statusbar-bodypixel-y633.png"
    data["Cleaning_Layer.status_bar.Right_Img.image_right"] = "images/statusbar-kop-rechts-y633.png"
  else
    data["Cleaning_Layer.icons_group_standard.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_quick.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_deep.grd_hidden"] = true
    data["Cleaning_Layer.icons_group_generic.grd_hidden"] = false
    data["Cleaning_Layer.icons_group_generic.Recipe_Text.text"] = i18n:get("NLS_"..ActiveCleanProg.name)
    data["Cleaning_Layer.status_bar.Left_Img.image_left"] = "images/statusbar-kop-links-x118-y633.png"
    data["Cleaning_Layer.status_bar.Body_Img.image_body"] = "images/statusbar-bodypixel-y633.png"
    data["Cleaning_Layer.status_bar.Right_Img.image_right"] = "images/statusbar-kop-rechts-y633.png"
  end

  gre.set_data(data)
end

--
-- restore the active cleaning program progress
--
local function Cleaning_RestoreProgramState(mapargs)
  local data = {}

  data["Cleaning_Layer.cooldown_img.grd_hidden"] = not (ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling())
  data["Cleaning_Layer.CL_Bubbles_Img.grd_hidden"] = ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling() or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_waiting_for_detergent()
  data["CLEANING_Screen.StartCleaning_Overlay.grd_hidden"] = true
  data["CLEANING_Screen.DimFilter_Layer.grd_hidden"] = true
  data["Cleaning_Layer.CL_Pause_Btn.grd_hidden"] = ActiveCleanProg.state == CLEANPROG_STATE_PAUSED or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling()
  data["Cleaning_Layer.CL_Play_Btn.grd_hidden"] = ActiveCleanProg.state ~= CLEANPROG_STATE_PAUSED or ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling()
  data["Cleaning_Layer.CL_Cancel_Btn.grd_hidden"] = not (ActiveCleanProg:is_scheduled() or ActiveCleanProg:is_soap_cooling())
  data["Cleaning_Layer.temperature_ctrl.grd_hidden"] = not (ActiveCleanProg:is_soap_cooling() or ActiveCleanProg:is_cooling())
  data["Cleaning_Layer.target_temp_ctrl.grd_hidden"] = true

  setProgress((ActiveCleanProg.total_passed * 100) / ActiveCleanProg.total_time)

  if ActiveCleanProg.state == CLEANPROG_STATE_RUN then
    if ActiveCleanProg:is_scheduled() then
      local timestamp, ampm = format_time(ActiveCleanProg:scheduled_time())
      if ampm then timestamp = string.format("%s %s", timestamp, ampm) end

      data["Cleaning_Layer.delayedstart_ctrl.grd_hidden"] = false
      data["Cleaning_Layer.delayedstart_ctrl.text"] = timestamp
    elseif ActiveCleanProg:is_soap_cooling() then
      data["Cleaning_Layer.icons_group_place_soap.grd_hidden"] = false
    else
      data["Cleaning_Layer.icons_group_dont_open.grd_hidden"] = false
    end
  end

  gre.set_data(data)

  CheckBubblesAnimation()
end

--
-- start bubbles animation if needed
--
function CBCleaning_OnScreenShown(mapargs)
  gre.set_value("CLEANING_Screen.bubbles_animation_active", 0)
  CheckBubblesAnimation()
end

--
-- prepare cleaning screen
--
function CBCleaning_OnScreenShow(mapargs)
  gre.send_event("gra.screen.hold")

  local data = {}
  data["Cleaning_Layer.language_button.image"] = i18n:meta_data(i18n:language()).icon or image_path("vlag-engels x425-y59.png")
  data["Cleaning_Layer.temperature_ctrl.text"] = gTemperature:to_string()
  gre.set_data(data)

  if in_array(ActiveCleanProg.state, {CLEANPROG_STATE_RUN, CLEANPROG_STATE_PAUSED}) then
    -- cleaning program active -> display it
    Cleaning_ShowCleaningProgram()
    Cleaning_RestoreProgramState()

    Control_SetButtons(true, true, true, true, true, true, true)
  else
    -- no cleaning program active -> show cleaning program selection
    Cleaning_ShowProgramSelectionDialog()

    Control_SetButtons(true, true, true, true, true, true, false)
  end

  gre.send_event("gra.screen.release")
end

--
-- handler for when a cleaning program is started
--
function CBCleaning_OnStart(mapargs)
  local duration =  mapargs.context_event_data.total_time
  local name = mapargs.context_event_data.name
  local finish_time = format_time((ActiveCleanProg:scheduled_time() or now()) + duration)
  local _, _, _, h, min, _ = utc_time(duration)

  CBCleaning_OnScreenShow()

  local data = {}
  gCleaningType = name
  data["Cleaning_Layer.remaining_time_group.Remaining_Text.text"] = string.format("%02d:%02d", h, min)
  data["Cleaning_Layer.finish_time_group.Finish_Text.text"] = finish_time
  gre.set_data(data)
end

--
-- leaving screen, cleanup
--
function CBCleaning_OnScreenHide(mapargs)
  if gSwapTimer ~= nil then
    gre.timer_clear_timeout(gSwapTimer)
    gSwapTimer = nil
  end
end

--
-- select a cleaning program
--
function CBCleaning_OnSelectProgram(mapargs)
  local index = mapargs.context_row
  local type = gre.get_value("StartCleaning_Overlay.cleanselection_table.type."..index..".1")

  local start_time = Cleaning_GetStartingTime()
  if start_time > 0 then
    local _, _, _, h, min, _ = utc_time(start_time - now())
    local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
    dlg:set_message(string.format(i18n:get("NLS_CONFIRM_CLEANING_DELAY"), h, min))
    dlg:add_button(i18n:get("NLS_CANCEL"), function() CBStartTime_SelectTime_OnCancel(mapargs) end)
    dlg:add_button(i18n:get("NLS_CONTINUE"), function() Event:can_start_clean(type) end)
    dlg:show()
  else
    Event:can_start_clean(type)
  end
end

--
-- pause cleaning program
--
function CBCleaning_OnPause()
  ActiveCleanProg:pause()
end

--
-- resume cleaning program
--
function CBCleaning_OnPlay()
  Event:action("CONTINUE_ACTION")
end

--
-- cancel cleaning program
--
function CBCleaning_OnCancel()
  if ActiveCleanProg:is_cleaning() then
    -- cancel active program
    Event:cancel_recipe()
  end
end

--
-- send event to turn on light
--
function CBCleaning_LightOn()
  Event:light_on()
end

--
-- send event to turn off light
--
function CBCleaning_LightOff()
  Event:light_off()
end

--
-- handle "clean_resume" event send from CB_OnCleanResume()
--
-- @param gre#context mapargs
function CBCleaning_OnResume(mapargs)
  -- nothing special to do for now
end

--
-- start quick clean
--
local function OnStartProgram(type, name)
  if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
    local dlg = DialogBox.new(DIALOG_TYPE_WARNING)
    dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  else
    gre.set_value("CLEANING_Screen.StartCleaning_Overlay.grd_hidden", true)

    local timestamp = Cleaning_GetStartingTime()
    ActiveCleanProg:set_name(name)
    ActiveCleanProg:set_scheduled_time(timestamp)
    Event:action(name, timestamp)
  end
end

--
-- start a cleaning program
--
local function start_cleaning(type)
  local name = gType2NameTable[type]

  if name then OnStartProgram(type, name) end
end

--
-- received reply from application if cleaning program can be started
--
function CBCleaning_OnCanStartReply(mapargs)
  local type = mapargs.context_event_data.type
  local reply = mapargs.context_event_data.reply
  local message = mapargs.context_event_data.info

  if reply == -1 then
    -- error
    local dlg = DialogBox.new(DIALOG_TYPE_ERROR)
    dlg:set_message(i18n:get(message))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  elseif reply == 0 then
    -- start cleaning program
    start_cleaning(type)
  elseif reply == 1 then
    -- display information message, then start cleaning program
    local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
    dlg:set_message(i18n:get(message))
    dlg:add_button(i18n:get("NLS_CANCEL"))
    dlg:add_button(i18n:get("NLS_CONTINUE"), function() start_cleaning(type) end)
    dlg:show()
  end
end

--
-- temperature update received
--
function CBCleaning_OnTemperatureUpdate(mapargs)
  gTemperature:setC(mapargs.context_event_data.temp / 10.0)

  if mapargs.context_screen == "CLEANING_Screen" then
    gre.set_value("Cleaning_Layer.temperature_ctrl.text", gTemperature:to_string())
  end
end

--
-- language changed
--
function CBCleaning_OnLanguageChanged()
  gre.set_value("Idle_Layer.language_button.image", i18n:meta_data(i18n:language()).icon or image_path("vlag-engels x425-y59.png"))
end

--
-- set clean_*-controls to wanted hidden state
--
local function DoorOpen_Cleaning_Layer_Reset()
 local data = {}
 data["DoorOpen_Cleaning_Layer.clean_cartridge_img.grd_hidden"] = true
 data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_hidden"] = true
 data["DoorOpen_Cleaning_Layer.clean_tablet_or_sachet_img.grd_hidden"] = true
 gre.set_data(data)
end

--
-- handle "io_clean_door_show" event: evt_clean_door_show
--  param flag 0(add detergent) or 1(remove detergent)
--
--  showing the DoorOpen_Layer without rotor and DoorOpen_Cleaning_Layer
--
--  @param gre#context mapargs
function CBCleaning_OnCleanDoorShow(mapargs)

  local name = ActiveCleanProg.name
  local remove_cleaning = (mapargs.context_event_data.flag == 1) and true or false

  -- disable all menu functionality except the standby button
  Control_SetButtons(true, false, false, false, false, false, false)

  local data = {}
  data["CLEANING_Screen.DoorOpen_Layer.grd_hidden"] = false
  data["CLEANING_Screen.DoorOpen_Cleaning_Layer.grd_hidden"] = false
  data["DoorOpen_Layer.rotate_img.grd_hidden"] = true -- without rotor as door is not open

  -- image and position of DoorOpen_Cleaning_Layer depends
  if name == "CLEAN_DAILY" then
    data["DoorOpen_Cleaning_Layer.clean_cartridge_img.grd_hidden"] = false
    data["DoorOpen_Cleaning_Layer.clean_cartridge_img.grd_x"] = remove_cleaning and 275 or 323 -- show 'add' detergent or 'remove' detergent.
    data["DoorOpen_Cleaning_Layer.clean_cartridge_img.grd_y"] = remove_cleaning and 199 or 89  -- show 'add' detergent or 'remove' detergent.

    data["DoorOpen_Cleaning_Layer.clean_tablet_or_sachet_img.grd_hidden"] = true

    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_hidden"] = false
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_x"] = 205
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_y"] = 126
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.image"] = remove_cleaning and image_path("icon-clean-arrow-cartridge_remove.png") or image_path("icon-clean-arrow-cartridge_add.png")

  else   -- assume CLEAN_QUICK or CLEAN_DEEP
    data["DoorOpen_Cleaning_Layer.clean_cartridge_img.grd_hidden"] = true

    data["DoorOpen_Cleaning_Layer.clean_tablet_or_sachet_img.grd_hidden"] = false
    data["DoorOpen_Cleaning_Layer.clean_tablet_or_sachet_img.image"] = (name == "CLEAN_QUICK") and image_path("icon-clean-sachet-small.png") or image_path("icon-clean-tablet-small.png")

    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_hidden"] = false
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_x"] = 205
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.grd_y"] = 395 -- other
    data["DoorOpen_Cleaning_Layer.clean_arrow_img.image"] = image_path("icon-clean-arrow-tablet_or_sachet_add.png")
  end

  gre.set_data(data)

end

--
-- handle io_clean_door_show_rotor event:
--
--  event send after io_clean_door_show event
--   -> make rotor visible
--
function CBCleaning_OnCleanDoorShowRotor()
 local data = {}
 data["DoorOpen_Layer.rotate_img.grd_hidden"] = false
 gre.set_data(data)
end

--
-- handle io_clean_door_hide event:
--   reset everything changed in CBCleaning_OnCleanDoorShow()
--
function CBCleaning_OnCleanDoorHide()

 Control_SetButtons(true, true, true, true, true, true, true)

 local data = {}
 data["CLEANING_Screen.DoorOpen_Layer.grd_hidden"] = true
 data["CLEANING_Screen.DoorOpen_Cleaning_Layer.grd_hidden"] = true
 data["DoorOpen_Layer.rotate_img.grd_hidden"] = false
 gre.set_data(data)

 DoorOpen_Cleaning_Layer_Reset()
end

--
-- handle io_clean_totaltimes_reply event
--
--  @param gre#context mapargs
function CBCleaning_OnCleanTotalTimesReply(mapargs)
  for i,name in ipairs(split_string(mapargs.context_event_data.names, DELIMITER_TOKEN)) do
    CleaningTotalTimes:set(name, mapargs.context_event_data.totaltimes[i])
  end

  if CleaningTotalTimes.showProgramSelection == true then
    Cleaning_ShowProgramSelectionDialog()
  end
  CleaningTotalTimes.showProgramSelection = false
end

