------------------------------------------------------
-- implementation of the screen to create/edit recipes
------------------------------------------------------

require("Scrollbar")
require("Circle")
require("Dialog")
require("History")
require("language")
require("RoleManager")
require("Utf8")

local TIME_FORMAT = "%01d:%02d"                                          -- string format for time "0:00"
-- limits 
local MIN_TEMP_C = 200                                                   -- minimum temperature in .1°C
local MAX_TEMP_C = 2500                                                  -- maximum temperature in .1°C
local MAX_PREHEAT_TEMP_C = 1500                                          -- maximum preheat temperature in .1°C
local MIN_TIME_MINS = 0                                                  -- minimum step duration in minutes
local MAX_TIME_MINS = 240                                                -- maximum step duration in minutes
local MAX_RECIPE_COUNT = 150                 
local MAX_RECIPE_STEP = 9 
local MAX_RECIPE_STRLEN = 13
                                              
local PREHEAT_DEFAULT_KEY = "PREHEAT_DEFAULT"                            -- preheat default temp key in the settings database

local gEditName = nil           -- the new name of the recipe if applicable
local gRecipeName = nil         -- the name of the recipe (the original name if it's being edited)
local gRecipe = {}              -- the recipe being edited or created
local gActiveStep = 1           -- the currently selected step in the steps table
local gPressedStep = -1         -- the currently pressed step in the steps table
local gSegmentId = 1            -- next available id to create an additional step button on the progress arc
local gTimeSegments = {}        -- list of ids of step buttons on the progress arc
local gTimerId = nil            -- timer to determine how long a press event lasts (to see if it's a short or long press)
local gSelectedIcon = {i=0,j=0} -- the coordinates on the icon selection grid of the currently active icon for the recipe
local gUsedIcons = nil          -- a list of icons that are already in use by a recipe
local gDisabledIcons = {}       -- if total icons is not a multiple of 4 some grid cells should be disabled
local gBtnOffset = 25           -- distance of the edge of the step button to its center (half the width of the image control)
local gOldValue = nil           -- old time/duration value used as fallback when input is invalid
local gTempNumber = nil         -- partial/temporary number for edit step time/duration control
local gTempText = nil           -- partial/temporary text for edit step time/duration control
local gEditingName = false
local gEditingIcon = false
local gEditingStepName = false
local gEditingStepTime = false
local gEditingStepTemperature = false
local gOldTableOffset = 0
local gAddStepPressed = false
local gActiveInsertLine = -1
local gStepIsPressed = 0
local gPressStepY = 0
local gInsertAtNumber = -1

--
-- return maximum time in minutes of a recipe step
--
local function Max_Time_Mins()
  return Settings:get_def("RECIPE_TIME_MAX", MAX_TIME_MINS)
end


--
-- return maximum temperature in C for a recipe step
--
local function Max_Temp_C()
  return Settings:get_def("RECIPE_TEMP_MAX", MAX_TEMP_C) / 10
end

--
-- return maximum temperature in C for a recipe step
--
local function Min_Temp_C()
  return Settings:get_def("RECIPE_TEMP_MIN", MIN_TEMP_C) / 10
end
--
-- return maximum temperature in C for the preheat step
--
local function Max_PreHeat_Temp_C()
  return Settings:get_def("PREHEAT_TEMP_MAX", MAX_PREHEAT_TEMP_C) / 10
end

--
-- return maximum number of recipe
--
local function Max_Recipe_Count()
  return Settings:get_def("RECIPE_COUNT_MAX", MAX_RECIPE_COUNT)
end

--
-- return maximum number of step
--
local function Max_Recipe_Step()
  return Settings:get_def("RECIPE_STEP_MAX", MAX_RECIPE_STEP)
end

--
-- return max length for strings used a recipe
--  recipe, step and icon name
--
local function Max_Recipe_StrLen()
  return math.min(Settings:get_def("RECIPE_STRLEN_MAX", MAX_RECIPE_STRLEN), MAX_RECIPE_STRLEN)
end

--
-- calculate the total time of a recipe and the number of empty steps
--
local function calculateStepsTime(steps)
  local total_time = 0
  local empty_steps = 0
  
  for _, step in ipairs(steps) do
    total_time = total_time + step.time
    
    if step.time == 0 then
      empty_steps = empty_steps + 1
    end
  end
  
  return {time = total_time, empty = empty_steps}
end

--- @param gre#context mapargs
local function RenameRecipe(mapargs)
  local tmp = gre.get_value("CR_Layer_v2.recipeheader_group.title_text.text")
  local value = string.sub(tmp, 1, #tmp - 1) or ""

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    value = Utf8_RemoveLastCharacter(value)
  else
    -- append char
    local tmp = value..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= Max_Recipe_StrLen() then
      value = tmp
    end
  end
  
  gre.set_value("CR_Layer_v2.recipeheader_group.title_text.text", value.."|")
end

--
-- scroll the active step to the bottom of the screen (and make sure total time is visible)
--
local function scroll_to_active_step()
  local table_height = gre.get_table_attrs("CR_Layer_v2.steps_group.steps_table", "height").height
  local table_y_pos = gre.get_table_attrs("CR_Layer_v2.steps_group.steps_table", "y").y
  local cell_height = gre.get_table_cell_attrs("CR_Layer_v2.steps_group.steps_table", gActiveStep, 1, "height").height

  local cell_count = gActiveStep == #gRecipe.steps and gActiveStep + 1 or gActiveStep
  local cell_bottom_y = table_y_pos + (cell_count * cell_height)
  local table_bottom_y = table_y_pos + table_height
  if cell_bottom_y > table_bottom_y then
    gre.set_table_attrs("CR_Layer_v2.steps_group.steps_table", { yoffset = table_bottom_y - cell_bottom_y })
  end
  if ((#gRecipe.steps + 1) * cell_height) < table_height then
    gre.set_table_attrs("CR_Layer_v2.steps_group.steps_table", { yoffset = 0 })
  end
end

--
-- display recipe steps on the screen
--
local function display_steps()
  local data = {}

  local total_hours, total_minutes = calculate_time(gRecipe.steps)
  local count = #gRecipe.steps + 1
  gre.set_table_attrs("CR_Layer_v2.steps_group.steps_table", {rows = count})
  
  -- recipe steps
  for i, step in ipairs(gRecipe.steps) do
    local index = i
    local temp = Temperature.new{temp = step.temp}
    local hours, minutes = calculate_time({step})
    
    data["CR_Layer_v2.steps_group.steps_table.name."..index..".1"] = step.custom and step.name or i18n:get(step.name)
    data["CR_Layer_v2.steps_group.steps_table.time."..index..".1"] = not step.notify and ((step.preheat or step.hold) and "-" or string.format(TIME_FORMAT, hours, minutes)) or ""
    data["CR_Layer_v2.steps_group.steps_table.notification_icon."..index..".1"] = step.notify and 255 or 0
    data["CR_Layer_v2.steps_group.steps_table.temperature."..index..".1"] = step.notify and "-" or temp:to_string()
    data["CR_Layer_v2.steps_group.steps_table.hours_active."..index..".1"] = 0
    data["CR_Layer_v2.steps_group.steps_table.minutes_active."..index..".1"] = 0
    data["CR_Layer_v2.steps_group.steps_table.selected."..index..".1"] = i == gActiveStep and 255 or 0
    data["CR_Layer_v2.steps_group.steps_table.step_row_images."..index..".1"] = 255
    data["CR_Layer_v2.steps_group.steps_table.total_time_row."..index..".1"] = 0
    data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha."..index..".1"] = 0
  end
  
  -- total time table entry
  data["CR_Layer_v2.steps_group.steps_table.name."..count..".1"] = ""
  data["CR_Layer_v2.steps_group.steps_table.time."..count..".1"] = string.format(TIME_FORMAT, total_hours, total_minutes)
  data["CR_Layer_v2.steps_group.steps_table.notification_icon."..count..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.temperature."..count..".1"] = ""
  data["CR_Layer_v2.steps_group.steps_table.hours_active."..count..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.minutes_active."..count..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.selected."..count..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.step_row_images."..count..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.total_time_row."..count..".1"] = 255
  data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha."..count..".1"] = 0
  
  gre.set_data(data)
end

local function log(message)
  if false then
    print(message)
  end
end

--
-- handle keyboard key press event, pass it to the recipe name or the active step name
--
--- @param gre#context mapargs
function CBCR_InputKeyEvent(mapargs)
  if gEditingStepName then
    CBCR_StepName_InputKeyEvent(mapargs)
  elseif gEditingName then
    RenameRecipe(mapargs)
  elseif gEditingStepTime then
     CBCR_StepTime_InputKeyEvent(mapargs)
  elseif gEditingStepTemperature then
     CBCR_StepTemperature_InputKeyEvent(mapargs)
  end
end

--
-- keyboard cancel event
--
function CBCR_OnKeyboardCancel()
  gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = true })
  if gEditingStepName then
    CBCR_StepName_OnKeyboardCancel()
  elseif gEditingName then
    gre.set_value("CR_Layer_v2.recipeheader_group.title_text.text", gTempText)
    gEditingName = false
  elseif gEditingStepTime then
    gEditingStepTime = false
    gre.set_layer_attrs("EditTime_Overlay", { hidden = true })
  elseif gEditingStepTemperature then
    gEditingStepTime = false
    gre.set_layer_attrs("EditTemperature_Overlay", { hidden = true })
  end
  gTempText = nil
end

--
-- finished editing step time
--
function CBCR_StepTime_Save()
  local new_time = StepTime_GetAndClose()
  
  gRecipe.steps[gActiveStep].time = new_time
  local total_hours, total_minutes = calculate_time(gRecipe.steps)
  
  local data = {}
  data["CR_Layer_v2.steps_group.steps_table.time."..gActiveStep..".1"] = string.format("%01d:%02d", new_time / 60, new_time % 60)
  data["CR_Layer_v2.steps_group.steps_table.time."..(#gRecipe.steps + 1)..".1"] = string.format(TIME_FORMAT, total_hours, total_minutes)
  data["CR_Layer_v2.steps_group.steps_table.hours_active."..gActiveStep..".1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.minutes_active."..gActiveStep..".1"] = 0
  gre.set_data(data)
  
  gEditingStepTime = false
end

--
-- finished editing step temperature
--
function CBCR_StepTemperature_Save()
  local new_temperature = StepTemperature_GetAndClose()
  
  gRecipe.steps[gActiveStep].temp = new_temperature:getC()
  gre.set_value("CR_Layer_v2.steps_group.steps_table.temperature."..gActiveStep..".1", new_temperature:to_string())
                
  gEditingStepTemperature = false
end

--
-- keyboard confirm event
--
function CBCR_OnKeyboardConfirm()
  gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = true })
  if gEditingStepName then
    CBCR_StepName_OnKeyboardConfirm()
  elseif gEditingName then
    local newname = gre.get_value("CR_Layer_v2.recipeheader_group.title_text.text")
    newname = string.sub(newname, 1, #newname - 1)
    newname = #newname > 0 and newname or gTempText
    gre.set_value("CR_Layer_v2.recipeheader_group.title_text.text", newname)
    gRecipeName = newname
    gEditingName = false
  elseif gEditingStepTime then
    CBCR_StepTime_Save()
  elseif gEditingStepTemperature then
    CBCR_StepTemperature_Save()
  end
  gTempText = nil
end

--
-- save the recipe that is being created/edited
--
function CBCR_SaveRecipe()
  if gRecipeName == "" or #gRecipe.steps == 0 then
    return
  end
  
  if (gEditName ~= nil and gEditName ~= gRecipeName and Recipes:is_name_taken(gRecipeName))
      or (gEditName == nil and Recipes:is_name_taken(gRecipeName)) then
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(string.format(i18n:get("NLS_RECIPE_NAME_TAKEN"), string.gsub(gRecipeName, " ", NBSP)))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
    return
  end
  
  local recipe = table_copy(gRecipe)
  if gEditName ~= nil then
    Event:update_recipe(gEditName, gRecipeName, recipe)
    Recipes:update(gEditName, gRecipeName, recipe)
  else
    Event:create_recipe(gRecipeName, recipe)
    Recipes:put(gRecipeName, recipe)
  end

  gre.send_event("recipes_screen")
end

--
-- exit the create recipe screen by either pressing the back button or the cancel button
--
local function ExitCRScreen(back_button)
  local back_lambda = function() History:pop() end
  local cancel_lambda = function() gre.send_event("recipes_screen") end
  local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
  dlg:set_message(string.format(i18n:get("NLS_RECIPE_DISCARD"), string.gsub(gRecipeName, " ", NBSP)))
  dlg:add_button(i18n:get("NLS_NO"))
  dlg:add_button(i18n:get("NLS_YES"), back_button and back_lambda or cancel_lambda)
  dlg:show()
end

--
-- return to recipes screen (cancel button pressed)
--
function CBCR_CancelEditRecipe()
  ExitCRScreen(false)
end

--
-- return to screen on top of history stack (back button pressed)
--
function CR_OnBackPressed()
  if gEditingStepName or gEditingIcon then
    History:pop()
    gEditingStepName = false
    gEditingIcon = false
    gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = true })
    gre.set_layer_attrs("StepName_Layer", { hidden = true })
    gre.set_layer_attrs("SelectIcon_Layer", { hidden = true })
    gre.set_layer_attrs("EditStepName_Overlay", { hidden = true })
  else
    ExitCRScreen(true)
  end
end

--
-- delete the recipe that is open in the editor
--
function CBCR_DeleteRecipe()
  local delete_action = function()
    if gEditName ~= nil then
      Event:delete_recipe(gEditName)
      Recipes:delete(gEditName)
    end
    gre.send_event("recipes_screen")
  end

  local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
  dlg:set_message(string.format(i18n:get("NLS_RECIPE_DELETE"), string.gsub(gRecipeName, " ", NBSP)))
  dlg:add_button(i18n:get("NLS_NO"))
  dlg:add_button(i18n:get("NLS_YES"), delete_action)
  dlg:show()
end

--
-- delete the selected step from the recipe
--
--- @param gre#context mapargs
function CBCR_DeleteStep(mapargs)
  -- don't allow last step to be deleted
  if #gRecipe.steps == 1 then
    return
  end
  
  table.remove(gRecipe.steps, gActiveStep)
  gActiveStep = (gActiveStep <= #gRecipe.steps) and gActiveStep or ((gActiveStep > 1) and (gActiveStep - 1) or -1)
  display_steps()
  scroll_to_active_step()
end

--
-- add a new step to the recipe (insert it after the currently active step)
--
function CBCR_PressAdd(mapargs)
  --gre.set_control_attrs("CR_Layer_v2.steps_group.table_overlay", dk_data) line disabled because dk_data does not exist
  log("press")
  gAddStepPressed = true
end

--
-- Check where the touch event on the step happened. If it's within the bounds of
-- the temperature text, we need to open the control to edit the temperature. If it's
-- within the bounds of the time text, we need to open the control to edit the time.
--
--- @param gre#context mapargs
local function checkEditStep(mapargs)
  local x = mapargs.context_event_data.x
  if x > 15 and x < 373 then
    -- name pressed
    CBCR_RenameStep()
  elseif x > 373 and x < 429 and not (gRecipe.steps[gActiveStep].preheat or gRecipe.steps[gActiveStep].notify) then
    -- time pressed
    CBCR_ChangeStepTime()
  elseif x > 429 and not gRecipe.steps[gActiveStep].notify then
    -- temperature pressed
    CBCR_ChangeStepTemp()
  end
end

local function ResetInsertLine()
  -- Don't show previous insert line anymore
  if not (gActiveInsertLine == -1) then
    local data = {}
    if(gActiveInsertLine == 0) then
      data["CR_Layer_v2.recipeheader_group.Rectangle.alpha"] = 0
    else
      data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha."..gActiveInsertLine..".1"] = 0
    end
    gre.set_data(data)
    gActiveInsertLine = -1
  end
end

local function MarkInsertLine(insertAtNumber)
  ResetInsertLine()
  
  -- Mark active insert line
  if (insertAtNumber <= #gRecipe.steps) then
    local data = {}
    if(insertAtNumber == 0) then
      data["CR_Layer_v2.recipeheader_group.Rectangle.alpha"] = 255
    else
      data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha."..tostring(insertAtNumber)..".1"] = 255
    end
    gre.set_data(data)
    gActiveInsertLine = insertAtNumber
  end
end

--
-- add a new step to the recipe (insert it after the currently active step)
--
function CBCR_ReleaseAdd(mapargs)
  log("release")
  local tableInfo = gre.get_table_attrs("CR_Layer_v2.steps_group.steps_table", "yoffset", "y", "height")

  local pos = {}
  pos["y"] = 0
  gre.set_control_attrs("CR_Layer_v2.newstep_group", pos)
  gAddStepPressed = false

  ResetInsertLine()
  
  if (gInsertAtNumber >= 0) and
     (gInsertAtNumber <= #gRecipe.steps) and
     (mapargs.context_event_data.y < (tableInfo.y+tableInfo.height)) then
    log(#gRecipe.steps)
    local text = #gRecipe.steps == 0 and i18n:get("NLS_STEPNAME_PREHEAT") or string.format("%s %d", i18n:get("NLS_STEPNAME_CUSTOM"), gInsertAtNumber)
    local step = { name = text, temp = Min_Temp_C(), time = 0, custom = true }
    table.insert(gRecipe.steps, gInsertAtNumber+1, step)
    gActiveStep = gInsertAtNumber+1
    display_steps()
  end
end

--
-- Drag a new step to the recipe
--
function CBCR_DragStep(mapargs)
  if (gAddStepPressed == false) then
    return
  end

  -- Calculate where to drag the extra step rectangle (y position of newstep_group)
  local ev_data = mapargs.context_event_data

  local tableInfo = gre.get_table_attrs("CR_Layer_v2.steps_group.steps_table", "yoffset", "y", "height")
  local cell_height = gre.get_table_cell_attrs("CR_Layer_v2.steps_group.steps_table", 1, 1, "height").height
  local insertAtNumber = math.floor((ev_data.y - tableInfo.y + (cell_height/2) - tableInfo.yoffset)/cell_height)

  log(insertAtNumber)
  
  -- if below table, insert at end
  if (insertAtNumber>#gRecipe.steps) then
    insertAtNumber = #gRecipe.steps
  end

  if (insertAtNumber >= 0) and
     (insertAtNumber <= #gRecipe.steps) and
     (mapargs.context_event_data.y < (tableInfo.y+tableInfo.height)) then
    gInsertAtNumber = insertAtNumber
  
    MarkInsertLine(gInsertAtNumber)
    
    log(tableInfo.height)
    log(tableInfo.y)
  else
    ResetInsertLine()
    gInsertAtNumber = -1
  end  
  local maxy = (tableInfo.y+tableInfo.height+(cell_height/2))
  if (ev_data.y > tableInfo.y) and (ev_data.y <= maxy) then
    -- set the control to the new position
    local pos = {}
    pos["x"] = 0
    log(ev_data.y)
    pos["y"] = ev_data.y - maxy
    log(pos["y"])
    gre.set_control_attrs("CR_Layer_v2.newstep_group", pos)
  end
end

--
-- delete button pressed
--
function CBCR_OnDelete()
  if gActiveStep == -1 then
    CBCR_DeleteRecipe()
  else
    CBCR_DeleteStep()
  end
end

--
-- rename recipe
--
function CBCR_OnRenameRecipe()
  if not gEditingName then
    gre.set_layer_attrs("EditTime_Overlay", { hidden = true })
    gre.set_layer_attrs("EditTemperature_Overlay", { hidden = true })
    gEditingStepTime = false
    gEditingStepTemperature = false
    CBCR_OnKeyboardConfirm()
    gEditingName = true
    gTempText = gre.get_value("CR_Layer_v2.recipeheader_group.title_text.text")
    gre.set_value("CR_Layer_v2.recipeheader_group.title_text.text", "|")
    gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = false })
  end
end

--
-- open icon selection overlay
--
function CBCR_OnSelectIcon()
  CBCR_OnKeyboardConfirm()
  History:force_push("CR_Screen")
  gEditingIcon = true
  CBCR_IconSelect_OnOverlayShow()
end

--
-- new icon selected
--
function CBCR_OnIconSelected(mapargs)
  gEditingIcon = false
  local icon = mapargs.context_event_data.icon
  gRecipe.icon = icon
  gre.set_value("CR_Layer_v2.recipeheader_group.icon_img.image", icon_path(icon))
end

--
-- rename a recipe step
--
function CBCR_RenameStep()
  if gEditingStepName then
    CBCR_StepName_OnConfirm()
    local metadata = GetSelectedStep()
    local oldname = gRecipe.steps[gActiveStep].name
    gRecipe.steps[gActiveStep].name = metadata.name
    gRecipe.steps[gActiveStep].custom = metadata.custom
    if metadata.preheat then
      gRecipe.steps[gActiveStep].preheat = true
      gRecipe.steps[gActiveStep].time = 0
    elseif gRecipe.steps[gActiveStep].preheat then
      gRecipe.steps[gActiveStep].preheat = false
    end
    if metadata.hold then
      gRecipe.steps[gActiveStep].hold = true
      gRecipe.steps[gActiveStep].time = 0
    elseif gRecipe.steps[gActiveStep].hold then
      gRecipe.steps[gActiveStep].hold = false
    end
    if metadata.notify then
      gRecipe.steps[gActiveStep].notify = true
      gRecipe.steps[gActiveStep].time = 0
      gRecipe.steps[gActiveStep].temp = 0
    elseif gRecipe.steps[gActiveStep].notify then
      gRecipe.steps[gActiveStep].notify = false
    end
    local notify = gRecipe.steps[gActiveStep].notify
    local hide_time = gRecipe.steps[gActiveStep].preheat or gRecipe.steps[gActiveStep].hold
    local hide_temperature = gRecipe.steps[gActiveStep].notify
    local hours, minutes = calculate_time({gRecipe.steps[gActiveStep]})
    local temp = Temperature.new{temp = gRecipe.steps[gActiveStep].temp}
    gre.set_value("CR_Layer_v2.steps_group.steps_table.time."..gActiveStep..".1", not notify and (hide_time and "-" or string.format(TIME_FORMAT, hours, minutes)) or "")
    gre.set_value("CR_Layer_v2.steps_group.steps_table.notification_icon."..gActiveStep..".1", notify and 255 or 0)
    gre.set_value("CR_Layer_v2.steps_group.steps_table.temperature."..gActiveStep..".1", hide_temperature and "-" or temp:to_string())
    gre.set_value("CR_Layer_v2.steps_group.steps_table.name."..gActiveStep..".1", metadata.custom and metadata.name or i18n:get(metadata.name))
    gTempText = nil
    gEditingStepName = false
  else
    gEditingStepName = true
    gTempText = gRecipe.steps[gActiveStep].name
    History:force_push("CR_Screen")
    CBCR_StepName_OnOverlayShow(gTempText, gRecipe.steps[gActiveStep].custom, gRecipe.steps[gActiveStep].notify)
  end
end

--
-- cancel rename step
--
function CBCR_RenameStepCancelled()
  gEditingStepName = false
  CBCR_StepName_OnCancel()
end

--
-- edit step time
--
function CBCR_ChangeStepTime()
  gEditingStepTime = true
  
  CBCR_StepTime_OnOverlayShow(gRecipe.steps[gActiveStep].time, 1, Max_Time_Mins())
end

--
-- edit step temperature
--
function CBCR_ChangeStepTemp()
  gEditingStepTemperature = true

  local step = gRecipe.steps[gActiveStep]
  local max_temp = Max_Temp_C()
  if step.preheat then
    max_temp = Max_PreHeat_Temp_C()
  end
  CBCR_StepTemperature_OnOverlayShow(step.temp, Min_Temp_C(), max_temp)
end

--
-- select a recipe step
--
--- @param gre#context mapargs
function CBCR_SelectStep(mapargs)
  if gEditingName or gEditingStepName or gEditingStepTime or gEditingStepTemperature then
    -- don't allow selection change while editing
    CBCR_OnKeyboardConfirm()
    gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = true })
  elseif gActiveStep == mapargs.context_row then
    -- step already active, see if we need to edit temperature or time
    if mapargs.context_event_data ~= nil then
      checkEditStep(mapargs)
    end
  elseif mapargs.context_row == (#gRecipe.steps + 1) then
    -- last element in list is not a recipe step, it has the total time
    if gActiveStep ~= -1 then
      gre.set_value("CR_Layer_v2.steps_group.steps_table.selected."..gActiveStep..".1", 0)
      gActiveStep = -1
    end
  else
    -- step not yet active, so set the step as active
    if gActiveStep ~= -1 then
      gre.set_value("CR_Layer_v2.steps_group.steps_table.selected."..gActiveStep..".1", 0)
    end
    gre.set_value("CR_Layer_v2.steps_group.steps_table.selected."..mapargs.context_row..".1", 255)
    gActiveStep = mapargs.context_row
  end
end

--
-- Press to drag a step to a new position
--
--- @param gre#context mapargs
function CBCR_PressStep(mapargs)
  log("p")
  -- remember start position to cancel timer when scrolling
  gPressStepY = mapargs.context_event_data.y
  
  gPressedStep = mapargs.context_row
  gStepIsPressed = 1
  -- Start Timer to detect long pressed to drag steps
  gTimerId = gre.timer_set_interval(CBCR_OnTimer, 800)
end

--
-- Release after step press 
--
--- @param gre#context mapargs
function CBCR_ReleaseStep(mapargs)
  -- If we get here the touch/mouse was released before the timer elapsed
  -- => we don't want to drag to a new position => cancel the timer
  gStepIsPressed = 0
  gAddStepPressed = false
  if gTimerId ~= nil then
    gre.timer_clear_interval(gTimerId)
    gTimerId = nil
  end
  
  log("r")
end

function CBCR_MotionStep(mapargs)
  log("m")
  log(mapargs.context_event_data.y)
  -- Cancel the dragging timer if we are scrolling 
  if(gStepIsPressed == 1) and math.abs( mapargs.context_event_data.y - gPressStepY) > 5 then  
    if gTimerId ~= nil then
      gre.timer_clear_interval(gTimerId)
      gTimerId = nil
    end
  end
end

function CBCR_MotionMove(mapargs)
  log("M")
  if(gStepIsPressed == 1) then
  
    -- Calculate where to drag the extra step rectangle (y position of newstep_group)
    local ev_data = mapargs.context_event_data
  
    local tableInfo = gre.get_table_attrs("CR_Layer_v2.steps_group.steps_table", "yoffset", "y", "height")
    local cell_height = gre.get_table_cell_attrs("CR_Layer_v2.steps_group.steps_table", 1, 1, "height").height
    local insertAtNumber = math.floor((ev_data.y - tableInfo.y + (cell_height/2) - tableInfo.yoffset)/cell_height)
  
    log(insertAtNumber)
  
    log(ev_data.y)
    
    log(gPressedStep)
    
    MarkInsertLine(insertAtNumber)
    
  end
end

function CBCR_PressMove()
  log("P")
end

function CBCR_ReleaseMove()
  local data = {}

  local tmp = gActiveInsertLine  
  ResetInsertLine()
  gActiveInsertLine = tmp
  
  log(gActiveInsertLine)
  log(gPressedStep)
  log(#gRecipe.steps)
  
  -- move the step
  if (gPressedStep>0) and not (gActiveInsertLine == -1) and 
    ((gActiveInsertLine>gPressedStep) or (gActiveInsertLine<(gPressedStep-1))) then
    
    local step = gRecipe.steps[gPressedStep]
    table.insert(gRecipe.steps, gActiveInsertLine+1, step)
    if (gActiveInsertLine>gPressedStep) then
      table.remove(gRecipe.steps, gPressedStep)
      gActiveStep = gActiveInsertLine  
    else
      table.remove(gRecipe.steps, gPressedStep+1)
      gActiveStep = gActiveInsertLine+1  
    end 
    gActiveInsertLine = -1 
    display_steps()
  end
  
  gre.set_data(data)
  gActiveInsertLine = -1
  gStepIsPressed = 0

  local dk_data = {}
  dk_data["hidden"] = 1
  gre.set_control_attrs("CR_Layer_v2.steps_group.table_overlay", dk_data)
  
  log("R")
end

function CBCR_OnTimer()
  log("Timer")
  log(gStepIsPressed)
  
  -- Reset timer
  if (gTimerId ~= nil) then
    gre.timer_clear_interval(gTimerId)
    gTimerId = nil
  end
  
  if (gStepIsPressed == 1) then    
    log("drag")
    if gActiveStep ~= -1 then
      gre.set_value("CR_Layer_v2.steps_group.steps_table.selected."..gActiveStep..".1", 0)
      gActiveStep = -1
    end
    -- last element in list is not a recipe step, it has the total time
    if gPressedStep ~= (#gRecipe.steps + 1) then
      -- step not yet active, so set the step as active
      gre.set_value("CR_Layer_v2.steps_group.steps_table.selected."..gPressedStep..".1", 255)
      gActiveStep = gPressedStep

      -- unhide overlay so auto scroll of the table is off
      local dk_data = {}
      dk_data["hidden"] = 0
      gre.set_control_attrs("CR_Layer_v2.steps_group.table_overlay", dk_data)
    end
  end
end


--
-- open the create/update recipe screen to create a new recipe
--
function CBCR_NewRecipe()
  gEditName = nil
  gRecipeName = i18n:get("NLS_DEFAULT_RECIPE_NAME")
  gRecipe = {}
  gRecipe.sort_index = Recipes:getn() + 1 -- append to end by default
  gRecipe.time = 0
  gRecipe.steps = { { name = "NLS_STEPNAME_PREHEAT", time = 0, temp = Settings:get(PREHEAT_DEFAULT_KEY) / 10, preheat = true, custom = false } }
  gActiveStep = -1

  gre.set_table_attrs("CR_Layer_v2.steps_group.steps_table", {rows = 2})

  local data = {}
  local step = gRecipe.steps[1]
  local temp = Temperature.new{temp = step.temp}
  data["CR_Layer_v2.title_group.title_text.text"] = i18n:get("NLS_CREATE_RECIPE_TITLE")
  data["CR_Layer_v2.recipeheader_group.title_text.text"] = gRecipeName
  data["CR_Layer_v2.recipeheader_group.icon_img.image"] = image_path("icon-generiek.png")
  data["CR_Layer_v2.steps_group.steps_table.name.1.1"] = i18n:get(step.name)
  data["CR_Layer_v2.steps_group.steps_table.time.1.1"] = "-"
  data["CR_Layer_v2.steps_group.steps_table.temperature.1.1"] = temp:to_string()
  data["CR_Layer_v2.steps_group.steps_table.hours_active.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.minutes_active.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.selected.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.step_row_images.1.1"] = 255
  data["CR_Layer_v2.steps_group.steps_table.total_time_row.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.hours_active.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.minutes_active.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha.1.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.name.2.1"] = ""
  data["CR_Layer_v2.steps_group.steps_table.time.2.1"] = "-"
  data["CR_Layer_v2.steps_group.steps_table.temperature.2.1"] = ""
  data["CR_Layer_v2.steps_group.steps_table.hours_active.2.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.minutes_active.2.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.selected.2.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.step_row_images.2.1"] = 0
  data["CR_Layer_v2.steps_group.steps_table.total_time_row.2.1"] = 255
  data["CR_Layer_v2.steps_group.steps_table.insert_line_alpha.2.1"] = 0
  data["CR_Layer_v2.button_group.ok_btn.grd_hidden"] = false
  data["CR_Layer_v2.button_group.delete_btn.grd_hidden"] = false 
  gre.set_data(data)
end

--
-- open the create/update recipe screen to update an existing recipe
--
--- @param gre#context mapargs
function CBCR_EditRecipe(mapargs)
  local name = mapargs.context_event_data.recipe

  gEditName = name
  gRecipeName = gEditName
  gRecipe = table_copy(Recipes:get(name))
  gActiveStep = -1

  local data = {}
  data["CR_Layer_v2.title_group.title_text.text"] = i18n:get("NLS_EDIT_RECIPE_TITLE")
  data["CR_Layer_v2.recipeheader_group.title_text.text"] = gRecipeName
  data["CR_Layer_v2.recipeheader_group.icon_img.image"] = gRecipe.icon and icon_path(gRecipe.icon) or image_path("icon-generiek.png")
  data["CR_Layer_v2.button_group.ok_btn.grd_hidden"] = false
  data["CR_Layer_v2.button_group.delete_btn.grd_hidden"] = false
  gre.set_data(data)

  display_steps()
  
  -- disable saving of changes when oven is running
  if ActiveRecipe:is_cooking() and ActiveRecipe.name == name then 
    local data = {}
    data["CR_Layer_v2.title_group.title_text.text"] = "View only"
    data["CR_Layer_v2.button_group.ok_btn.grd_hidden"] = true
    data["CR_Layer_v2.button_group.delete_btn.grd_hidden"] = true
    gre.set_data(data)
  end
end

--
-- called before screen is displayed
--
function CBCR_Prepare()
  gEditingName = false
  gEditingStepName = false
  gEditingStepTime = false
  gEditingStepTemperature = false

  Control_SetButtons(true, true, true, true, true, true, false)

  local login_required = Settings:get_def("ENDUSER_EDIT", 1) == 0 and not RoleManager:can_access(MANAGER_ROLE)

  gre.set_layer_attrs("CR_Screen.Login_Layer", { hidden = not login_required })
  gre.set_layer_attrs("CR_Screen.SelectIcon_Layer", { hidden = true })
  gre.set_layer_attrs("CR_Screen.Keyboard_Layer_v2", { hidden = true })
  gre.set_layer_attrs("CR_Screen.EditStepName_Overlay", { hidden = true })
  gre.set_layer_attrs("CR_Screen.EditTime_Overlay", { hidden = true })
  gre.set_layer_attrs("CR_Screen.EditTemperature_Overlay", { hidden = true })
  gre.set_layer_attrs("CR_Screen.StepName_Layer", { hidden = true })
  gre.set_table_attrs("CR_Layer_v2.steps_group.steps_table", { yoffset = 0 })
  gre.set_value("CR_Layer_v2.recipeheader_group.edittemp_img.image",
                Temperature.isC() and image_path("edit-icon-tempC-x431-y214.png") or image_path("edit-icon-tempF-x431-y214.png"))
end

--
-- login required, but close button pressed --> return to previous screen
--
function CBCR_OnLoginCanceled(mapargs)
  gre.send_event_target("gre.touch", "Icon_Back_Layer.CR_Icon_Back_Img")
end

--
-- role changed, if role lowered, check if new role is allowed to edit recipes
--
function CBCR_OnRoleChanged(mapargs)
  local login_required = Settings:get_def("ENDUSER_EDIT", 1) == 0 and not RoleManager:can_access(MANAGER_ROLE)
  gre.set_layer_attrs("CR_Screen.Login_Layer", { hidden = not login_required })
end

--
-- let application know that we are still active (every minute)
--
function CBCR_OnRefreshElevationTimer(mapargs)
  if Settings:get_def("ENDUSER_EDIT", 1) == 0 and RoleManager:can_access(MANAGER_ROLE) then
    RoleManager:kick()
  end
end
