-------------------------------------
-- recipe step name selection overlay
-------------------------------------

require("events")
require("language")
require("Settings")
require("Utf8")

json = require("json")

local STEPNAME_JSON_FILE = "recipesteps.json"
local NLS_CUSTOM_NAME_KEY = "NLS_STEPNAME_CUSTOM"
local CUSTOMSTEPNAME_LAYER = "EditStepName_Overlay"
local NOTIFYTEXT_LAYER = "EditNotifyText_Overlay"
local KEYBOARD_LAYER = "Keyboard_Layer_v2"
local STEPNAME_LAYER = "StepName_Layer"
local STEPNAME_GROUP = STEPNAME_LAYER..".stepname_group"
local STEPNAME_TABLE = STEPNAME_GROUP..".stepnames_table"
local SLIDER_OVERLAY = STEPNAME_GROUP..".slider_overlay"
local ACTIVE_TEXT_COLOR = 0xFFFFFF
local NORMAL_TEXT_COLOR = 0x000000
local ACTIVE_IMAGE_ALPHA = 255
local NORMAL_IMAGE_ALPHA = 0
local ACTIVE_SEGMENT_ALPHA = 255
local HIDDEN_SEGMENT_ALPHA = 0
local SLIDER_IMAGE_HEIGHT = 40
local MAX_STEP_STRLEN = 12
local MAX_NOTIFY_STRLEN = 30
local ALIGN_CENTER = 2
local ALIGN_RIGHT = 3

local gStepNames = {}
local gSelectedIndex = 0
local gCustomStepName = ""
local gNotifyText = ""
local gOldTableOffset = 0
local gSliderIndex = 0
local gSliderY = 0

--
-- return max length for strings used a recipe
-- recipe, step and icon name
--
local function Max_Step_StrLen()
  return math.min(Settings:get_def("RECIPE_STRLEN_MAX", MAX_STEP_STRLEN), MAX_STEP_STRLEN)
end

--
-- return max length for strings used a recipe
-- notification step text
--
local function Max_Notify_StrLen()
  return math.min(Settings:get_def("RECIPE_STRLEN_MAX", MAX_NOTIFY_STRLEN), MAX_NOTIFY_STRLEN)
end

--
-- get the meta data for the selected step
--
function GetSelectedStep()
  local index = gre.get_value(STEPNAME_GROUP..".step_name_index")
  local step_name = gre.get_value(STEPNAME_GROUP..".step_name")
  
  if index <= #gStepNames then
    return { name = gStepNames[index].notify and gNotifyText or gStepNames[index].key, preheat = gStepNames[index].preheat, hold = gStepNames[index].hold, custom = false, notify = gStepNames[index].notify }
  else
    return { name = step_name, preheat = false, hold = false, custom = true, notify = false }
  end
end

--
-- check if the selected step is a notification step
-- note: index is a table index, not step name list index
--
local function IsNotificationStep(index)
  local step_name_index = index - 1
  return (0 < step_name_index) and (step_name_index <= table_count(gStepNames)) and gStepNames[step_name_index].notify
end

local function UpdateNotifyTextAlignment()
  -- center if text fits, right align if text does not fit
  local notify_text = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.text")
  local font = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.font")
  local size = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.size")
  local width = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.grd_width") - 20
  local text_width = gre.get_string_size(font, size, notify_text).width
  gre.set_value(NOTIFYTEXT_LAYER..".notify_text.hAlign", (text_width < width) and ALIGN_CENTER or ALIGN_RIGHT)
end

local function UpdateCustomStepNameAlignment()
  -- center if text fits, right align if text does not fit
  local step_name = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text")
  local font = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.font")
  local size = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.size")
  local width = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.grd_width") - 20
  local text_width = gre.get_string_size(font, size, step_name).width
  gre.set_value(CUSTOMSTEPNAME_LAYER..".stepname_text.hAlign", (text_width < width) and ALIGN_CENTER or ALIGN_RIGHT)
end

--
-- notification text - key pressed
--
local function NotificationTextInputKeyEvent(mapargs)
  local notify_text = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.text")
  notify_text = (string.sub(notify_text, #notify_text, #notify_text) == "|") and string.sub(notify_text, 1, #notify_text - 1) or notify_text
  
  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    notify_text = Utf8_RemoveLastCharacter(notify_text)
  else
    -- append char
    local tmp = notify_text..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= Max_Notify_StrLen() then
      notify_text = tmp
    end
  end
  
  notify_text = notify_text.."|"
  gre.set_value(NOTIFYTEXT_LAYER..".notify_text.text", notify_text)
  
  UpdateNotifyTextAlignment()
end

--
-- custom step name - key pressed
--
local function CustomStepNameInputKeyEvent(mapargs)
  local step_name = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text")
  step_name = (string.sub(step_name, #step_name, #step_name) == "|") and string.sub(step_name, 1, #step_name - 1) or step_name
  
  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    step_name = Utf8_RemoveLastCharacter(step_name)
  else
    -- append char
    local tmp = step_name..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= Max_Step_StrLen() then
      step_name = tmp
    end
  end
  
  step_name = step_name.."|"
  gre.set_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text", step_name)
  
  UpdateCustomStepNameAlignment()
end

--
-- key pressed
--
function CBCR_StepName_InputKeyEvent(mapargs)
  if IsNotificationStep(gSelectedIndex) then
    NotificationTextInputKeyEvent(mapargs)
  else -- custom step name
    CustomStepNameInputKeyEvent(mapargs)
  end
end

--
-- notification text - confirm
--
local function ConfirmNotificationText()
  -- hide edit notification text overlay
  gre.set_layer_attrs(NOTIFYTEXT_LAYER, { hidden = true })
  -- remove | from end of notify text
  local notify_text = gre.get_value(NOTIFYTEXT_LAYER..".notify_text.text")
  notify_text = (string.sub(notify_text, #notify_text, #notify_text) == "|") and string.sub(notify_text, 1, #notify_text - 1) or notify_text
  if notify_text ~= "" then
    gNotifyText = notify_text
  else
    notify_text = gNotifyText
  end
end

--
-- custom step name - confirm
--
local function ConfirmCustomStepName()
  -- hide edit custom step name overlay
  gre.set_layer_attrs(CUSTOMSTEPNAME_LAYER, { hidden = true })
  -- remove | from end of custom name
  local step_name = gre.get_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text")
  step_name = (string.sub(step_name, #step_name, #step_name) == "|") and string.sub(step_name, 1, #step_name - 1) or step_name
  if step_name ~= "" then
    gCustomStepName = step_name
  else
    step_name = gCustomStepName
  end
  gre.set_value(STEPNAME_TABLE..".step_name."..gSelectedIndex..".1", step_name)
end

--
-- confirm editing
--
function CBCR_StepName_OnKeyboardConfirm()
  if IsNotificationStep(gSelectedIndex) then
    ConfirmNotificationText()
  else -- custom step name
    ConfirmCustomStepName()
  end
end

--
-- cancel editing
--
function CBCR_StepName_OnKeyboardCancel()
  gre.set_layer_attrs(CUSTOMSTEPNAME_LAYER, { hidden = true })
  gre.set_layer_attrs(NOTIFYTEXT_LAYER, { hidden = true })
end

--
-- cancel step name selection, close overlay
--
function CBCR_StepName_OnCancel()
  gre.set_layer_attrs(STEPNAME_LAYER, { hidden = true })
end

--
-- confirm step name selection
--
function CBCR_StepName_OnConfirm()
  local index = gSelectedIndex - 1
  local step_name = gre.get_value(STEPNAME_TABLE..".step_name."..gSelectedIndex..".1")
  
  -- check if custom name equals preset name
  if index > #gStepNames then
    for i, step_name in ipairs(gStepNames) do
      if step_name == i18n:get(step_name.key) then
        index = i
        break
      end
    end
  end
  
  local data = {}
  data[STEPNAME_GROUP..".step_name_index"] = index
  data[STEPNAME_GROUP..".step_name"] = step_name
  gre.set_data(data)

  gre.set_layer_attrs(STEPNAME_LAYER, { hidden = true })
end

--
-- select a new step name (if manually typing, cancel)
--
function CBCR_StepName_OnSelect(mapargs)
  local i = mapargs.context_row
  local count = gre.get_table_attrs(STEPNAME_TABLE, "rows").rows
  
  gre.set_value(SLIDER_OVERLAY..".grd_hidden", true)

  if i > 1 and i < count and i ~= gSelectedIndex then
    local data = {}

    -- cancel editing with keyboard
    if IsNotificationStep(gSelectedIndex) or gSelectedIndex == (count - 1) then
      -- hide keyboard
      gre.set_layer_attrs(CUSTOMSTEPNAME_LAYER, { hidden = true })
      gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = true })
      -- cancel editing
      CBCR_StepName_OnKeyboardConfirm()
    end
    
    -- set active item
    data[STEPNAME_TABLE..".text_color."..gSelectedIndex..".1"] = NORMAL_TEXT_COLOR
    data[STEPNAME_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = NORMAL_IMAGE_ALPHA
    data[STEPNAME_TABLE..".text_color."..i..".1"] = ACTIVE_TEXT_COLOR
    data[STEPNAME_TABLE..".active_image_alpha."..i..".1"] = ACTIVE_IMAGE_ALPHA
    
    gSelectedIndex = i
    
    gre.set_data(data)
  end

  -- start editing custom step name
  if gre.get_layer_attrs(KEYBOARD_LAYER, "hidden").hidden then
    if IsNotificationStep(gSelectedIndex) then
      -- show keyboard
      gre.set_layer_attrs(NOTIFYTEXT_LAYER, { hidden = false })
      gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
      -- overwrite existing notify text
      gre.set_value(NOTIFYTEXT_LAYER..".notify_text.text", "|")
      UpdateNotifyTextAlignment()
    elseif gSelectedIndex == (count - 1) then
      -- show keyboard
      gre.set_layer_attrs(CUSTOMSTEPNAME_LAYER, { hidden = false })
      gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
      -- overwrite existing custom name
      gre.set_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text", "|")
      UpdateCustomStepNameAlignment()
    end
  end
end

--
-- leave slider area -> cancel slider
--
function CBCR_StepName_OnSliderOutbound(mapargs)  
  local data = {}
  data[SLIDER_OVERLAY..".grd_hidden"] = true
  data[STEPNAME_TABLE..".text_color."..gSliderIndex..".1"] = NORMAL_TEXT_COLOR
  data[STEPNAME_TABLE..".text_color."..gSelectedIndex..".1"] = ACTIVE_TEXT_COLOR
  data[STEPNAME_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = ACTIVE_IMAGE_ALPHA
  gre.set_data(data)
end

--
-- release slider -> accept choice if within bounds, otherwise cancel
--
function CBCR_StepName_OnSliderRelease(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(STEPNAME_TABLE, "rows").rows
  
  gSliderIndex = calculate_table_cell_index(STEPNAME_TABLE, y)
  
  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBLanguage_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[SLIDER_OVERLAY..".grd_hidden"] = true
    data[STEPNAME_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[STEPNAME_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[STEPNAME_TABLE..".active_image_alpha."..gSliderIndex..".1"] = ACTIVE_IMAGE_ALPHA
    gre.set_data(data)
    
    if gSelectedIndex ~= gSliderIndex then
      if IsNotificationStep(gSelectedIndex) or gSelectedIndex == (count - 1) then
        gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = true }) -- hide keyboard
        CBCR_StepName_OnKeyboardConfirm() -- cancel editing
      end
      if IsNotificationStep(gSliderIndex) then
        -- show keyboard
        gre.set_layer_attrs(NOTIFYTEXT_LAYER, { hidden = false })
        gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
        -- overwrite existing notify text
        gre.set_value(NOTIFYTEXT_LAYER..".notify_text.text", "|")
        UpdateNotifyTextAlignment()
      elseif gSliderIndex == (count - 1) then
        -- show keyboard
        gre.set_layer_attrs(CUSTOMSTEPNAME_LAYER, { hidden = false })
        gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
        -- overwrite existing custom name
        gre.set_value(CUSTOMSTEPNAME_LAYER..".stepname_text.text", "|")
        UpdateCustomStepNameAlignment()
      end
    end
    
    gSelectedIndex = gSliderIndex
  end
end

--
-- slider motion -> update highlighted item if within bounds, otherwise cancel
--
function CBCR_StepName_OnSliderMotion(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(STEPNAME_TABLE, "rows").rows
  
  gSliderIndex = calculate_table_cell_index(STEPNAME_TABLE, y)
  
  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBLanguage_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[STEPNAME_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[STEPNAME_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- grab slider button to start sliding
--
function CBCR_StepName_OnStartSlider(mapargs)
  local i = mapargs.context_row
  local x = mapargs.context_event_data.x
  local y = mapargs.context_event_data.y
  
  local slider_info = gre.get_control_attrs(SLIDER_OVERLAY, "x", "width")
  
  if i == gSelectedIndex and x >= slider_info.x and x <= (slider_info.x + slider_info.width) then
    gSliderIndex = i
    
    local data = {}
    data[STEPNAME_TABLE..".active_image_alpha."..i..".1"] = NORMAL_IMAGE_ALPHA
    data[SLIDER_OVERLAY..".grd_hidden"] = false
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- create a step name table entry
--
local function StepName_TableEntry(table, index, properties)
  table[STEPNAME_TABLE..".keyboard."..index..".1"] = properties.keyboard or ""
  table[STEPNAME_TABLE..".step_name."..index..".1"] = properties.step_name or ""
  table[STEPNAME_TABLE..".text_color."..index..".1"] = properties.selected and ACTIVE_TEXT_COLOR or NORMAL_TEXT_COLOR
  table[STEPNAME_TABLE..".active_image_alpha."..index..".1"] = properties.selected and ACTIVE_IMAGE_ALPHA or NORMAL_IMAGE_ALPHA
  table[STEPNAME_TABLE..".top_segment_alpha."..index..".1"] = properties.top_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
  table[STEPNAME_TABLE..".body_segment_alpha."..index..".1"] = properties.body_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
  table[STEPNAME_TABLE..".bottom_segment_alpha."..index..".1"] = properties.bottom_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
end

--
-- populate language selection table
--
function CBCR_StepName_OnOverlayShow(custom_step_name, is_custom_name, is_notify_step)
  gSelectedIndex = 0
  gSliderY = gre.get_control_attrs(SLIDER_OVERLAY, "y").y

  local data = {}
  local index = 1
  
  data[SLIDER_OVERLAY..".grd_hidden"] = true
  
  -- top segment of slider
  StepName_TableEntry(data, index, { top_segment = true })
  
  -- slider content: selectable step names
  for i, step_name in ipairs(gStepNames) do
    index = i + 1
    if is_notify_step then
      gSelectedIndex = gStepNames[i].notify and index or gSelectedIndex
    else
      gSelectedIndex = ((not is_custom_name) and (custom_step_name == step_name.key)) and index or gSelectedIndex
    end
    StepName_TableEntry(data, index, {
      keyboard = gStepNames[i].notify and image_path("edit-recipe-lijst-icon-toetsenbord-x266.png") or "",
      step_name = i18n:get(step_name.key),
      selected = gSelectedIndex == index,
      body_segment = true
    })
  end
  data[NOTIFYTEXT_LAYER..".notify_text.text"] = is_notify_step and custom_step_name or ""
  
  -- slider content: custom step name
  index = index + 1
  gSelectedIndex = (is_custom_name or gSelectedIndex == 0) and index or gSelectedIndex
  gCustomStepName = (gSelectedIndex == index) and custom_step_name or i18n:get(NLS_CUSTOM_NAME_KEY)
  data[CUSTOMSTEPNAME_LAYER..".stepname_text.text"] = gCustomStepName
  StepName_TableEntry(data, index, {
    keyboard = image_path("edit-recipe-lijst-icon-toetsenbord-x266.png"),
    step_name = gCustomStepName,
    selected = gSelectedIndex == index,
    body_segment = true
  })
  
  -- bottom segment of slider
  index = index + 1
  StepName_TableEntry(data, index, { bottom_segment = true })
  
  gre.set_data(data)
  gre.set_table_attrs(STEPNAME_TABLE, { rows = index, yoffset = 0 })
  gre.set_layer_attrs(STEPNAME_LAYER, { hidden = false })
end

--
-- read available steps from json file
--
function CBCR_StepName_OnInit()
  local f = assert(io.open(config_path(STEPNAME_JSON_FILE), "r"))
  local t = f:read("*all")
  f:close()
  
  local json = json.decode(t)
  
  for _, entry in ipairs(json.recipe_steps) do
    if entry.enabled then
      table.insert(gStepNames, {
        key = entry.nls_id,
        preheat = entry.is_preheat or false,
        hold = entry.is_holding or false,
        notify = entry.is_notification or false
      })
    end
  end
end
