---------------------------------
-- recipe step time input overlay
---------------------------------

require("language")

local TIME_FIELD = "EditTime_Overlay.time_text.text"
local OVERWRITE_MINUTES_TIMEOUT_MS = 5000

local gEditingHours = false
local gEditingMinutes = false
local gOverwriteMinutes = true
local gOverwriteMinutesTimer = nil
local gTimeText = ""
local gMinTime = 0
local gMaxTime = 0

--
-- get the new time and close the dialog
--
function StepTime_GetAndClose()
  gre.set_layer_attrs("CR_Screen.EditTime_Overlay", { hidden = true })
  
  local hours, minutes = gTimeText:match("(.*):(.*)")
  local current_time = (hours * 60) + minutes
  
  if current_time < gMinTime then current_time = gMinTime end
  if current_time > gMaxTime then current_time = gMaxTime end
  
  return current_time
end

--
-- formatting for rich text control, makes active segment gray
--
local function StepTime_FormatText(hours, minutes)
  local format_time = "<p style=\"color:black; text-align:center; vertical-align:baseline;\">%s:%s</p>"
  local format_hours = gEditingHours and "<span style=\"color:gray;\">%01d</span>" or "%01d"
  local format_minutes = gEditingMinutes and "<span style=\"color:gray;\">%02d</span>" or "%02d"

  return string.format(format_time, string.format(format_hours, hours), string.format(format_minutes, minutes))
end

--
-- key pressed
--
function CBCR_StepTime_InputKeyEvent(mapargs)
  local code = mapargs.context_event_data.code
  
  local hours, minutes = gTimeText:match("(.*):(.*)")
  local oldhours = hours or 0
  local oldminutes = minutes or 0
  
  if gOverwriteMinutesTimer ~= nil then
    gre.timer_clear_timeout(gOverwriteMinutesTimer)
    gOverwriteMinutesTimer = nil
  end
  
  if code == 8 then
    -- backspace, remove last char
    if gEditingHours then
      hours = string.sub(hours, 1, #hours-1)
    elseif gEditingMinutes then
      if gTimeText[#gTimeText] == ":" or minutes == "00" then
        gEditingHours = true
        gEditingMinutes = false
        hours = "0"
      else
        minutes = string.sub(minutes, 1, #minutes-1)
      end
    end
  else
    -- append char
    for _, char in ipairs({'0', '1', '2', '3', '4', '5', '6', '7', '8', '9'}) do
      if string.char(code) == char then
        if gEditingHours and tonumber(char) <= (gMaxTime / 60) then
          hours = char
          gEditingHours = false
          gEditingMinutes = true
          gOverwriteMinutes = true
        elseif gEditingMinutes then
          if gOverwriteMinutes then
            minutes = char
            gOverwriteMinutes = false
          else
            minutes = minutes..char
          end
        end
      end
    end
  end
  
  if gEditingMinutes then
    gOverwriteMinutesTimer = gre.timer_set_timeout(
      function()
        gOverwriteMinutes = true
        gOverwriteMinutesTimer = nil
      end, OVERWRITE_MINUTES_TIMEOUT_MS)
  end
  
  if hours == "" then hours = "0" end
  if tonumber(hours) > (gMaxTime / 60) then hours = oldhours end
  if minutes == "" then minutes = "0" end
  if tonumber(minutes) > 59 then minutes = tonumber(string.sub(minutes, #minutes, #minutes)) end
  
  gTimeText = string.format("%01d:%02d", tonumber(hours), tonumber(minutes))
  gre.set_value(TIME_FIELD, StepTime_FormatText(hours, minutes))
end

--
-- show edit step time dialog
--
function CBCR_StepTime_OnOverlayShow(actual, min, max)
  if actual < min and actual ~= 0 then actual = min end
  if actual > max then actual = max end
  
  gEditingHours = true
  gEditingMinutes = false
  gOverwriteMinutes = true
  gOverwriteMinutesTimer = nil
  gMinTime = min
  gMaxTime = max
  
  local minutes = actual % 60
  local hours = (actual - minutes) / 60
  gTimeText = string.format("%01d:%02d", hours, minutes)

  local data = {}
  
  data["EditTime_Overlay.min_text.text"] = string.format("%s %01d:%02d", i18n:get("NLS_MIN"), gMinTime / 60, gMinTime % 60)
  data["EditTime_Overlay.max_text.text"] = string.format("%s %01d:%02d", i18n:get("NLS_MAX"), gMaxTime / 60, gMaxTime % 60)
  data[TIME_FIELD] = StepTime_FormatText(hours, minutes)
  
  gre.set_data(data)
  
  gre.set_layer_attrs("CR_Screen.EditTime_Overlay", { hidden = false })
  gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = false })
end
