--------------------------------------------------
-- circle class to calculate positions from angles
--------------------------------------------------

-- circle class
Circle = {}
Circle.__index = Circle

-- Circle.origin = {x=0,y=0}
-- Circle.point  = {x=0,y=0}

--
-- create a new circle instance from an origin and point on the circle
--
function Circle.new(o)
  assert(o.origin and o.point, "origin and point required")
  assert(o.origin.x and o.origin.y and o.point.x and o.point.y, "origin and point require x and y coordinates")

  o = o or {}
  setmetatable(o, Circle)
  return o
end

--
-- calculate the circle radius from the origin and a point on it
--
local function calculate_radius(origin, point)
  return math.sqrt((origin.x - point.x)^2 + (origin.y - point.y)^2)
end

--
-- get the radius of the circle
--
function Circle:radius()
  self.r = self.r or calculate_radius(self.origin, self.point)
  return self.r
end

-- precalculated sine and cosine tables for angles from 122 -> 399 degrees
local SINCOS_START_ANGLE = 122
local SINCOS_END_ANGLE   = 399
local SINCOS_OFFSET      = SINCOS_START_ANGLE - 1
local gSinTable = {
   0.848 , 0.8387, 0.829 , 0.8192, 0.809 , 0.7986, 0.788 , 0.7771, 0.766 , 0.7547,
   0.7431, 0.7314, 0.7193, 0.7071, 0.6947, 0.682 , 0.6691, 0.6561, 0.6428, 0.6293,
   0.6157, 0.6018, 0.5878, 0.5736, 0.5592, 0.5446, 0.5299, 0.515 , 0.5   , 0.4848,
   0.4695, 0.454 , 0.4384, 0.4226, 0.4067, 0.3907, 0.3746, 0.3584, 0.342 , 0.3256,
   0.309 , 0.2924, 0.2756, 0.2588, 0.2419, 0.225 , 0.2079, 0.1908, 0.1736, 0.1564,
   0.1392, 0.1219, 0.1045, 0.0872, 0.0698, 0.0523, 0.0349, 0.0175, 0     ,-0.0175,
  -0.0349,-0.0523,-0.0698,-0.0872,-0.1045,-0.1219,-0.1392,-0.1564,-0.1736,-0.1908,
  -0.2079,-0.225 ,-0.2419,-0.2588,-0.2756,-0.2924,-0.309 ,-0.3256,-0.342 ,-0.3584,
  -0.3746,-0.3907,-0.4067,-0.4226,-0.4384,-0.454 ,-0.4695,-0.4848,-0.5   ,-0.515 ,
  -0.5299,-0.5446,-0.5592,-0.5736,-0.5878,-0.6018,-0.6157,-0.6293,-0.6428,-0.6561,
  -0.6691,-0.682 ,-0.6947,-0.7071,-0.7193,-0.7314,-0.7431,-0.7547,-0.766 ,-0.7771,
  -0.788 ,-0.7986,-0.809 ,-0.8192,-0.829 ,-0.8387,-0.848 ,-0.8572,-0.866 ,-0.8746,
  -0.8829,-0.891 ,-0.8988,-0.9063,-0.9135,-0.9205,-0.9272,-0.9336,-0.9397,-0.9455,
  -0.9511,-0.9563,-0.9613,-0.9659,-0.9703,-0.9744,-0.9781,-0.9816,-0.9848,-0.9877,
  -0.9903,-0.9925,-0.9945,-0.9962,-0.9976,-0.9986,-0.9994,-0.9998,-1     ,-0.9998,
  -0.9994,-0.9986,-0.9976,-0.9962,-0.9945,-0.9925,-0.9903,-0.9877,-0.9848,-0.9816,
  -0.9781,-0.9744,-0.9703,-0.9659,-0.9613,-0.9563,-0.9511,-0.9455,-0.9397,-0.9336,
  -0.9272,-0.9205,-0.9135,-0.9063,-0.8988,-0.891 ,-0.8829,-0.8746,-0.866 ,-0.8572,
  -0.848 ,-0.8387,-0.829 ,-0.8192,-0.809 ,-0.7986,-0.788 ,-0.7771,-0.766 ,-0.7547,
  -0.7431,-0.7314,-0.7193,-0.7071,-0.6947,-0.682 ,-0.6691,-0.6561,-0.6428,-0.6293,
  -0.6157,-0.6018,-0.5878,-0.5736,-0.5592,-0.5446,-0.5299,-0.515 ,-0.5   ,-0.4848,
  -0.4695,-0.454 ,-0.4384,-0.4226,-0.4067,-0.3907,-0.3746,-0.3584,-0.342 ,-0.3256,
  -0.309 ,-0.2924,-0.2756,-0.2588,-0.2419,-0.225 ,-0.2079,-0.1908,-0.1736,-0.1564,
  -0.1392,-0.1219,-0.1045,-0.0872,-0.0698,-0.0523,-0.0349,-0.0175, 0     , 0.0175,
   0.0349, 0.0523, 0.0698, 0.0872, 0.1045, 0.1219, 0.1392, 0.1564, 0.1736, 0.1908,
   0.2079, 0.225 , 0.2419, 0.2588, 0.2756, 0.2924, 0.309 , 0.3256, 0.342 , 0.3584,
   0.3746, 0.3907, 0.4067, 0.4226, 0.4384, 0.454 , 0.4695, 0.4848, 0.5   , 0.515 ,
   0.5299, 0.5446, 0.5592, 0.5736, 0.5878, 0.6018, 0.6157, 0.6293
}
local gCosTable = {
  -0.5299,-0.5446,-0.5592,-0.5736,-0.5878,-0.6018,-0.6157,-0.6293,-0.6428,-0.6561,
  -0.6691,-0.682 ,-0.6947,-0.7071,-0.7193,-0.7314,-0.7431,-0.7547,-0.766 ,-0.7771,
  -0.788 ,-0.7986,-0.809 ,-0.8192,-0.829 ,-0.8387,-0.848 ,-0.8572,-0.866 ,-0.8746,
  -0.8829,-0.891 ,-0.8988,-0.9063,-0.9135,-0.9205,-0.9272,-0.9336,-0.9397,-0.9455,
  -0.9511,-0.9563,-0.9613,-0.9659,-0.9703,-0.9744,-0.9781,-0.9816,-0.9848,-0.9877,
  -0.9903,-0.9925,-0.9945,-0.9962,-0.9976,-0.9986,-0.9994,-0.9998,-1     ,-0.9998,
  -0.9994,-0.9986,-0.9976,-0.9962,-0.9945,-0.9925,-0.9903,-0.9877,-0.9848,-0.9816,
  -0.9781,-0.9744,-0.9703,-0.9659,-0.9613,-0.9563,-0.9511,-0.9455,-0.9397,-0.9336,
  -0.9272,-0.9205,-0.9135,-0.9063,-0.8988,-0.891 ,-0.8829,-0.8746,-0.866 ,-0.8572,
  -0.848 ,-0.8387,-0.829 ,-0.8192,-0.809 ,-0.7986,-0.788 ,-0.7771,-0.766 ,-0.7547,
  -0.7431,-0.7314,-0.7193,-0.7071,-0.6947,-0.682 ,-0.6691,-0.6561,-0.6428,-0.6293,
  -0.6157,-0.6018,-0.5878,-0.5736,-0.5592,-0.5446,-0.5299,-0.515 ,-0.5   ,-0.4848,
  -0.4695,-0.454 ,-0.4384,-0.4226,-0.4067,-0.3907,-0.3746,-0.3584,-0.342 ,-0.3256,
  -0.309 ,-0.2924,-0.2756,-0.2588,-0.2419,-0.225 ,-0.2079,-0.1908,-0.1736,-0.1564,
  -0.1392,-0.1219,-0.1045,-0.0872,-0.0698,-0.0523,-0.0349,-0.0175, 0     , 0.0175,
   0.0349, 0.0523, 0.0698, 0.0872, 0.1045, 0.1219, 0.1392, 0.1564, 0.1736, 0.1908,
   0.2079, 0.225 , 0.2419, 0.2588, 0.2756, 0.2924, 0.309 , 0.3256, 0.342 , 0.3584,
   0.3746, 0.3907, 0.4067, 0.4226, 0.4384, 0.454 , 0.4695, 0.4848, 0.5   , 0.515 ,
   0.5299, 0.5446, 0.5592, 0.5736, 0.5878, 0.6018, 0.6157, 0.6293, 0.6428, 0.6561,
   0.6691, 0.682 , 0.6947, 0.7071, 0.7193, 0.7314, 0.7431, 0.7547, 0.766 , 0.7771,
   0.788 , 0.7986, 0.809 , 0.8192, 0.829 , 0.8387, 0.848 , 0.8572, 0.866 , 0.8746,
   0.8829, 0.891 , 0.8988, 0.9063, 0.9135, 0.9205, 0.9272, 0.9336, 0.9397, 0.9455,
   0.9511, 0.9563, 0.9613, 0.9659, 0.9703, 0.9744, 0.9781, 0.9816, 0.9848, 0.9877,
   0.9903, 0.9925, 0.9945, 0.9962, 0.9976, 0.9986, 0.9994, 0.9998, 1     , 0.9998,
   0.9994, 0.9986, 0.9976, 0.9962, 0.9945, 0.9925, 0.9903, 0.9877, 0.9848, 0.9816,
   0.9781, 0.9744, 0.9703, 0.9659, 0.9613, 0.9563, 0.9511, 0.9455, 0.9397, 0.9336,
   0.9272, 0.9205, 0.9135, 0.9063, 0.8988, 0.891 , 0.8829, 0.8746, 0.866 , 0.8572,
   0.848 , 0.8387, 0.829 , 0.8192, 0.809 , 0.7986, 0.788 , 0.7771
}

--
-- get the sine and cosine of an angle
--
local function sin_and_cos(angle)
  local index = math.floor(angle + 0.5) - SINCOS_OFFSET
  
  local s = gSinTable[index] or math.sin(math.rad(angle))
  local c = gCosTable[index] or math.cos(math.rad(angle))

  return s, c
end

--
-- convert an angle in degrees to a position on the circle, with optional offset
--
function Circle:position(angle, offset)
  offset = offset or 0
  
  local sin, cos = sin_and_cos(angle)
  local r = self:radius()
  local x = (r * cos) + self.origin.x + offset
  local y = (r * sin) + self.origin.y + offset
  
  return x, y
end

--
-- calculate sine and cosine and print them to stdout
--
local function precalculate_sine_cosine(from, to)
  local sine   = function(a) return math.sin(math.rad(a)) end
  local cosine = function(a) return math.cos(math.rad(a)) end
  local round  = function(x) return math.floor((x * 10000) + 0.5) / 10000  end

  for i=from,to do
    print(round(sine(i))..",")
  end
  print()
  for i=from,to do
    print(round(cosine(i))..",")
  end
end

--
-- benchmark calculating sin/cos vs lookup table
--
function benchmark_sine_cosine(count)
  local from = SINCOS_START_ANGLE
  local to = SINCOS_END_ANGLE

  local now = gre.mstime()
  for i=1,count do
    for a=from,to do
      sin_and_cos(a)
    end
  end
  local lookup_time = gre.mstime() - now
  
  now = gre.mstime()
  for i=1,count do
    for a=from,to do
      local r = math.rad(a)
      math.sin(r)
      math.cos(r)
    end
  end
  local calculate_time = gre.mstime() - now
  
  return lookup_time, calculate_time
end

--precalculate_sine_cosine(SINCOS_START_ANGLE, SINCOS_END_ANGLE)
--benchmark_sine_cosine(100000)
