------------------------------------
-- implementation of the menu slider
------------------------------------

require("Time")
require("Dialog")
require("events")
require("Settings")
require("ActiveCleanProg")
require("ActiveRecipe")
require("CleanTotalTimes")

--[[ Status bar at the top ]]--

local WIFI_MODE_SMART_CONNECT = 1  -- smart connect active (cloud connection, client mode)
local WIFI_MODE_DIRECT_CONNECT = 2 -- direct connect active (smartphone connection, access point mode)
local WIFI_NOT_CONNECTED = 0       -- not connected to wifi (only meaningful when wifi mode is smart connect)
local WIFI_CONNECTED = 1           -- connected to wifi (only meaningful when wifi mode is smart connect)
local WIFI_TRAFFIC = 1             -- indicate there was MQTT wifi traffic the last 'WIFI_MQTT_TRAFFIC_TIMEOUT' seconds
--
-- update system time (internal + status bar + idle screen)
--
local function LiveUpdateTime(mapargs)
  set_time(Settings:get("SYS_TIME"))

  local timestamp, ampm = format_time()
  if ampm then timestamp = string.format("%s %s", timestamp, ampm) end

  local font = gre.get_value("Status_Layer.Clock_Text.fontName")
  local font_size = gre.get_value("Status_Layer.Clock_Text.fontSize")
  local timestamp_width = gre.get_string_size(font, font_size, timestamp).width
  local wifi_icon_width = gre.get_value("Status_Layer.Wifi_Icon.grd_width")

  local data = {}
  data["Status_Layer.Wifi_Icon.grd_x"] = 480 - 15 - timestamp_width - wifi_icon_width
  data["Status_Layer.Clock_Text.grd_x"] = 480 - 15 - timestamp_width
  data["Status_Layer.Clock_Text.grd_width"] = timestamp_width
  data["Status_Layer.Clock_Text.text"] = timestamp
  gre.set_data(data)

  gre.send_event("update_time")
end

--
-- update number of active errors
-- if active_errors > 0 then display wrench icon in status bar
-- if active_errors == 0 then hide wrench icon in status bar
--
local function LiveUpdateErrors(mapargs)
  local active_errors = Settings:get("NR_ACTIVE_ERROR")
  gre.set_value("Status_Layer.Wrench_Icon.grd_hidden", active_errors == 0)
end

--
-- enable or disable cleaning
-- if cleaning is not enabled, replace the cleaning icon on the quick launch
-- bar with the clean screen icon
--
local function EnableDisableCleaning(mapargs)
  local cleaning_enabled = Settings:get("CLEANING_SUPPORTED") == 1
  local cleaning_button_disabled = gre.get_value("Control_Layer.Control_Small_Group.Cleaning_Button_Disabled.grd_hidden") == 0

  local data = {}

  -- icons on home idle screen
  data["Idle_Layer.menu_group.grd_hidden"] = not cleaning_enabled
  data["Idle_Layer.menu_group_noclean.grd_hidden"] = cleaning_enabled

  -- quick launch icons
  data["Control_Layer.Control_Small_Group.Cleaning_Button.grd_hidden"] = not cleaning_enabled or cleaning_button_disabled
  data["Control_Layer.Control_Small_Group.Cleaning_Button_Disabled.grd_hidden"] = not cleaning_enabled or not cleaning_button_disabled
  data["Control_Layer.Control_Small_Group.Recipes_Button.grd_x"] = cleaning_enabled and 114 or 143
  data["Control_Layer.Control_Small_Group.Recipes_Button_Disabled.grd_x"] = cleaning_enabled and 114 or 134
  data["Control_Layer.Control_Small_Group.Home_Button.grd_x"] = cleaning_enabled and 201 or 259
  data["Control_Layer.Control_Small_Group.Home_Button_Disabled.grd_x"] = cleaning_enabled and 201 or 259

  gre.set_data(data)
end

--
-- commissioning state change: show/hide not-commissioned icon
--
local function CommissionStateChanged(mapargs)
  gre.set_value("Status_Layer.Commission_Icon.grd_hidden", Settings:get_def("COMMISSION_TIME", 0) ~= 0)
end

--
-- message count changed, show/hide messages icon
--
local function MessageCountChanged(mapargs)
  local message_count = Settings:get("MSG_COUNT")
  local any_message = message_count > 0

  local data = {}
  data["Status_Layer.Messages_Icon.grd_hidden"] = not any_message
  data["Control_Layer.Control_Big_Group.Messages_Region_Disabled.grd_hidden"] = any_message
  data["Control_Layer.Control_Big_Group.Messages_Region.grd_hidden"] = not any_message
  gre.set_data(data)

  if mapargs.context_screen == "MSG_Screen" then CBMSG_OnScreenShow(mapargs) end
end

--
-- wifi mode or connection status changed
--
local function WifiStatusChanged(mapargs)
  local wifi_icon = ""

  local mode = Settings:get("WIFI_MODE")
  if mode == WIFI_MODE_SMART_CONNECT then
    local connected = Settings:get("WIFI_CONNECTED")
    if connected == WIFI_CONNECTED then
      local traffic = (Settings:get("WIFI_TRAFFIC") == WIFI_TRAFFIC)
      wifi_icon = image_path(traffic and "wifi active 8mm.png" or "Wifi 8mm.png")
    else
      wifi_icon = image_path("no-wifi 8mm.png")
    end
  elseif mode == WIFI_MODE_DIRECT_CONNECT then
    wifi_icon = image_path("icon-connectivity-B.png")
  end

  local data = {}
  data["Status_Layer.Wifi_Icon.image"] = wifi_icon

  local no_wifi = (Settings:get("CONNECT_DIRECT") == 0) and (Settings:get("CONNECT_SMART") == 0)
  if Settings:get("WIFI_AVAILABLE") == 0 then
    no_wifi = true
  end

  -- only show wifi status icon at top of screen, when there is a wifi module and it is turned on.
  data["Status_Layer.Wifi_Icon.grd_hidden"] = no_wifi

  -- show dimmed screenshot icon button, when there is no wifi module or wifi connection available.
  data["Control_Layer.Control_Big_Group.Camera_Region_Disabled.grd_hidden"] = not no_wifi
  data["Control_Layer.Control_Big_Group.Camera_Region.grd_hidden"] = no_wifi
  gre.set_data(data)
end

--
-- handle io_demo_mode_changed event:
--
function CBStatusBar_OnDemoModeChanged()
  DemoModeChanged()
end

-- Indicate if demo mode is active by showing "DEMO" icon on the top status bar.
function DemoModeChanged()
  local demo_mode_active = false
  if Settings:get("IS_DEMO_ACTIVE") ~= 0 then
    demo_mode_active = true
  end
  -- only show demo icon at top of screen when demo parameter is turned on.
  gre.set_value("Status_Layer.Demo_Icon.grd_hidden", not demo_mode_active)
end

--
-- actions linked to live variable updates
--
local gLiveUpdateActionMap = {
  ["SYS_TIME"] = LiveUpdateTime,
  ["NR_ACTIVE_ERROR"] = LiveUpdateErrors,
  ["CLEANING_SUPPORTED"] = EnableDisableCleaning,
  ["COMMISSION_TIME"] = CommissionStateChanged,
  ["MSG_COUNT"] = MessageCountChanged,
  ["WIFI_MODE"] = WifiStatusChanged,
  ["WIFI_CONNECTED"] = WifiStatusChanged,
  ["WIFI_TRAFFIC"] = WifiStatusChanged
}

--
-- update the clock and error indicator in the status bar
--
function CBStatusBar_LiveUpdate(mapargs)
  local live_action = gLiveUpdateActionMap[mapargs.context_event_data.id]
  if live_action then live_action(mapargs) end
end

--
-- update the status bar (called on every "gre.screenshow.pre" event, when changing to another screen)
--
function CBStatusBar_Update(mapargs)
  WifiStatusChanged(mapargs)
  DemoModeChanged()
end

--
-- wifi status icon at top of screen was touched
--
function CBStatusBar_OnWifiIconPressed(mapargs)
  if Settings:get("WIFI_MODE") == WIFI_MODE_SMART_CONNECT or Settings:get("WIFI_MODE") == WIFI_MODE_DIRECT_CONNECT and mapargs.context_screen ~= "WIFI_Screen" then
    gre.set_value("toScreen", "WIFI_Screen")
    gre.send_event("switch_screen")
  end
end

--[[ Menu Dragging ]]--

local MENU_DISPLAY_Y = 70       -- y-coordinate of the menu when displayed
local MENU_HIDE_Y = 560          -- y-coordinate of the menu when hidden
local MENU_OFFSET = 25           -- offset for finger tracking

local gDragging = false          -- is the menu being dragged?
local gPrevY = nil               -- previous y coordinate
local gDirection = nil           -- direction of motion (true for down, false for up)
local gDisplayMenuActive = false -- display menu animation running?
local gHideMenuActive = false    -- hide menu animation running?
local gUpDownIndicator = false
local gMenuPressed = false
local gMenuPressY = 0
local gMenuPressYStart = 0
local gMenuGoingUp = false
local gMenuScrolling = false

local function log(message)
  if false then
    print(message)
  end
end

local function EnableDisableCleaning(mapargs)
  local cleaning_enabled = Settings:get("CLEANING_SUPPORTED") == 1
  local cleaning_button_disabled = gre.get_value("Control_Layer.Control_Small_Group.Cleaning_Button_Disabled.grd_hidden") == 0

  local data = {}

  -- icons on home idle screen
  data["Idle_Layer.menu_group.grd_hidden"] = not cleaning_enabled
  data["Idle_Layer.menu_group_noclean.grd_hidden"] = cleaning_enabled

  -- quick launch icons
  data["Control_Layer.Control_Small_Group.Cleaning_Button.grd_hidden"] = not cleaning_enabled or cleaning_button_disabled
  data["Control_Layer.Control_Small_Group.Cleaning_Button_Disabled.grd_hidden"] = not cleaning_enabled or not cleaning_button_disabled
  data["Control_Layer.Control_Small_Group.Recipes_Button.grd_x"] = cleaning_enabled and 114 or 143
  data["Control_Layer.Control_Small_Group.Recipes_Button_Disabled.grd_x"] = cleaning_enabled and 114 or 134
  data["Control_Layer.Control_Small_Group.Home_Button.grd_x"] = cleaning_enabled and 201 or 259
  data["Control_Layer.Control_Small_Group.Home_Button_Disabled.grd_x"] = cleaning_enabled and 201 or 259

  gre.set_data(data)
end

--
-- enable/disable menu buttons
--
function Control_SetButtons(bStandby, bRecipes, bHome, bClean, bHelp, bMenu, bUpDownIndicator)
  gUpDownIndicator = bUpDownIndicator

  local cleaning_enabled = Settings:get("CLEANING_SUPPORTED") == 1

  local data = {}
  data["Control_Layer.Control_Small_Group.Standby_Button_Disabled.grd_hidden"] = bStandby
  data["Control_Layer.Control_Small_Group.Standby_Button.grd_hidden"] = not bStandby
  data["Control_Layer.Control_Small_Group.Recipes_Button_Disabled.grd_hidden"] = bRecipes
  data["Control_Layer.Control_Small_Group.Recipes_Button.grd_hidden"] = not bRecipes
  data["Control_Layer.Control_Small_Group.Home_Button_Disabled.grd_hidden"] = bHome
  data["Control_Layer.Control_Small_Group.Home_Button.grd_hidden"] = not bHome
  data["Control_Layer.Control_Small_Group.Cleaning_Button_Disabled.grd_hidden"] = not cleaning_enabled or bClean
  data["Control_Layer.Control_Small_Group.Cleaning_Button.grd_hidden"] = not cleaning_enabled or not bClean
  data["Control_Layer.Control_Small_Group.Help_Button_Disabled.grd_hidden"] = bHelp
  data["Control_Layer.Control_Small_Group.Help_Button.grd_hidden"] = not bHelp
  data["Control_Layer.menu_enabled"] = bMenu
  data["Control_Layer.open_close_group.grd_hidden"] = not gUpDownIndicator
  data["Control_Layer.open_close_group.close_img.grd_gidden"] = true
  data["Control_Layer.open_close_group.open_img.grd_gidden"] = false
  gre.set_data(data)
end

--
-- start the display menu animation
--
function CBControl_DisplayMenu()
  if gHideMenuActive then
    gre.animation_stop("hide_menu")
    gHideMenuActive = false
  end
  if not gDisplayMenuActive then
    gDisplayMenuActive = true
    gre.set_value("Control_Layer.Control_Small_Group.grd_hidden", true)
    gre.set_value("Control_Layer.Overlay.grd_hidden", false)

    gre.animation_trigger("display_menu")
  end
end

--
-- start the hide menu animation
--
function CBControl_HideMenu()
  if gDisplayMenuActive then
    gre.animation_stop("display_menu")
    gDisplayMenuActive = false
  end
  if not gHideMenuActive then
    gHideMenuActive = true
    gre.set_value("Control_Layer.Control_Big_Group.Close_Region.grd_hidden", true)
    gre.animation_trigger("hide_menu")
  end
end

--
-- Press to drag
--
--- @param gre#context mapargs
function CBControl_Press(mapargs)
  log("p")
  local ev_data = mapargs.context_event_data
  if (gre.get_value("Control_Layer.Control_Big_Group.Close_Region.grd_hidden") == 0)
        or (ev_data.y > 740) then
    if gre.get_value("Control_Layer.menu_enabled") == 1 then
      log("pressed")
      -- remember start position to cancel timer when scrolling
      gMenuPressY = mapargs.context_event_data.y
      gMenuPressYStart = gMenuPressY
      log(gMenuPressY)

      gMenuScrolling = false
      gMenuPressed = true
    end
  end
end

-- Move menu
--
--- @param gre#context mapargs
function CBControl_Motion(mapargs)
  if(gMenuPressed == true) then
    log("m")
    local ev_data = mapargs.context_event_data
    log(ev_data.y)
    log(gre.get_layer_attrs("Control_Layer", "y").y)

    local newY = gre.get_layer_attrs("Control_Layer", "y").y + ev_data.y - gMenuPressY
    log(newY)

    gMenuGoingUp = (gMenuPressY > ev_data.y)

    local swipeSensitivity = Settings:get("SWIPE_SENSITIVITY")

    if((gMenuScrolling == true) or (math.abs(gMenuPressYStart - ev_data.y) > swipeSensitivity)) then
      gMenuScrolling = true
      if(newY > MENU_DISPLAY_Y) then
        local data = {}
        data["Control_Layer.Control_Small_Group.grd_hidden"] = true
        data["Control_Layer.Overlay.grd_hidden"] = false
        data["Control_Layer.open_close_group.grd_hidden"] = true
        gre.set_data(data)
        gMenuPressY = ev_data.y
        log(gMenuPressY)
        log("newY = ")
        log(newY)
        gre.set_layer_attrs("Control_Layer", { y = newY })
      end
    end

  end
end

--
-- Release after press
--
--- @param gre#context mapargs
function CBControl_Release(mapargs)
  gMenuPressed = false
  log("r")

  if gMenuScrolling == true then
    if gMenuGoingUp then
      CBControl_DisplayMenu()
    else
      CBControl_HideMenu()
    end
    gMenuScrolling = false
  end
end

--
-- Close menu after release event above menu
--
--- @param gre#context mapargs
function CBControl_ReleaseOverlay(mapargs)
  gMenuPressed = false
  log("ro")

  CBControl_HideMenu()
end


--
-- Change screen when not scolling menu
--
--- @param screen
function CBControl_ChangeScreen(screen)
  gMenuPressed = false
  log("rr")

  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    gre.set_value("toScreen", screen)
    gre.send_event("switch_screen")
  end

end

--
-- Release recipe
--
--- @param gre#context mapargs
function CBControl_ReleaseRecipe(mapargs)
  local data = {}
  data["REC_Screen.StandbyDialog_Layer.grd_hidden"] = true
  gre.set_data(data)

  CBControl_ChangeScreen("REC_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnSettingsButtonReleased(mapargs)
  CBControl_ChangeScreen("SET_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnAboutButtonReleased(mapargs)
  CBControl_ChangeScreen("FRI_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnMessagesButtonReleased(mapargs)
  CBControl_ChangeScreen("MSG_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnHelpButtonReleased(mapargs)
  CBControl_ChangeScreen("HELP_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnLogMaintainanceButtonReleased(mapargs)
  CBControl_ChangeScreen("LOG_Screen")
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnCleanButtonReleased(mapargs)
  local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
  dlg:set_message(i18n:get("NLS_CLEAN_START"))
  dlg:add_button(i18n:get("NLS_NO"))
  dlg:add_button(i18n:get("NLS_YES"), function() CBControl_ChangeScreen("CLEAN_Screen") end)
  dlg:show()
end

--
-- Change screen or close menu
--
--- @param gre#context mapargs
function CBControl_OnCameraButtonReleased(mapargs)
  gMenuPressed = false

  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    CBControl_MakeScreenshot(mapargs)
  end
end

--
-- display_menu animation completed
--
function CBControl_DisplayMenuFinished()
  local data = {}
  data["Control_Layer.Control_Big_Group.Close_Region.grd_hidden"] = false
  --data["Control_Layer.open_close_group.close_img.grd_hidden"] = false
  --data["Control_Layer.open_close_group.open_img.grd_hidden"] = true
  gre.set_data(data)

  gDisplayMenuActive = false
end

--
-- hide_menu animation completed
--
function CBControl_HideMenuFinished()
  local data = {}
  data["Control_Layer.Control_Small_Group.grd_hidden"] = false
  data["Control_Layer.Overlay.grd_hidden"] = true
  data["Control_Layer.open_close_group.grd_hidden"] = not gUpDownIndicator
  --data["Control_Layer.open_close_group.close_img.grd_hidden"] = true
  --data["Control_Layer.open_close_group.open_img.grd_hidden"] = false
  gre.set_data(data)

  gHideMenuActive = false
end

--
-- Dummy function
--
--- @param gre#context mapargs
function CBControl_Dummy(mapargs)
  log("d")
end

--[[ Screenshot button ]]--

--
-- make a screenshot
--
function CBControl_MakeScreenshot(mapargs)
  gre.set_layer_attrs("TakeScreenshot_Layer", { hidden = false })
end

--[[ Menu buttons ]]--

--
-- home button pressed, show home screen or cleaning screen
--
function CBControl_OnHomeButtonPressed(mapargs)
  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    local data = {}
    data["COOK_Screen.Language_Layer.grd_hidden"] = true
    data["COOK_Screen.StandbyDialog_Layer.grd_hidden"] = true
    data["CLEANING_Screen.Language_Layer.grd_hidden"] = true
    data["CLEANING_Screen.StandbyDialog_Layer.grd_hidden"] = true
    gre.set_data(data)

    gre.set_value("toScreen", ActiveCleanProg:is_cleaning() and "CLEANING_Screen" or "COOK_Screen")
    gre.send_event("switch_screen")
  end
end

--
-- help button pressed
--
function CBControl_OnHelpButtonPressed(mapargs)
  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    gre.set_value("toScreen", "HELP_Screen")
    gre.send_event("switch_screen")
  end
end

--
-- cleaning button pressed, show selection dialog or go to cleaning screen
--
function CBControl_OnCleanButtonPressed(mapargs)
  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    local data = {}
    data["CLEANING_Screen.Language_Layer.grd_hidden"] = true
    data["CLEANING_Screen.StandbyDialog_Layer.grd_hidden"] = true
    gre.set_data(data)

    if mapargs.context_screen == "CLEANING_Screen" then
      if not ActiveCleanProg:is_cleaning() then
        Cleaning_TriggerProgramSelection()
      end
    elseif ActiveRecipe:is_cooking() then
      local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
      dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
      dlg:add_button(i18n:get("NLS_OK"))
      dlg:show()
    else
      Cleaning_TriggerProgramSelection()
      gre.set_value("toScreen", "CLEANING_Screen")
      gre.send_event("switch_screen")
    end
  end
end

function Cleaning_TriggerProgramSelection()
  -- request total time for all cleaning programs
  CleaningTotalTimes.showProgramSelection = true
  Event:get_cleantotaltimes()
  -- CBCleaning_OnCleanTotalTimesReply() will receive clean total times & show the program selection on the GUI.
end

--[[ Standby button ]]--

--
-- standby action selected
--
function CBControl_OnStandbyActionSelected(mapargs)
  local index = mapargs.context_row
  local action = gre.get_value("StandbyDialog_Layer.standby_group.standbyaction_table.action."..index..".1")

  local cancel_recipe = function()
    if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
      -- cancel active program
      Event:cancel_recipe()
    elseif ActiveRecipe:is_scheduled() then
      ActiveRecipe:set_scheduled_time(0)
      Event:cancel_recipe()
    elseif ActiveCleanProg:is_scheduled() then
      -- cancel scheduled program
      Event:live_set("SC_TIME", 0)
      ActiveCleanProg:cancel()
    end
  end

  local action_table = {
    [1] = function() end,
    [2] = cancel_recipe,
    [3] = function() Event:request_standby() end
  }

  local action_handler = action_table[action]
  if action_handler then
    action_handler()
  end
  gre.set_layer_attrs("StandbyDialog_Layer", {hidden=true})
end

--
-- open standby dialog
--
local function display_standby_dialog(mapargs)
  local table = "StandbyDialog_Layer.standby_group.standbyaction_table"
  local cell_height = gre.get_table_cell_attrs(table, 1, 1, "height").height

  local data = {}

  local index = 0

  -- back button
  index = index + 1
  data[table..".text."..index..".1"] = i18n:get("NLS_STANDBYDLG_BACK")
  data[table..".icon."..index..".1"] = image_path("icon-terug-voor-board-10-4-19.png")
  data[table..".icon_y_offset."..index..".1"] = -12
  data[table..".action."..index..".1"] = 1
  -- cancel recipe, scheduled recipe or cleaning program button
  if ActiveRecipe:is_cooking() or ActiveRecipe:is_scheduled() or ActiveCleanProg:is_cleaning() or ActiveCleanProg:is_scheduled() then
    index = index + 1
    if ActiveRecipe:is_cooking() or ActiveRecipe:is_scheduled() then
      data[table..".text."..index..".1"] = i18n:get("NLS_STANDBYDLG_CANCEL_RECIPE")
    else
      data[table..".text."..index..".1"] = i18n:get("NLS_STANDBYDLG_CANCEL_CLEANING")
    end
    data[table..".icon."..index..".1"] = image_path("icon-stop-voor-board-10-4-19.png")
    data[table..".icon_y_offset."..index..".1"] = -77
    data[table..".action."..index..".1"] = 2
  end
  -- standby button
  index = index + 1
  data[table..".text."..index..".1"] = i18n:get("NLS_STANDBYDLG_STANDBY")
  data[table..".icon."..index..".1"] = image_path("icon-uit-voor-board-10-4-19.png")
  data[table..".icon_y_offset."..index..".1"] = -138
  data[table..".action."..index..".1"] = 3

  local height = index * cell_height
  data[table..".grd_height"] = height
  data[table..".grd_y"] = 8 + ((204 - height) / 2)

  gre.set_data(data)
  gre.set_table_attrs(table, {rows=index})
  gre.set_layer_attrs("StandbyDialog_Layer", {hidden=false})
end

--
-- open standby dialog if needed
--
function CBControl_RequestStandby(mapargs)
  if gMenuScrolling == true then
    CBControl_Release(mapargs)
  else
    display_standby_dialog(mapargs)
  end
end
