------------------------------------------------------
-- implementation of the main screen (cooking recipes)
------------------------------------------------------

require("events")
require("Circle")
require("language")
require("ActiveRecipe")
require("Temperature")
require("Dialog")

local PROGRESS_ARC_IMG = image_path("cirkel-%d.png")        -- circular recipe progress bar
local TIME_FORMAT = "%01d:%02d"                             -- string format for total time "0:00"

local gRecipeName = ""     -- the name of the currently active recipe
local gRecipeIcon = ""     -- the large icon of the recipe
local gStepId = 1          -- variable keeping track of the next available id for step icons on the progress arc
local gStepIcons = {}      -- a list of all the step icons on the progress arc
local gStepIconIds = {}    -- a mapping of the step number to the id used for its icon on the progress arc
local gSwapTimer = nil     -- timer used to swap between remaining time and finish time
local gProgressArc = Circle.new{origin={x=240,y=331},point={x=240,y=135}} -- progress arc circle object
local gTemperature = Temperature.new{temp = 0} -- temperature
local gSlidingY = 0
local gSlidingHours = false
local gSlidingMinutes = false

--
-- delete all steps from the recipe progress arc
--
local function clear_step_icons()
  if #gStepIcons > 0 then
    gStepId = gStepIcons[#gStepIcons] + 1
  else
    gStepId = 1
  end

  for i, ctrl_i in ipairs(gStepIcons) do
    gre.delete_object("Cook_Layer_v2.steps_group.step_img_"..ctrl_i)
    gStepIcons[i] = nil
  end
end

--
-- build the progress arc
--
local function make_progressbar(data)
  local start = 50
  local endspace = 64
  local total = 360 - (start + endspace)

  local angle = start
  for i, step in ipairs(ActiveRecipe:get_recipe().steps) do
    if step.time > 0 then
      angle = angle + ((step.time / ActiveRecipe:get_time()) * total)
    end

    local new_x, new_y = gProgressArc:position(angle + 90)
    new_x = new_x - 12
    new_y = new_y - 13

    local icon_x_key = "Cook_Layer_v2.steps_group.step_img.grd_x"
    local icon_y_key = "Cook_Layer_v2.steps_group.step_img.grd_y"
    local icon_img_key = "Cook_Layer_v2.steps_group.step_img.image"
    local icon_hidden_key = "Cook_Layer_v2.steps_group.step_img.grd_hidden"
    local icon_angle_key = "Cook_Layer_v2.steps_group.step_img.angle"
    local icon_centerx_key = "Cook_Layer_v2.steps_group.step_img.centerX"
    local icon_centery_key = "Cook_Layer_v2.steps_group.step_img.centerY"
    local icon_xoffset_key = "Cook_Layer_v2.steps_group.step_img.x_offset"

    if i > 1 then
      local ctrl_i = gStepId
      if #gStepIcons > 0 then
        ctrl_i = gStepIcons[#gStepIcons] + 1
      end

      icon_x_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".grd_x"
      icon_y_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".grd_y"
      icon_img_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".image"
      icon_hidden_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".grd_hidden"
      icon_angle_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".angle"
      icon_centerx_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".centerX"
      icon_centery_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".centerY"
      icon_xoffset_key = "Cook_Layer_v2.steps_group.step_img_"..ctrl_i..".x_offste"

      gre.clone_object("Cook_Layer_v2.steps_group.step_img", "step_img_"..ctrl_i, "Cook_Layer_v2.steps_group")
      table.insert(gStepIcons, ctrl_i)
      gStepIconIds[i] = ctrl_i
    else
      gStepIconIds[i] = i
    end

    data[icon_x_key] = new_x
    data[icon_y_key] = new_y
    data[icon_img_key] = "images/streepje-voor-op-cirkel.png"
    data[icon_hidden_key] = false
    data[icon_angle_key] = angle - 180
  end
end

--[[ Event handlers for recipe ]]--

--
-- recipe progress timer tick (sec)
--
--- @param gre#context mapargs
function CBCOOK_OnRecipeStepProgress(mapargs)
  local step_time_passed = mapargs.context_event_data.step_passed
  local time_passed = mapargs.context_event_data.total_passed
  local total_time = ActiveRecipe:get_time() * 60

  local percent_done = math.floor(((time_passed / total_time) * 100) + 0.5)
  total_time = total_time + ActiveRecipe.cook_correction + ActiveRecipe.extra_time + ActiveRecipe.extra_time_accu
  local hours, minutes, seconds = calculate_time_seconds({}, (total_time - time_passed))

  -- get the previous image so we can remove it from the cache as it is no
  -- longer needed and thus only taking precious space
  local prev_img = gre.get_value("Cook_Layer_v2.arc_img.image")
  local new_img = string.format(PROGRESS_ARC_IMG, math.min(percent_done, 100))

  local data = {}
  data["Cook_Layer_v2.arc_img.image"] = new_img

  if not ActiveRecipe:is_scheduled() then
  --if time_passed ~= 0 then -- remove recipe scheduled time icon when recipe is busy.
    data["Cook_Layer_v2.delayed_recipe_start.grd_hidden"] = true
  end

  local seconds_remaining = total_time - time_passed
  if seconds_remaining > 59 then
    local rounded_minutes = seconds > 0 and minutes + 1 or minutes
    local rounded_hours = rounded_minutes == 60 and hours + 1 or hours
    rounded_minutes = rounded_minutes < 60 and rounded_minutes or 0
    data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, rounded_hours, rounded_minutes)
  else
    data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, 0, seconds_remaining)
  end

  if percent_done > 100 then
    data["Cook_Layer_v2.time_changed_img.grd_hidden"] = false
    data["Cook_Layer_v2.time_changed_img.image"] = "images/toegevoegde-tijd-rood-x365-y400.png"
  else
    data["Cook_Layer_v2.time_changed_img.grd_hidden"] = true
  end
  gre.set_data(data)

  -- remove the previous precentage done image from the cache, keeping all of
  -- them cached makes the application run into memory problems and crash
  if new_img ~= prev_img then
    gre.dump_resource("image", prev_img)
  end
end

--
-- extra time added in minutes (to last actual step of recipe)
--
function CBCOOK_OnTimeAdded(mapargs)
  local h, min, s = calculate_time_seconds({}, mapargs.context_event_data.extra_time)
  min = (h * 60) + min

  local data = {}
  data["AddTime_Layer.time_text.minutes"] = string.format("%02d", min)
  data["AddTime_Layer.time_text.minutes_less"] = min >= 1 and string.format("%02d", min - 1) or "30"
  data["AddTime_Layer.time_text.minutes_more"] = min <= 29 and string.format("%02d", min + 1) or "00"

  if ActiveRecipe:is_cooking() or mapargs.context_event_data.extra_time ~= 0 then
    data["Cook_Layer_v2.glow_img.grd_hidden"] = false
    data["Cook_Layer_v2.recipe_img.grd_hidden"] = false
    data["Cook_Layer_v2.finish_img.grd_hidden"] = true
  end

  gre.set_data(data)

  CBCOOK_OnAutoCorrect({context_event_data = {delta = ActiveRecipe.cook_correction}})
end

--
-- autocorrect event received
--
--- @param gre#context mapargs
function CBCOOK_OnAutoCorrect(mapargs)
  local data = {}

  local delta = mapargs.context_event_data.delta + ActiveRecipe.extra_time
  local correction_hours, correction_minutes, correction_seconds = calculate_time_seconds({}, math.abs(delta))
  local total_hours, total_minutes, total_seconds = calculate_time_seconds(ActiveRecipe:get_recipe().steps, (delta - (ActiveRecipe.seconds_passed + ActiveRecipe.step_passed)))
  local finish_time = format_time(now() + (total_hours * 3600) + (total_minutes * 60) + total_seconds)
  local finish_time_font = gre.get_value("Cook_Layer_v2.finish_time_text.fontName")
  local finish_time_font_size = gre.get_value("Cook_Layer_v2.finish_time_text.fontSize")
  local finish_time_width = gre.get_string_size(finish_time_font, finish_time_font_size, finish_time).width

  if total_hours >= 0 and total_minutes >= 0 and total_seconds >= 0 then
    local seconds_remaining = (total_hours * 3600) + (total_minutes * 60) + total_seconds
    if seconds_remaining > 59 then
      local rounded_minutes = total_seconds > 0 and total_minutes + 1 or total_minutes
      local rounded_hours = rounded_minutes == 60 and total_hours + 1 or total_hours
      rounded_minutes = rounded_minutes < 60 and rounded_minutes or 0
      data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, rounded_hours, rounded_minutes)
    else
      data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, 0, seconds_remaining)
    end
    data["Cook_Layer_v2.finish_time_text.text"] = finish_time
    data["Cook_Layer_v2.finish_time_text.grd_width"] = finish_time_width > 144 and finish_time_width or 144
  end

  if delta > 0 then
    data["Cook_Layer_v2.extra_time_text.grd_hidden"] = false
    data["Cook_Layer_v2.extra_time_text.text"] = string.format("+ %01d:%02d", correction_hours, correction_minutes + ((correction_seconds + 30)/60))
    data["Cook_Layer_v2.extra_time_text.color"] = 0xB32422
    data["Cook_Layer_v2.extra_time_img.grd_hidden"] = false
  elseif delta < 0 then
    data["Cook_Layer_v2.extra_time_text.grd_hidden"] = false
    data["Cook_Layer_v2.extra_time_text.text"] = string.format("- %01d:%02d", correction_hours, correction_minutes + ((correction_seconds + 30)/60))
    data["Cook_Layer_v2.extra_time_text.color"] = 0x00C400
    data["Cook_Layer_v2.extra_time_img.grd_hidden"] = false
  else
    data["Cook_Layer_v2.extra_time_text.grd_hidden"] = true
    data["Cook_Layer_v2.extra_time_img.grd_hidden"] = true
  end

  gre.set_data(data)
end


--
--
-- A scheduled recipe has actually started
-- triggered by "io_recipe_started" event.
--
function CBCOOK_OnRecipeStarted(mapargs)
  ActiveRecipe:set_scheduled_time(0)
  CBCOOK_OnScreenShow()
end

--
-- new recipe step starting
--
--- @param gre#context mapargs
function CBCOOK_OnRecipeStep(mapargs)
  local step = mapargs.context_event_data.step
  local active_step = ActiveRecipe:get_recipe().steps[step]
  local setpoint = Temperature.new{temp = active_step.temp}

  local data = {}
  data["Cook_Layer_v2.add_time_btn.grd_hidden"] = false
  data["Cook_Layer_v2.actualtemp_text.text"] = active_step.preheat and setpoint:to_string() or gTemperature:to_string()
  data["Cook_Layer_v2.actualtemp_text.color"] = not active_step.preheat and 0xFFFFFF or 0x000000
  data["Cook_Layer_v2.temperature_text.text"] = active_step.preheat and gTemperature:to_string() or setpoint:to_string()
  data["Cook_Layer_v2.temperature_text.color"] = active_step.preheat and 0xFFFFFF or 0x000000
  data["Cook_Layer_v2.recipe_steps_table.row_color."..(step - 1)..".1"] = 0x415B8F
  data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha."..(step - 1)..".1"] = 0
  data["Cook_Layer_v2.recipe_steps_table.row_color."..(step)..".1"] = 0x599EE0
  data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha."..(step)..".1"] = 255
  gre.set_data(data)

  -- scroll to new step in steps table
  local cell_attrs = gre.get_table_cell_attrs("Cook_Layer_v2.recipe_steps_table", 1, 1, "height")
  local yoffset = -1 * (step - 2) * cell_attrs.height
  gre.set_table_attrs("Cook_Layer_v2.recipe_steps_table", {yoffset = yoffset})

  -- reset cook correction
  mapargs.context_event_data.delta = 0
  CBCOOK_OnAutoCorrect(mapargs)
end

--
-- oven is preheating
--
function CBCOOK_OnPreheat()
  local data = {}
  data["Cook_Layer_v2.recipe_steps_table.row_color.1.1"] = 0x599EE0
  data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha.1.1"] = 255
  data["Cook_Layer_v2.warmup_img.grd_hidden"] = false
  data["Cook_Layer_v2.delayed_recipe_start.grd_hidden"] = true
  gre.set_data(data)
end

--
-- preheat done
--
function CBCOOK_OnPreheated()
  local data = {}
  data["Cook_Layer_v2.warmup_img.grd_hidden"] = true
  data["Cook_Layer_v2.delayed_recipe_start.grd_hidden"] = true
  gre.set_data(data)

  local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
  dlg:set_message(i18n:get("NLS_PREHEAT_DONE"))
  dlg:show()
end

--
-- oven is keeping the chickens warm
--
function CBCOOK_OnHolding()
  -- set progress arc to 100%
  -- set last step icon on progress arc to done
  -- set last step color in steps table to active

  CBCOOK_OnRecipeStep({context_event_data = {step = ActiveRecipe:get_active_step()}})

  local data = {}
  data["Cook_Layer_v2.arc_img.image"] = string.format(PROGRESS_ARC_IMG, 100)
  data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, 0, 0)
  data["Cook_Layer_v2.time_changed_img.grd_hidden"] = true
  data["Cook_Layer_v2.extra_time_img.grd_hidden"] = true
  data["Cook_Layer_v2.extra_time_text.grd_hidden"] = true
  gre.set_data(data)
end

--
-- entry function, starting a recipe
--
--- @param gre#context mapargs
function CBCOOK_StartRecipe(mapargs)
  local name = mapargs.context_event_data.recipe
  local starttime = mapargs.context_event_data.starttime
  local reference_cook = (mapargs.context_event_data.reference_cook == 1)

  gRecipeName = name
  ActiveRecipe:set_recipe(Recipes:get(name), name, reference_cook)
  ActiveRecipe:set_scheduled_time(starttime)
  Event:start_recipe(gRecipeName, reference_cook, starttime)
end

--
-- temperature update event received
--
function CBCOOK_OnTemperatureUpdate(mapargs)
  gTemperature:setC(mapargs.context_event_data.temp / 10.0)

  if mapargs.context_screen == "COOK_Screen" then
    if (ActiveRecipe:get_recipe() ~= nil) and (ActiveRecipe.active_step ~= nil) then
      local setpoint = Temperature.new{temp = ActiveRecipe:get_recipe().steps[ActiveRecipe.active_step].temp}
      local bPreheat = ActiveRecipe:is_preheating() or ActiveRecipe:is_loading()

      local text_field = bPreheat and "Cook_Layer_v2.temperature_text.text" or "Cook_Layer_v2.actualtemp_text.text"
      gre.set_value(text_field, gTemperature:to_string())
    end
  end
end

--
-- screen hide event, clean up step indicators on progress arc
--
function CBCOOK_OnScreenHide()
  clear_step_icons()

  if gSwapTimer ~= nil then
    gre.timer_clear_timeout(gSwapTimer)
    gSwapTimer = nil
  end
end

--
-- recipe is done
--
function CBCOOK_OnFinished()
  -- set progress arc to 100%
  -- set last step icon on progress arc to done
  -- set last step color in steps table to done
  if ActiveRecipe.canceled then
    CBCOOK_OnScreenShow()
  else
    local data = {}
    data["Cook_Layer_v2.arc_img.image"] = string.format(PROGRESS_ARC_IMG, 100)
    data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, 0, 0)
    data["Cook_Layer_v2.glow_img.grd_hidden"] = true
    data["Cook_Layer_v2.recipe_steps_table.row_color."..(#ActiveRecipe:get_recipe())..".1"] = 0x415B8F
    data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha."..(#ActiveRecipe:get_recipe())..".1"] = 0
    data["Cook_Layer_v2.time_changed_img.grd_hidden"] = true
    data["Cook_Layer_v2.extra_time_img.grd_hidden"] = true
    data["Cook_Layer_v2.extra_time_text.grd_hidden"] = true
    data["Cook_Layer_v2.recipe_img.grd_hidden"] = true
    data["Cook_Layer_v2.finish_img.grd_hidden"] = false
    gre.set_data(data)
  end
end

--
-- request system time
--
function CBCOOK_OnScreenShown()
  gre.send_event("gra.screen.hold")

  CBStatusBar_LiveUpdate({context_event_data = {id = "SYS_TIME"}})

  if ActiveRecipe.recipe ~= nil then
    local data = {}

    local step_time_passed = ActiveRecipe.step_passed
    local time_passed = ActiveRecipe.seconds_passed + step_time_passed
    local total_time = ActiveRecipe:get_time() * 60

    local percent_done = total_time == 0 and 0 or math.floor(((time_passed / total_time) * 100) + 0.5)
    local correction = ActiveRecipe.cook_correction + ActiveRecipe.extra_time
    local correction_hours, correction_minutes, correction_seconds = calculate_time_seconds({}, math.abs(correction))
    total_time = total_time + correction + ActiveRecipe.extra_time_accu
    local total_hours, total_minutes, total_seconds = calculate_time_seconds({}, (total_time - time_passed))
    local seconds_remaining = (total_hours * 3600) + (total_minutes * 60) + total_seconds
    local finish_time = format_time(now() + seconds_remaining)
    local finish_time_font = gre.get_value("Cook_Layer_v2.finish_time_text.fontName")
    local finish_time_font_size = gre.get_value("Cook_Layer_v2.finish_time_text.fontSize")
    local finish_time_width = gre.get_string_size(finish_time_font, finish_time_font_size, finish_time).width
    local temp = Temperature.new{temp = ActiveRecipe:get_recipe().steps[ActiveRecipe:get_active_step()].temp}

    gRecipeIcon = ActiveRecipe:get_recipe().icon
    if gRecipeIcon then -- may be nil
      gRecipeIcon = icon_path(gRecipeIcon)
      local large_icon = string.sub(gRecipeIcon, 1, -5) .. "_large.png"
      if file_exists(absolute_path(large_icon)) then gRecipeIcon = large_icon end
    else
      gRecipeIcon = ""
    end

    if ActiveRecipe:is_scheduled() then
      local timestamp, ampm = format_time(ActiveRecipe:scheduled_time())
      if ampm then timestamp = string.format("%s %s", timestamp, ampm) end
      data["Cook_Layer_v2.delayed_recipe_start.text"] = timestamp
      data["Cook_Layer_v2.delayed_recipe_start.grd_hidden"] = false
      gRecipeIcon = ""
    else
      data["Cook_Layer_v2.delayed_recipe_start.grd_hidden"] = true
    end


    data["COOK_Screen.Idle_Layer.grd_hidden"] = true
    data["Cook_Layer_v2.ReferenceCook_Img.grd_hidden"] = not ActiveRecipe:is_reference_cook()
    data["Cook_Layer_v2.arc_img.image"] = string.format(PROGRESS_ARC_IMG, math.min(percent_done, 100))
    if percent_done > 100 then
      data["Cook_Layer_v2.time_changed_img.grd_hidden"] = false
      data["Cook_Layer_v2.time_changed_img.image"] = "images/toegevoegde-tijd-rood-x365-y400.png"
    else
      data["Cook_Layer_v2.time_changed_img.grd_hidden"] = true
    end
    data["Cook_Layer_v2.temperature_text.text"] = temp:to_string()
    data["Cook_Layer_v2.actualtemp_text.grd_hidden"] = true
    data["Cook_Layer_v2.temperature_text.grd_hidden"] = false
    data["Cook_Layer_v2.time_text.grd_hidden"] = false
    if seconds_remaining > 59 then
      local rounded_minutes = total_seconds > 0 and total_minutes + 1 or total_minutes
      local rounded_hours = rounded_minutes == 60 and total_hours + 1 or total_hours
      rounded_minutes = rounded_minutes < 60 and rounded_minutes or 0
      data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, rounded_hours, rounded_minutes)
    else
      data["Cook_Layer_v2.time_text.text"] = string.format(TIME_FORMAT, 0, seconds_remaining)
    end
    data["Cook_Layer_v2.finish_time_text.grd_hidden"] = true
    data["Cook_Layer_v2.finish_time_text.text"] = finish_time
    data["Cook_Layer_v2.finish_time_text.grd_width"] = finish_time_width > 144 and finish_time_width or 144
    data["Cook_Layer_v2.recipe_text.grd_hidden"] = false
    data["Cook_Layer_v2.recipe_text.text"] = ActiveRecipe.name
    data["Cook_Layer_v2.add_time_btn.grd_hidden"] = false
    data["Cook_Layer_v2.finish_at_img.grd_hidden"] = true
    data["Cook_Layer_v2.remaining_time_img.grd_hidden"] = false
    data["Cook_Layer_v2.recipe_img.image"] = gRecipeIcon
    data["Cook_Layer_v2.glow_img.grd_hidden"] = false
    data["Cook_Layer_v2.recipe_steps_table.grd_hidden"] = false
    data["Cook_Layer_v2.steps_group.grd_hidden"] = false
    data["Cook_Layer_v2.warmup_img.grd_hidden"] = not ActiveRecipe:is_preheating()

    local cell_attrs = gre.get_table_cell_attrs("Cook_Layer_v2.recipe_steps_table", 1, 1, "height")
    local yoffset = -1 * (ActiveRecipe.active_step - 2) * cell_attrs.height
    gre.set_table_attrs("Cook_Layer_v2.recipe_steps_table", {rows = #ActiveRecipe:get_recipe().steps, yoffset = yoffset})

    for i, step in ipairs(ActiveRecipe:get_recipe().steps) do
      local hours, minutes = calculate_time({{time = step.time}})
      temp = Temperature.new{temp = step.temp}

      data["Cook_Layer_v2.recipe_steps_table.step_text."..i..".1"] = (step.custom or step.notify) and step.name or i18n:get(step.name)
      data["Cook_Layer_v2.recipe_steps_table.probe_text."..i..".1"] = ""
      data["Cook_Layer_v2.recipe_steps_table.time_text."..i..".1"] = not step.notify and (step.time == 0 and "-" or string.format(TIME_FORMAT, hours, minutes)) or ""
      data["Cook_Layer_v2.recipe_steps_table.notification_icon."..i..".1"] = step.notify and 255 or 0
      data["Cook_Layer_v2.recipe_steps_table.temp_text."..i..".1"] = step.temp == 0 and "-" or temp:to_string()
      data["Cook_Layer_v2.recipe_steps_table.fan_text."..i..".1"] = ""
      data["Cook_Layer_v2.recipe_steps_table.steam_text."..i..".1"] = ""
      data["Cook_Layer_v2.recipe_steps_table.valve_img."..i..".1"] = ""
      data["Cook_Layer_v2.recipe_steps_table.valve_text."..i..".1"] = ""
      data["Cook_Layer_v2.recipe_steps_table.row_color."..i..".1"] = 0x415B8F
      data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha."..i..".1"] = 0
    end
    data["Cook_Layer_v2.recipe_steps_table.row_color."..ActiveRecipe.active_step..".1"] = 0x599EE0
    data["Cook_Layer_v2.recipe_steps_table.selected_step_alpha."..ActiveRecipe.active_step..".1"] = 255

    local active_step = ActiveRecipe:get_recipe().steps[ActiveRecipe.active_step]
    local setpoint = Temperature.new{temp = active_step.temp}

    data["Cook_Layer_v2.add_time_btn.grd_hidden"] = active_step.hold or false
    data["Cook_Layer_v2.actualtemp_text.text"] = active_step.preheat and setpoint:to_string() or gTemperature:to_string()
    data["Cook_Layer_v2.actualtemp_text.color"] = not active_step.preheat and 0xFFFFFF or 0x000000
    data["Cook_Layer_v2.temperature_text.text"] = active_step.preheat and gTemperature:to_string() or setpoint:to_string()
    data["Cook_Layer_v2.temperature_text.color"] = active_step.preheat and 0xFFFFFF or 0x000000

    -- HeatUp recipe is not allowed to be paused
    data["Cook_Layer_v2.pause_btn.grd_hidden"] = ActiveRecipe:is_heatup() or false

    make_progressbar(data)

    gre.set_data(data)

    if ActiveRecipe.extra_time > 0 and not ActiveRecipe:is_done() then
      CBCOOK_OnTimeAdded({context_event_data = {extra_time = ActiveRecipe.extra_time}})
      CBCOOK_OnRecipeStepProgress({context_event_data = {total_passed = ActiveRecipe.seconds_passed + ActiveRecipe.step_passed, step_passed = ActiveRecipe.step_passed}})
    end

    if ActiveRecipe:is_done() then
      CBCOOK_OnFinished()
    end
  end

  gre.send_event("gra.screen.release")
end

--
-- prepare to display a recipe and gray out home button
--
function CBCOOK_OnScreenShow()
  local cleaning_enabled = Settings:get("CLEANING_SUPPORTED") == 1

  local data = {}
  data["Idle_Layer.language_button.image"] = i18n:meta_data(i18n:language()).icon or image_path("vlag-engels x425-y59.png")
  data["Idle_Layer.menu_group.grd_hidden"] = not cleaning_enabled
  data["Idle_Layer.menu_group_noclean.grd_hidden"] = cleaning_enabled
  data["COOK_Screen.Idle_Layer.grd_hidden"] = false
  data["Cook_Layer_v2.temperature_text.text"] = ""
  data["Cook_Layer_v2.temperature_text.grd_hidden"] = false
  data["Cook_Layer_v2.actualtemp_text.text"] = gTemperature:to_string()
  data["Cook_Layer_v2.actualtemp_text.grd_hidden"] = true
  data["Cook_Layer_v2.arc_img.image"] = "images/cirkel-donkerblauw.png"
  data["Cook_Layer_v2.time_text.grd_hidden"] = true
  data["Cook_Layer_v2.finish_time_text.grd_hidden"] = true
  data["Cook_Layer_v2.recipe_text.grd_hidden"] = true
  data["Cook_Layer_v2.time_changed_img.grd_hidden"] = true
  data["Cook_Layer_v2.add_time_btn.grd_hidden"] = true
  data["Cook_Layer_v2.extra_time_text.grd_hidden"] = true
  data["Cook_Layer_v2.extra_time_img.grd_hidden"] = true
  data["Cook_Layer_v2.finish_at_img.grd_hidden"] = true
  data["Cook_Layer_v2.remaining_time_img.grd_hidden"] = true
  data["Cook_Layer_v2.recipe_img.image"] = "images/home-leeg-embleem_1x120-y251.png"
  data["Cook_Layer_v2.glow_img.grd_hidden"] = true
  data["Cook_Layer_v2.recipe_steps_table.grd_hidden"] = true
  data["Cook_Layer_v2.steps_group.grd_hidden"] = true
  data["Cook_Layer_v2.recipe_img.grd_hidden"] = false
  data["Cook_Layer_v2.finish_img.grd_hidden"] = true
  data["Cook_Layer_v2.warmup_img.grd_hidden"] = true
  data["Cook_Layer_v2.delay_recipe_start.grd_hidden"] = true
  gre.set_data(data)

  Control_SetButtons(true, true, true, true, true, true, true)

  CBCOOK_OnScreenShown()
end

--
-- swap between time remaining and finishes at
--
function CBCOOK_SwapRemainingArrival(mapargs)
  local data = {}

  if not mapargs.context_event_data.force_remaining and gre.get_value("Cook_Layer_v2.remaining_time_img.grd_hidden") == 0 then
    -- remaining time is visible, swap to finishes at time and start timer to swap back
    data["Cook_Layer_v2.remaining_time_img.grd_hidden"] = true
    data["Cook_Layer_v2.finish_at_img.grd_hidden"] = false
    data["Cook_Layer_v2.time_text.grd_hidden"] = true
    data["Cook_Layer_v2.finish_time_text.grd_hidden"] = false

    if gSwapTimer == nil then
      gSwapTimer = gre.timer_set_timeout(function()
        gSwapTimer = nil
        CBCOOK_SwapRemainingArrival({ context_event_data = { force_remaining = true } })
      end, 5000)
    end
  else
    -- finishes at time is visible, swap to remaining time
    data["Cook_Layer_v2.finish_at_img.grd_hidden"] = true
    data["Cook_Layer_v2.remaining_time_img.grd_hidden"] = false
    data["Cook_Layer_v2.finish_time_text.grd_hidden"] = true
    data["Cook_Layer_v2.time_text.grd_hidden"] = false
  end

  gre.set_data(data)
end

--
-- recipe canceled, goto recipe selection screen
--
function CBCOOK_OnRecipeCanceled()
  gre.set_value("toScreen", "REC_Screen")
  gre.send_event("switch_screen")
end

--
-- recipe state changed to idle, remove it from screen
--
function CBCOOK_OnRecipeIdle()
  CBCOOK_OnScreenShow()
end

--
-- pause button pressed
--
function CBCOOK_OnPauseButtonPressed()
  ActiveRecipe:pause()
end

--
-- press event on hours, minutes or seconds
-- make the respective control active for time scrolling
--
function CBAddTime_Press(mapargs)
  gSlidingY = tonumber(mapargs.context_event_data.y)
  gSlidingMinutes = false

  local minutes = tonumber(gre.get_value("AddTime_Layer.time_text.minutes"))
  local data = {}
  data["AddTime_Layer.time_text.minutes_less"] = ""
  data["AddTime_Layer.time_text.minutes_more"] = ""

  if mapargs.context_control == "AddTime_Layer.slide_control_mins" then
    gSlidingMinutes = true
    data["AddTime_Layer.time_text.minutes_less"] = minutes >= 1 and string.format("%02d", minutes - 1) or "30"
    data["AddTime_Layer.time_text.minutes_more"] = minutes <= 29 and string.format("%02d", minutes + 1) or "00"
  end

  gre.set_data(data)
end

--
-- release event on hours, minutes or seconds
-- depending on the y-coordinate compared to the y-coordinate from the press event, scroll up or down
--
function CBAddTime_Release(mapargs)
  local y = tonumber(mapargs.context_event_data.y)
  local minutes = tonumber(gre.get_value("AddTime_Layer.time_text.minutes"))

  local data = {}
  data["AddTime_Layer.time_text.minutes_less"] = ""
  data["AddTime_Layer.time_text.minutes_more"] = ""

  if mapargs.context_control == "AddTime_Layer.slide_control_mins" and gSlidingMinutes then
    if y >= (gSlidingY + 5) then
      minutes = minutes - 1
    elseif y <= (gSlidingY - 5) then
      minutes = minutes + 1
    else
    end

    if minutes < 0 then minutes = 30 end
    if minutes > 30 then minutes = 0 end

    data["AddTime_Layer.time_text.minutes"] = string.format("%02d", minutes)
    data["AddTime_Layer.time_text.minutes_less"] = minutes >= 1 and string.format("%02d", minutes - 1) or "30"
    data["AddTime_Layer.time_text.minutes_more"] = minutes <= 29 and string.format("%02d", minutes + 1) or "00"
  end
  gre.set_data(data)
end

--
-- ok button pressed on add-time dialog, send request for extra time to backend
--
function CBAddTime_OnOkPressed()
  local minutes = tonumber(gre.get_value("AddTime_Layer.time_text.minutes"))
  ActiveRecipe:add_time((minutes * 60))
end

--
-- language changed
--
function CBCOOK_OnLanguageChanged()
  gre.set_value("Idle_Layer.language_button.image", i18n:meta_data(i18n:language()).icon or image_path("vlag-engels x425-y59.png"))

  FirstUseWizard_OnExitLanguageSelection()
end

--
-- new language selection cancelled
--
function CBCOOK_OnLanguageNotChanged()
  FirstUseWizard_OnExitLanguageSelection()
end

--
-- time changed, update clock on idle screen
--
function CBCOOK_OnUpdateTime(mapargs)
  local timestamp, ampm = format_time()

  local data = {}
  data["Idle_Layer.clock_text.text"] = timestamp
  data["Idle_Layer.ampm_text.text"] = ampm or ""
  gre.set_data(data)
end

--
-- handle "heatup_done" event
--  show the heat up time + idle screen
--
-- @param gre#context mapargs
function CBCOOK_OnHeatUp_Done(mapargs)
  local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
  local varname =  i18n:get(mapargs.context_event_data.varname)
  dlg:set_message(varname .. " " .. tostring(mapargs.context_event_data.heatuptime) .. "s")
  dlg:add_button("OK", function() ActiveRecipe:set_idle() end)
  dlg:show()
end
