----------------------------
-- Dialog box implementation
----------------------------

require("events")
require("Config")
require("language")
require("Util")

--
-- Usage:
--
-- 1) Create a new dialogbox instance
-- 2) Set the message text
-- 3) Add buttons (as many as you want)
-- 4) Call the show function (if a dialogbox is currently visible, the new
--    dialogbox will be added to a queue and shown when it's its turn)
--
-- Example:
--
-- local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
-- dlg:set_message("Are you sure you want to do stuff?")
-- dlg:add_button("Yes", function() print("Yes clicked") end)
-- dlg:add_button("No" , function() print("No clicked" ) end)
-- dlg:show()
--

--[[ Dialog box implementation ]]--

-- dialog types
DIALOG_TYPE_ERROR        = 1 -- dialog box is for displaying an error
DIALOG_TYPE_WARNING      = 2 -- dialog box is for displaying a warning
DIALOG_TYPE_INFORMATION  = 3 -- dialog box is for displaying information
DIALOG_TYPE_CONFIRMATION = 4 -- dialog box is for asking for confirmation
DIALOG_TYPE_AUTOCLOSE    = 5 -- dialog box without buttons
DIALOG_TYPE_RESET        = 6 -- dialog box is for asking for reset action on gas burner

-- mapping of dialog types to icon and title
local gDialogTypeMap = {
  [DIALOG_TYPE_ERROR       ] = { icon = "icon-!.png", title = "DIALOG_TYPE_ERROR_TITLE"       , type = DIALOG_TYPE_ERROR        },
  [DIALOG_TYPE_WARNING     ] = { icon = "icon-!.png", title = "DIALOG_TYPE_WARNING_TITLE"     , type = DIALOG_TYPE_WARNING      },
  [DIALOG_TYPE_INFORMATION ] = { icon = "icon-!.png", title = "DIALOG_TYPE_INFORMATION_TITLE" , type = DIALOG_TYPE_INFORMATION  },
  [DIALOG_TYPE_CONFIRMATION] = { icon = "icon-!.png", title = "DIALOG_TYPE_CONFIRMATION_TITLE", type = DIALOG_TYPE_CONFIRMATION },
  [DIALOG_TYPE_AUTOCLOSE   ] = { icon = "icon-!.png", title = "DIALOG_TYPE_AUTOCLOSE"         , type = DIALOG_TYPE_AUTOCLOSE    },
  [DIALOG_TYPE_RESET       ] = { icon = "icon-!.png", title = "DIALOG_TYPE_INFORMATION_TITLE" , type = DIALOG_TYPE_INFORMATION  }
}

-- dialog box class
DialogBox = {}
DialogBox.__index = DialogBox

DialogBox.pending_dialogs = {} -- list of pending dialog boxes
DialogBox.active_dialog = nil  -- currently active dialog box
DialogBox.suspended = false    -- dialog boxes suspended
DialogBox.button_id = 1        -- button id, increments across dialogs to stay unique

--
-- create a new dialog box instance
-- static method, call this on DialogBox directly
--
function DialogBox.new(dialog_type, id)
  dialog_type = dialog_type or DIALOG_TYPE_ERROR
  assert(dialog_type > 0 and dialog_type <= #gDialogTypeMap)
  assert(dialog_type ~= DIALOG_TYPE_AUTOCLOSE or id ~= nil)
  local o = { type = gDialogTypeMap[dialog_type], id = id or -1, message = nil, image = nil, buttons = {} }
  setmetatable(o, DialogBox)
  return o
end

--
-- can a new dialog box be shows?
-- static method, call this on DialogBox directly
--
function DialogBox:can_show(dlg)
  if not self.active_dialog and not self.suspended then
    self.active_dialog = dlg
    return true
  else
    table.insert(self.pending_dialogs, dlg)
    return false
  end
end

--
-- show the next dialog box in the pending queue
-- static method, call this on DialogBox directly
--
function DialogBox:show_next()
  self.active_dialog = nil

  local next_dlg = table.remove(self.pending_dialogs, 1)
  if not next_dlg or not next_dlg:show() then
    gre.set_value("Dialog_Layer.DimFilter_Img.grd_hidden", true)
  end
end

--
-- check if a dialog matches the specified id
-- note that ids are only supported for auto closing dialogs
--
local function dialog_matches(dlg, id)
  return dlg ~= nil and dlg.type.type == DIALOG_TYPE_AUTOCLOSE and dlg.id == id
end

--
-- close an auto-closing dialog box
-- static method, call this on DialogBox directly
--
function DialogBox:remove(id)
  if dialog_matches(self.active_dialog, id) then
    self.active_dialog:hide()
  else
    for i,dlg in ipairs(self.pending_dialogs) do
      if dialog_matches(dlg, id) then
        table.remove(self.pending_dialogs, i)
        break
      end
    end
  end
end

--
-- set the dialog box message
--
function DialogBox:set_message(msg)
  self.message = msg
end

--
-- set the dialog box image
--
function DialogBox:set_image(img)
  self.image = img
end

--
-- add a button with optional callback
--
function DialogBox:add_button(name, callback)
  table.insert(self.buttons, { name = name, callback = callback })
end

--
-- add multiple buttons at once in a single call
--
function DialogBox:add_buttons(buttons)
  for _, button in ipairs(buttons) do
    self:add_button(button.name, button.callback)
  end
end

--
-- show the dialogbox
--
function DialogBox:show()
  if not DialogBox:can_show(self) then return false end

  local data = {}

  data["Dialog_Layer.Dialog_Group.Icon_Img.image"] = image_path(self.type.icon)
  data["Dialog_Layer.Dialog_Group.Title_Text.text"] = i18n:get(self.type.title)
  data["Dialog_Layer.Dialog_Group.Message_Text.text"] = self.message
  data["Dialog_Layer.Dialog_Group.Btn_1.grd_hidden"] = #self.buttons == 0
  -- show image?
  if self.image ~= nil then
    data["Dialog_Layer.Dialog_Group.Message_Img.image"] = self.image
    data["Dialog_Layer.Dialog_Group.Message_Img.grd_hidden"] = false
  else
    data["Dialog_Layer.Dialog_Group.Message_Img.grd_hidden"] = true
  end

  if #self.buttons > 0 then
    local dialog_padding = 10
    local dialog_width = gre.get_value("Dialog_Layer.Dialog_Group.Background_Img.grd_width") - (2 * dialog_padding)
    local font = gre.get_value("Dialog_Layer.Dialog_Group.Btn_1.fontName")
    local font_size = gre.get_value("Dialog_Layer.Dialog_Group.Btn_1.fontSize")
    local btn_padding = gre.get_value("Dialog_Layer.Dialog_Group.Btn_1.padding")

    local btn_width = gre.get_string_size(font, font_size, self.buttons[1].name).width + btn_padding
    for i = 2, #self.buttons do
      gre.clone_object("Dialog_Layer.Dialog_Group.Btn_1", string.format("Btn_Extra_%d", DialogBox.button_id + i), "Dialog_Layer.Dialog_Group")
      btn_width = math.max(btn_width, gre.get_string_size(font, font_size, self.buttons[i].name).width + btn_padding)
    end

    local inter_btn_width = (dialog_width - (btn_width * #self.buttons)) / (#self.buttons + 1)
    for i = 1, #self.buttons do
      local button_id =  i == 1 and 1 or DialogBox.button_id + i
      local btn_ctrl = i == 1 and "Dialog_Layer.Dialog_Group.Btn_1" or string.format("Dialog_Layer.Dialog_Group.Btn_Extra_%d", button_id)
      data[btn_ctrl .. ".color"] = 0xB1D5CF
      data[btn_ctrl .. ".image"] = image_path("dialog-knop-op.png")
      data[btn_ctrl .. ".text"] = self.buttons[i].name
      data[btn_ctrl .. ".id"] = i
      data[btn_ctrl .. ".grd_x"] = dialog_padding + (i * inter_btn_width) + ((i-1) * btn_width)
      data[btn_ctrl .. ".grd_width"] = btn_width
      data[btn_ctrl .. ".grd_hidden"] = false
    end
  end

  -- display dialog (animation is not working properly)
  data["Dialog_Layer.DimFilter_Img.grd_hidden"] = false
  data["Dialog_Layer.Dialog_Group.grd_x"] = 9

  gre.set_data(data)

  --gre.animation_trigger("display_dialog")

  return true
end

--
-- hide the dialogbox
--
function DialogBox:hide()
  --gre.animation_trigger("hide_dialog")

  for i = 2, #self.buttons do
    gre.delete_object(string.format("Dialog_Layer.Dialog_Group.Btn_Extra_%d", DialogBox.button_id + i))
  end
  DialogBox.button_id = DialogBox.button_id + #self.buttons - 1

  -- hide dialog and display next (animation is not working properly)
  local data = {}
  data["Dialog_Layer.DimFilter_Img.grd_hidden"] = true
  data["Dialog_Layer.Dialog_Group.grd_x"] = -500
  gre.set_data(data)

  DialogBox:show_next()
end

--
-- button pressed
--
function DialogBox:on_button_pressed(id)
  local button = self.buttons[id]
  if button and button.callback then button.callback() end
  self:hide()
end

--
-- suspend all dialogs from being shown and put active dialogs back in the queue
-- static method, call this on DialogBox directly
--
function DialogBox:suspend(suspend_dialogs)
  if suspend_dialogs ~= self.suspended then
    self.suspended = suspend_dialogs
    if self.suspended then
      if self.active_dialog ~= nil then
        if #self.active_dialog.buttons > 0 then
          table.insert(self.pending_dialogs, 1, self.active_dialog)
        end
        self.active_dialog:hide()
      end
    else
      self:show_next()
    end
  end
end

--
-- global dialog button pressed handler
-- static method, call this on DialogBox directly
--
function DialogBox:on_dlg_btn_pressed(id)
  if self.active_dialog then self.active_dialog:on_button_pressed(id) end
end

--
-- return image to show assocated with given nls_id
--
-- todo :  lookup in a table
function DialogBox:get_image(nls_id)
  if nls_id == nil then
    return nil
  elseif nls_id == "NLS_PLACE_CAPS" then
    return image_path("icon-clean-sachet.png")
  elseif nls_id == "NLS_PLACE_DETERGENT" or nls_id == "NLS_REMOVE_DETERGENT" then
    return image_path("icon-clean-cartridge.png")
  elseif nls_id == "NLS_PLACE_DESCALE" then
    return image_path("icon-clean-tablet.png")
  else
    return nil
  end
end

--[[ Relevant events ]]--

--
-- application wide gre.screenshow.post event handler
-- enable dialog iff the active screen supports them
--
--- @param gre#context mapargs
function CBDLG_OnScreenShow(mapargs)
  DialogBox:suspend(gre.get_layer_attrs(mapargs.context_screen .. ".Dialog_Layer", "y") == nil)
end

--
-- application wide gre.screenhide.pre event handler
--
--- @param gre#context mapargs
function CBDLG_OnScreenHide(mapargs)
  DialogBox:suspend(true)
end

--
-- dialog button pressed
--
--- @param gre#context mapargs
function CBDLG_OnBtnPressed(mapargs)
  DialogBox:on_dlg_btn_pressed(gre.get_value(mapargs.context_control..".id"))
end

--
-- dialog hide animation completed
--
function CBDLG_OnHideComplete()
  DialogBox:show_next()
end

--
-- error event received from backend
--
--- @param gre#context mapargs
function CB_OnError(mapargs)
  local error_code = mapargs.context_event_data.code

  local dlg = DialogBox.new(DIALOG_TYPE_ERROR)
  dlg:set_message(i18n:get("ERROR_" .. error_code))
  dlg:add_button(i18n:get("NLS_OK"))
  dlg:show()
end

--
-- dialog event with free text received from backend
--
--- @param gre#context mapargs
function CB_OnDialog(mapargs)
  local type = mapargs.context_event_data.type
  local id = mapargs.context_event_data.id
  local nls_id = mapargs.context_event_data.text

  local dlg = DialogBox.new(type, id)
  dlg:set_message(i18n:get(nls_id))
  dlg:set_image(dlg:get_image(nls_id))

  if type == DIALOG_TYPE_CONFIRMATION then
    dlg:add_button(i18n:get("NLS_CANCEL"), function() Event:action("CANCEL_ACTION") end)
    dlg:add_button(i18n:get("NLS_CONTINUE"), function() Event:action("CONTINUE_ACTION") end)
  else
    if type == DIALOG_TYPE_RESET then
      dlg:add_button(i18n:get("NLS_RESET"), function() Event:reset_burner() end)
    else
      --if type ~= DIALOG_TYPE_AUTOCLOSE then
      dlg:add_button(i18n:get("NLS_OK")) -- FID3905 allow DIALOG_TYPE_AUTOCLOSE also be closed with OK button press.
      --end
    end
  end

  dlg:show()
end

--
-- dialog event to close an auto-closing dialog received fromb backend
--
--- @param gre#context mapargs
function CB_OnCloseDialog(mapargs)
  DialogBox:remove(mapargs.context_event_data.id)
end

--
-- newsoftware event received from backend
--
--- @param gre#context mapargs
function CB_OnNewSoftware(mapargs)
  local error_code = mapargs.context_event_data.code

  local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
  dlg:set_message(i18n:get("NLS_NEWSOFTWARE"))
  dlg:add_button(i18n:get("NLS_CANCEL"))
  dlg:add_button(i18n:get("NLS_RESTART"), function() Event:action("RESTART_HARD_MANAGER") end)
  dlg:show()
end

