--------------------------------------------
-- simulate events being sent by the backend
--------------------------------------------

require("Config")
require("Util")

local OUTPUT_CHANNEL = "input"

local evt_startupdelayreply  = { name = "io_startupdelay"        , format = "2u1 delay"                                              }
local evt_listdirreply       = { name = "io_listdir_reply"       , format = "1s0 files"                                              } -- with files = list of files, first entry is the directory
local evt_recipelist         = { name = "io_recipelist"          , format = "2u1 count"                                              }
local evt_recipe             = { name = "io_recipe"              , format = "1u1 sort_index 1u1 calibrated 1u1 steps 1s0 data"       } -- with data = name+icon
local evt_recipestep         = { name = "io_recipe_step"         , format = "2u1 temp 2u1 time 1u%d params 1s0 name"                 }
local evt_recipestate        = { name = "io_recipe_state"        , format = "1u1 state 1u1 step"                                     }
local evt_tempupdate         = { name = "io_temperature"         , format = "2u1 temp"                                               }
local evt_wifirequest        = { name = "io_wifi_request"        , format = "1s0 device"                                             }
local evt_wifilistreply      = { name = "io_wifi_list_reply"     , format = "1u1 ssid_count"                                         } -- with ssid_count = number of ssids discovered
local evt_wifilistreply_ssid = { name = "io_wifi_list_reply_ssid", format = "1u1 connected 1s0 ssid"                                 }
local evt_wifidatastored     = { name = "io_wifi_data_stored"                                                                        }
local evt_parameterreply     = { name = "io_parameter_reply"     , format = "2u1 last 2u%d values 1s0 ids"                           } -- with ids = list of parameter names
local evt_pauserecipe        = { name = "io_recipe_pause"                                                                            }
local evt_resumerecipe       = { name = "io_recipe_resume"                                                                           }
local evt_cancelrecipe       = { name = "io_recipe_cancel"                                                                           }
local evt_error              = { name = "io_error"               , format = "2u1 device 2u1 code"                                    }
local evt_standby            = { name = "io_standby"                                                                                 }
local evt_proximityalert     = { name = "io_proximity_alert"                                                                         }
local evt_autocorrect        = { name = "io_autocorrect"         , format = "2s1 delta"                                              }
local evt_msgreceived        = { name = "io_msg_received"        , format = "2u1 id 2u1 type 1s0 message"                            }
local evt_logentries         = { name = "io_logentries"          , format = "2s1 log"                                                } -- with log = list of log entries
local evt_elevationgranted   = { name = "io_elevation_granted"   , format = "1s0 role"                                               }
local evt_elevationdenied    = { name = "io_elevation_denied"                                                                        }
local evt_wifirequest        = { name = "io_wifi_request"        , format = "1s0 device"                                             }
local evt_errorlogreply      = { name = "io_errorlog_reply"      , format = "4u%d start 4u%d stop 1u%d code 1u%d active 1u%d device" } -- with %d the number of log entries
local evt_paramoptionsreply  = { name = "io_parameter_options"   , format = "1s0 options"                                            } -- with options = list of options, first option is parameter
local evt_live_value         = { name = "io_live_value"          , format = "4u1 value 1s0 id"                                       }
local evt_clean_done         = { name = "io_clean_done"                                                                              }
local evt_clean_info         = { name = "io_clean_info"          , format = "4u1 totaltime"                                          }
local evt_clean_passed       = { name = "io_clean_passed"        , format = "4u1 totalpassed"                                        }
local evt_clean_step         = { name = "io_clean_step"          , format = "1u1 stepno 1s0 stepdesc"                                }
local evt_clean_resume       = { name = "io_clean_resume"        , format = "4u1 timestamp 1s0 name"                                 }
local evt_msglist_reply      = { name = "io_msglist_reply"       , format = "1u%d ids 1s0 dir"                                       }

local gTimerId = nil     -- timer that ticks every second
local gSecondsPassed = 0 -- seconds passed so far

--
-- send an event without payload
--
local function send_event(event)
  gre.send_event(event.name)
end

--
-- send an event with payload
--
local function send_event_data(event, data)
  gre.send_event_data(event.name, event.format, data)
end

--
-- send a list of icons
--
local function send_icons()
  local icons = icons_dir()
  icons = icons .. DELIMITER_TOKEN .. "beef.png"
  icons = icons .. DELIMITER_TOKEN .. "beef_large.png"
  icons = icons .. DELIMITER_TOKEN .. "butterfly-chicken.png"
  icons = icons .. DELIMITER_TOKEN .. "butterfly-chicken_large.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken_large.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken-thighs.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken-thighs_large.png"
  icons = icons .. DELIMITER_TOKEN .. "chops.png"
  icons = icons .. DELIMITER_TOKEN .. "chops_large.png"
  icons = icons .. DELIMITER_TOKEN .. "fish.png"
  icons = icons .. DELIMITER_TOKEN .. "fish_large.png"
  icons = icons .. DELIMITER_TOKEN .. "fried-parts.png"
  icons = icons .. DELIMITER_TOKEN .. "fried-parts_large.png"
  icons = icons .. DELIMITER_TOKEN .. "fried-wings.png"
  icons = icons .. DELIMITER_TOKEN .. "fried-wings_large.png"
  icons = icons .. DELIMITER_TOKEN .. "grilled-sausages.png"
  icons = icons .. DELIMITER_TOKEN .. "grilled-sausages_large.png"
  icons = icons .. DELIMITER_TOKEN .. "grilled-veggies.png"
  icons = icons .. DELIMITER_TOKEN .. "grilled-veggies_large.png"
  icons = icons .. DELIMITER_TOKEN .. "legs.png"
  icons = icons .. DELIMITER_TOKEN .. "legs_large.png"
  icons = icons .. DELIMITER_TOKEN .. "meatballs.png"
  icons = icons .. DELIMITER_TOKEN .. "meatballs_large.png"
  icons = icons .. DELIMITER_TOKEN .. "meatloaf.png"
  icons = icons .. DELIMITER_TOKEN .. "meatloaf_large.png"
  icons = icons .. DELIMITER_TOKEN .. "organic-chicken.png"
  icons = icons .. DELIMITER_TOKEN .. "organic-chicken_large.png"
  icons = icons .. DELIMITER_TOKEN .. "pork-belly.png"
  icons = icons .. DELIMITER_TOKEN .. "pork-belly_large.png"
  icons = icons .. DELIMITER_TOKEN .. "potatoes.png"
  icons = icons .. DELIMITER_TOKEN .. "potatoes_large.png"
  icons = icons .. DELIMITER_TOKEN .. "potatoe-wedges.png"
  icons = icons .. DELIMITER_TOKEN .. "potatoe-wedges_large.png"
  icons = icons .. DELIMITER_TOKEN .. "roast-beef.png"
  icons = icons .. DELIMITER_TOKEN .. "roast-beef_large.png"
  icons = icons .. DELIMITER_TOKEN .. "roast-loin.png"
  icons = icons .. DELIMITER_TOKEN .. "roast-loin_large.png"
  icons = icons .. DELIMITER_TOKEN .. "salmon-fillet.png"
  icons = icons .. DELIMITER_TOKEN .. "salmon-fillet_large.png"
  icons = icons .. DELIMITER_TOKEN .. "salmon-slice.png"
  icons = icons .. DELIMITER_TOKEN .. "salmon-slice_large.png"
  icons = icons .. DELIMITER_TOKEN .. "sausages.png"
  icons = icons .. DELIMITER_TOKEN .. "sausages_large.png"
  icons = icons .. DELIMITER_TOKEN .. "spare-ribs.png"
  icons = icons .. DELIMITER_TOKEN .. "spare-ribs_large.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken-XL.png"
  icons = icons .. DELIMITER_TOKEN .. "chicken-XL_large.png"
  send_event_data(evt_listdirreply, {files = icons})
end

-- list of configuration parameters and their values
local gParameters = {
  ["TEMPERATURE_UNIT"   ] = 1,
  ["VOLUME_UNIT"        ] = 1,
  ["TIMEZONE_OFFSET"    ] = 2,
  ["TIME_FORMAT"        ] = 1,
  ["DATE_FORMAT"        ] = 0,
  ["PREHEAT_MODE"       ] = 0,
  ["PREHEAT_DELTA"      ] = 0,
  ["AUTODOOR_MODE"      ] = 0,
  ["OPEN_FOR_COOLING"   ] = 0,
  ["AUTO_RECIPE_START"  ] = 0,
  ["BUZZER_SET"         ] = 0,
  ["SW_VERSION"         ] = 1,
  ["MANAGER_PIN"        ] = 1111,
  ["SAVE_RECIPE"        ] = 0,
  ["LOAD_RECIPE"        ] = 0,
  ["SET_LIGHT"          ] = 0,
  ["SET_TIME"           ] = 0,
  ["SET_DATE"           ] = 0,
  ["BUZZER_TEST"        ] = 0,
  ["SAVE_PARAM"         ] = 0,
  ["LOAD_PARAM"         ] = 0,
  ["LANGUAGE"           ] = 0,
  ["CLEAN_QUICK_ENABLED"] = 1,
  ["CLEAN_DAILY_ENABLED"] = 1,
  ["CLEAN_DEEP_ENABLED" ] = 1,
  ["ASK_WEIGHT"         ] = 0,
  ["ENDUSER_EDIT"       ] = 1,
  ["CONNECT_DIRECT"     ] = 1,
  ["CONNECT_SMART"      ] = 0,
  ["RECIPE_STRLEN_MAX"  ] = 30,
  ["CLEANSCREEN_TIME"   ] = 10,
  ["DEVICE_TYPE"        ] = 8,
  ["LIMEFILTER_TIME"    ] = 1574148988,
  ["COMMISSION_TIME"    ] = 0,
  ["WIFI_MODE"          ] = 1
}

--
-- send an event with the configuration parameters and their values
--
local function set_parameters()
  local event = { name = evt_parameterreply.name, format = (string.format(evt_parameterreply.format, table_count(gParameters))) }
  local values = {}
  local ids = ""
  for id, value in pairs(gParameters) do
    ids = ids .. DELIMITER_TOKEN .. id
    table.insert(values, value)
  end
  send_event_data(event, { last=1, values = values, ids = string.sub(ids, 2, -1) })
end

--
-- live values to send to the GUI
--
local gLiveValues = {
  ["CLEANING_SUPPORTED" ] = 1,
  ["MSG_COUNT"] = 3
}

--
-- send live values to the GUI
--
local function set_live_values()
  for id, value in pairs(gLiveValues) do
    send_event_data(evt_live_value, { value = value, id = id })
  end
end

-- list of recipes
local gRecipes = {
  ['Chicken large'] = {
    sort_index = 3,
    calibrated = true,
    icon = 'chicken.png',
    steps = {
      { name = 'NLS_STEPNAME_PREHEAT', temp = 180, preheat = true, time = 0, custom = false },
      { name = 'Schroeien', temp = 250, time = 12, custom = true },
      { name = 'NLS_STEPNAME_GRILLING', temp = 210, time = 40, custom = false },
      { name = 'NLS_STEPNAME_COOKING', temp = 250, time = 20, custom = false },
      { name = 'Blussen', temp = 1, time = 5, custom = true },
      { name = 'NLS_STEPNAME_HOLDING', temp = 120, hold = true, time = 0, custom = false }
    }
  },
  ['Kip Mexico'] = {
    sort_index = 2,
    calibrated = true,
    icon = 'butterfly-chicken.png',
    steps = {
      { name = 'Preheating', temp = 150, preheat = true, time = 0 },
      { name = 'Step 2', temp = 200, time = 30 },
      { name = 'Step 3', temp = 120, time = 10 }
    }
  },
  ['Chicken wings'] = {
    sort_index = 5,
    calibrated = false,
    icon = 'fried-wings.png',
    steps = {
      { name = 'Preheating', temp = 175, preheat = true, time = 0 },
      { name = 'Step 2', temp = 200, time = 10 }
    }
  },
  ['Chicken legs'] = {
    sort_index = 1,
    calibrated = true,
    icon = 'legs.png',
    steps = {
      { name = 'Preheating', temp = 180, preheat = true, time = 0 },
      { name = 'Step 2', temp = 200, time = 10 }
    }
  },
  ['Chicken parts'] = {
    sort_index = 4,
    calibrated = false,
    icon = 'fried-parts.png',
    steps = {
      { name = 'Preheating', temp = 150, preheat = true, time = 0 },
      { name = 'Step 2', temp = 200, time = 10 }
    }
  }
}

--
-- send the events to transmit the recipes database
--
local function set_recipes()
  send_event_data(evt_recipelist, { count = table_count(gRecipes) })
  for name, recipe in pairs(gRecipes) do
    send_event_data(evt_recipe, { sort_index = recipe.sort_index,
                                  calibrated = recipe.calibrated,
                                  steps = #recipe.steps,
                                  data = string.format("%s%s%s", name, DELIMITER_TOKEN, recipe.icon) })
    for _, step in ipairs(recipe.steps) do
      local params = {}
      table.insert(params, step.preheat and 1               or 0) -- is this the preheating step? [0-1]
      table.insert(params, step.hold    and 1               or 0) -- is this the hold step at the end to keep the oven warm? [0-1]
      table.insert(params, step.probe   and 1               or 0) -- core probe enabled [0-1]
      table.insert(params, step.fan                         or 0) -- fan speed [0-2] or [0-4] or [0-5]
      table.insert(params, step.steam                       or 0) -- steam [0-3] off or *, **, ***
      table.insert(params, step.vent                        or 0) -- vent/exhaust valve: 0 (closed), 1 (open), 5 (open for last 5 minutes)
      table.insert(params, step.drying  and step.drying * 2 or 0) -- drying: 0-9.5 minutes in steps of 0.5, so [0-19]
      table.insert(params, step.custom  and 1               or 0) -- 0 = default step name, 1 = custom step name
      table.insert(params, step.notify  and 1               or 0) -- is this a notification step?
      gre.send_event_data(evt_recipestep.name, string.format(evt_recipestep.format, table_count(params)), {
        temp = step.temp, time = step.time or 0, params = params, name = step.name, custom = step.custom
      })
    end
  end
end

--
-- send a list of available device types
-- FID4423 add new OVEN_LDR_9S_AC_USA with number 18 for selection in menu "Service-->Device Type"
--
local function list_device_types()
  local device_types = "DEVICE_TYPE"
  device_types = device_types .. DELIMITER_TOKEN .. "7"
  device_types = device_types .. DELIMITER_TOKEN .. "8"
  device_types = device_types .. DELIMITER_TOKEN .. "9"
  device_types = device_types .. DELIMITER_TOKEN .. "10"
  device_types = device_types .. DELIMITER_TOKEN .. "11"
  device_types = device_types .. DELIMITER_TOKEN .. "12"
  device_types = device_types .. DELIMITER_TOKEN .. "13"
  device_types = device_types .. DELIMITER_TOKEN .. "14"
  device_types = device_types .. DELIMITER_TOKEN .. "15"
  device_types = device_types .. DELIMITER_TOKEN .. "16"
  device_types = device_types .. DELIMITER_TOKEN .. "17"
  --device_types = device_types .. DELIMITER_TOKEN .. "18" --TODO:FID4423 add OVEN_LDR_9S_AC_USA
  send_event_data(evt_paramoptionsreply, { options = device_types })
end

--
-- send the events for the startup delay, configuration parameters, and recipes database
--
local function set_data()
  -- enable send_icons when running locally, disable when running on device
  list_device_types()
  send_icons()
  send_event_data(evt_startupdelayreply, { delay = 500 })
  set_parameters()
  set_live_values()
  set_recipes()
end

--
-- send an event with the current actual oven temperature
--
local function temperature_update()
  send_event_data(evt_tempupdate, { temp = math.random(0,250) })
end

--
-- send a bunch of error events
--
local function spam_errors()
  for i = 1, 15 do
    send_event_data(evt_error, { device = 1, code = i })
  end
end

-- storyline of this event generator, the keys are the seconds at which to execute the value function
local gEventGeneratorTable = {
  [ 2] = set_data,
--[ 5] = function() send_event_data(evt_elevationgranted, { role = "SERVICE" }) end,
--[10] = function() send_event_data(evt_wifirequest, { device = "John Doe" }) end,
--[10] = spam_errors,
}

--
-- set the default value of a table's entries
--
local function setDefault(t, d)
  local mt = { __index = function() return d end }
  setmetatable(t, mt)
end

--
-- timer action function, called each second, calls the function in the storyline table for that tick
--
local function onTimer()
  gSecondsPassed = gSecondsPassed + 1
  gEventGeneratorTable[gSecondsPassed]()
  --temperature_update()
end

--
-- start the timer to begin generating events
--
local function start()
  math.randomseed(epoch_time())
  setDefault(gEventGeneratorTable, function() end)
  gSecondsPassed = 0
  gTimerId = gre.timer_set_interval(onTimer, 1000)
end

--
-- stop the timer to stop generating events
--
local function stop()
  if gTimerId ~= nil then
    gre.timer_clear_interval(gTimerId)
    gTimerId = nil
  end
end

local gRecipeTimer = nil

--
-- have some fun with a demo recipe
--
local function recipe()
  local events = {
    [ 1] = function() send_event_data(evt_recipestate , { state =  2, step = 1 }) end,
    [ 2] = function() send_event_data(evt_recipestate , { state =  3, step = 2 }) end,
    [ 7] = function() send_event_data(evt_autocorrect , { delta =  5           }) end,
    [19] = function() send_event_data(evt_recipestate , { state =  3, step = 3 }) end,
    [25] = function() send_event     (evt_pauserecipe                           ) end,
    [27] = function() send_event     (evt_resumerecipe                          ) end,
    [61] = function() send_event_data(evt_recipestate , { state =  3, step = 4 }) end,
    [70] = function() send_event_data(evt_autocorrect , { delta = -2           }) end,
    [79] = function() send_event_data(evt_recipestate , { state =  3, step = 5 }) end,
    [84] = function() send_event_data(evt_recipestate , { state =  5, step = 1 }) end
  }

  local seconds_passed = 0
  local event_counter = 0
  local timer = nil
  gRecipeTimer = gre.timer_set_interval(function()
    seconds_passed = seconds_passed + 1
    local event = events[seconds_passed]
    if event then
      event_counter = event_counter + 1
      event()
    end
    if event_counter == table_count(events) then
      gre.timer_clear_interval(gRecipeTimer)
      gRecipeTimer = nil
    end
  end, 1000)
end

--
-- cancel recipe simulation
--
local function cancel_recipe()
  if gRecipeTimer then
    gre.timer_clear_interval(gRecipeTimer)
    gRecipeTimer = nil
  end
end

local gCleanTimer = nil

--
-- demo cleaning program
--
local function clean()
  local events = {
    [ 1] = function() send_event_data(evt_clean_info  , { totaltime = 100                          }) end,
    [ 2] = function() send_event_data(evt_clean_step  , { stepno = 0, stepdesc = "Soap Cooling"    }) end,
    [12] = function() send_event_data(evt_clean_step  , { stepno = 1, stepdesc = "Place Detergent" }) end,
    [22] = function() send_event_data(evt_clean_step  , { stepno = 2, stepdesc = "Wait"            }) end,
    [32] = function() send_event_data(evt_clean_step  , { stepno = 3, stepdesc = "Cooling"         }) end,
    [42] = function() send_event_data(evt_clean_step  , { stepno = 4, stepdesc = "Cleaning"        }) end,
    [43] = function() send_event_data(evt_clean_passed, { totalpassed = 1                          }) end,
    [44] = function() send_event_data(evt_clean_passed, { totalpassed = 10                         }) end,
    [45] = function() send_event_data(evt_clean_passed, { totalpassed = 20                         }) end,
    [46] = function() send_event_data(evt_clean_passed, { totalpassed = 30                         }) end,
    [47] = function() send_event_data(evt_clean_passed, { totalpassed = 40                         }) end,
    [48] = function() send_event_data(evt_clean_passed, { totalpassed = 50                         }) end,
    [49] = function() send_event_data(evt_clean_passed, { totalpassed = 60                         }) end,
    [50] = function() send_event_data(evt_clean_passed, { totalpassed = 70                         }) end,
    [51] = function() send_event_data(evt_clean_passed, { totalpassed = 80                         }) end,
    [52] = function() send_event_data(evt_clean_passed, { totalpassed = 90                         }) end,
    [53] = function() send_event_data(evt_clean_passed, { totalpassed = 100                        }) end,
    [54] = function() send_event     (evt_clean_done                                                ) end
  }

  local seconds_passed = 0
  local event_counter = 0
  local timer = nil
  gCleanTimer = gre.timer_set_interval(function()
    seconds_passed = seconds_passed + 1
    local event = events[seconds_passed]
    if event then
      event_counter = event_counter + 1
      event()
    end
    if event_counter == table_count(events) then
      gre.timer_clear_interval(gCleanTimer)
      gCleanTimer = nil
    end
  end, 1000)
end

--
-- create some error log entries for testing
--
local function generate_error_log()
  local error_log = {
    code   = {  2, 16,  4, 12,  6, 10,  8, 12,  0 },
    active = {  1,  1,  1,  1,  0,  0,  0,  0,  0 },
    device = {  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    start  = {  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    stop   = {  0,  0,  0,  0,  0,  0,  0,  0,  0 }
  }
  local c = #error_log.code
  gre.send_event_data(evt_errorlogreply.name, string.format(evt_errorlogreply.format, c, c, c, c, c), error_log)
end

--
-- send a list of available sound sets
--
local function list_sound_sets()
  local sound_sets = "SOUNDSET_STR"
  sound_sets = sound_sets .. DELIMITER_TOKEN .. "Set 1"
  sound_sets = sound_sets .. DELIMITER_TOKEN .. "Set 2"
  sound_sets = sound_sets .. DELIMITER_TOKEN .. "Set 3"
  send_event_data(evt_paramoptionsreply, { options = sound_sets })
end

--
-- pretend a message is being sent
--
local function get_messages()
  gre.send_event_data(evt_msglist_reply.name,
                      string.format(evt_msglist_reply.format, 3),
                      { ids = { 1, 2, 3 }, dir = absolute_path("messages") })
end

--
-- send a list of wifi SSIDs
--
local function wifi_list(count, connected_id)
  local ssids = {}

  for i=1,count do
    table.insert(ssids, { name = string.format("SSID %d", i), connected = i == connected_id })
  end

  send_event_data(evt_wifilistreply, { ssid_count = table_count(ssids) })

  for _, ssid in ipairs(ssids) do
    send_event_data(evt_wifilistreply_ssid, { ssid = ssid.name, connected = ssid.connected and 1 or 0 })
  end
end

--
-- send io_wifi_data_stored event
--
local function wifi_data_stored()
  send_event(evt_wifidatastored)
end

-------------------------------------------------------------------------------------

return {
  start = start,
  stop = stop,
  recipe = recipe,
  cancel_recipe = cancel_recipe,
  generate_error_log = generate_error_log,
  list_sound_sets = list_sound_sets,
  clean = clean,
  get_messages = get_messages,
  wifi_list = wifi_list,
  wifi_data_stored = wifi_data_stored
}

-------------------------------------------------------------------------------------
