-----------------------------------------
-- Implementation of the first-use wizard
-----------------------------------------

require("Dialog")
require("Language_Layer")
require("Recipes")
require("Settings")
require("Utf8")
require("QrCode")

--[[ navigation ]]--

local gFirstUseWizardShown = false
local gFirstUseWizardActive = false
local gQrCode = QrCodeWidget.new{canvas="url_qrcode"} -- canvas for QR code in first use wizard.
local FirstUse_pages = {
  "FirstUseWizard_Layer.Welcome_Page",
  "FirstUseWizard_Layer.SetDate_Page",
  "FirstUseWizard_Layer.AmPm_Page",
  "FirstUseWizard_Layer.SetTime_Page",
  "FirstUseWizard_Layer.EnterLocation_Page",
  "FirstUseWizard_Layer.QrCode_Page",         --  page 6 with QR code must be skipped in Walmart ovens.
  "FirstUseWizard_Layer.AlmostReady_Page"
}
local gCurrentPageIndex = 1
local INDEX_QR_CODE_PAGE = 6
--
-- close the first use wizard
-- restore the language layer z-index to the normal position
--
local function FirstUseWizard_Close()
  for i, page in ipairs(FirstUse_pages) do
    gre.set_group_attrs(page, { hidden = true })
  end
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = true })
  gFirstUseWizardActive = false
  gCurrentPageIndex = 1
end

local function FirstUseWizard_ShowCurrentPage()
  if (gCurrentPageIndex < 1) or (gCurrentPageIndex > #FirstUse_pages) then
    print("Invalid gCurrentPageIndex: " .. tostring(gCurrentPageIndex))
    return
  end

  -- Loop through all FirstUse pages and hide them, except for the current page.
  gre.set_data(data)
  for i, page in ipairs(FirstUse_pages) do
    gre.set_group_attrs(page, { hidden = (i ~= gCurrentPageIndex) })
  end
end

function FirstUseWizard_OnNextPage()
  local walmart_oven = (Settings:get("CUSTOMER_ID") == 1)
  local next_index = gCurrentPageIndex + 1

  if walmart_oven and (next_index == INDEX_QR_CODE_PAGE) then
    next_index = next_index + 1  -- Skip QrCode_Page
  end
  if next_index <= #FirstUse_pages then
    gCurrentPageIndex = next_index
    FirstUseWizard_ShowCurrentPage()
  end
end

function FirstUseWizard_OnPrevPage()
  local walmart_oven = (Settings:get("CUSTOMER_ID") == 1)
  local prev_index = gCurrentPageIndex - 1

  if walmart_oven and (prev_index == INDEX_QR_CODE_PAGE) then
    prev_index = prev_index - 1  -- Skip QrCode_Page
  end
  if prev_index >= 1 then
    gCurrentPageIndex = prev_index
    FirstUseWizard_ShowCurrentPage()
  end
end

--
-- first use wizard was completed, last step was to start a first use recipe (removed in FID4439)
--
function FirstUseWizard_OnFinish()
  FirstUseWizard_Close()
  -- FID4439: end first use wizard here, don't start the first use recipe anymore.
  -- show a dialog to inform user that first use wizard has been completed.
  local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
  dlg:set_message(i18n:get("NLS_FIRST_USE_WIZARD_COMPLETED"))
  dlg:add_button(i18n:get("NLS_OK"))
  dlg:show()
  Event:commissioning_done() -- notify backend that commissioning was completed.
end

--
-- save selected time format, and choose to display AM/PM entry field.
--
function CBCOOK_FirstUseWizard_OnSaveTimeFormat()
  local data = {}
  local hour_value = Settings:get("SET_HOUR")
  local timeformat_is_ampm = Settings:get("TIME_FORMAT")
  Event:change_parameter("TIME_FORMAT", timeformat_is_ampm)
  data["FirstUseWizard_Layer.SetTime_Page.ampm_text.grd_hidden"] = true

  if timeformat_is_ampm ~= 0 then -- TIME_FORMAT: 0 == 24hr format, 1 == 12hr AM/PM format
    local meta_data = get_type_meta_data("AM_OR_PM")
    local ampm_string = i18n:get("SET_AM_OR_PM." .. (Settings:get("SET_AM_OR_PM") == 0 and meta_data.min or meta_data.max))
    data["FirstUseWizard_Layer.SetTime_Page.ampm_text.value"] = ampm_string
    data["FirstUseWizard_Layer.SetTime_Page.ampm_text.grd_hidden"] = false
    hour_value = Settings:get("SET_12HOUR")
  end
  data["FirstUseWizard_Layer.SetTime_Page.hours_text.value"] = hour_value
  gre.set_data(data)
end

--
-- store newly entered date/time
--
function CBCOOK_FirstUseWizard_OnSaveDateTime()
  Event:change_parameter("TIMEZONE_OFFSET", Settings:get("TIMEZONE_OFFSET"))
  local new_year = Settings:get("SET_YEAR")
  local new_month = Settings:get("SET_MONTH")
  local new_day = Settings:get("SET_DAY")
  local new_minute = Settings:get("SET_MINUTE")
  local new_24hour = Settings:get("SET_HOUR")
  local new_12hour = 0
  local pm_active = 0
  if(Settings:get("TIME_FORMAT") ~= 0) then -- 0 == 24hr format, 1 == 12hr AM/PM format
    new_12hour = Settings:get("SET_12HOUR")
    pm_active = Settings:get("SET_AM_OR_PM") -- 0 == AM, 1 == PM
    if (pm_active == 0) then -- AM time active
      if new_12hour == 12 then
        new_24hour = 0
      else
        new_24hour = new_12hour
      end
    else -- PM time active
      if new_12hour == 12 then
        new_24hour = 12
      else
        new_24hour = new_12hour + 12
      end
    end
  end
  Event:live_set("SYS_TIME", local_to_epoch(new_year, new_month, new_day, new_24hour, new_minute, 0))
  print("CBCOOK_FirstUseWizard_OnSaveDateTime() set SYS_TIME hrs=", new_24hour)
  Event:live_get("SYS_TIME")
end

--
-- cancel first use wizard, limited functionality will be available
--
function FirstUseWizard_OnCancel()
  FirstUseWizard_Close()
end

--
-- first use wizard should only be shown after startup if not yet commissioned
--
local function FirstUseWizard_ShouldShow()
  if gFirstUseWizardShown then return false end

  gFirstUseWizardShown = true

  return (Settings:get_def("COMMISSION_TIME", 0) == 0)
     and string.len(Settings:get("SERIALNR_STR")) > 0
end

--
-- language selection layer opened
--
function FirstUseWizard_OnEnterLanguageSelection()
  Control_SetButtons(false, false, false, false, false, false, true)
  CBLanguage_OnOverlayShow()
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = true })
end

--
-- language selection layer closed
--
function FirstUseWizard_OnExitLanguageSelection()
  if gFirstUseWizardActive then
    Control_SetButtons(true, true, true, true, true, true, true)
    gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = false })
  end
end

--
-- first use wizard should be shown again after standbys
--
function FirstUseWizard_OnWakeUpFromStandby()
  gFirstUseWizardShown = false
end

--
-- open the first use wizard
-- move language layer z-index up so it appears on top of the first use wizard
--
function CBCOOK_FirstUseWizard_OnShow()
  if not FirstUseWizard_ShouldShow() then return end
  local meta_data = get_type_meta_data("24HR_AMPM")
  local time_format = i18n:get("TIME_FORMAT." .. (Settings:get("TIME_FORMAT") == 0 and meta_data.min or meta_data.max))
  local hour24_value = Settings:get("SET_HOUR")
  local serial_number = string.format("%s", Settings:get("SERIALNR_STR"))
  local enroll_url_string = string.format("%s", Settings:get("ENROLL_URL_STR"))
  local wifi_available = (Settings:get("WIFI_AVAILABLE") == 1)

  local data = {}
  data["FirstUseWizard_Layer.AmPm_Page.format_text.value"] = time_format
  data["FirstUseWizard_Layer.SetDate_Page.year_text.value"] = Settings:get("SET_YEAR")
  data["FirstUseWizard_Layer.SetDate_Page.month_text.value"] = Settings:get("SET_MONTH")
  data["FirstUseWizard_Layer.SetDate_Page.day_text.value"] = Settings:get("SET_DAY")
  data["FirstUseWizard_Layer.SetTime_Page.hours_text.value"] = hour24_value
  data["FirstUseWizard_Layer.SetTime_Page.minutes_text.value"] = Settings:get("SET_MINUTE")
  data["FirstUseWizard_Layer.SetTime_Page.timezone_text.value"] = Settings:get("TIMEZONE_OFFSET")
  data["FirstUseWizard_Layer.EnterLocation_Page.location_ctrl.text"] = Settings:get_def("STORENUMBER_STR", "")

  local timeformat_is_ampm = Settings:get("TIME_FORMAT")
  if timeformat_is_ampm ~= 0 then -- TIME_FORMAT: 0 == 24hr format, 1 == 12hr AM/PM format
    data["FirstUseWizard_Layer.SetTime_Page.hours_text.value"] = Settings:get("SET_12HOUR")
    local meta_data = get_type_meta_data("AM_OR_PM")
    local ampm_string = i18n:get("SET_AM_OR_PM." .. (Settings:get("SET_AM_OR_PM") == 0 and meta_data.min or meta_data.max))
    data["FirstUseWizard_Layer.SetTime_Page.ampm_text.value"] = ampm_string
  end

  gre.set_data(data)
  gre.set_group_attrs("FirstUseWizard_Layer.Welcome_Page", { hidden = false })
  gre.set_layer_attrs("COOK_Screen.FirstUseWizard_Layer", { hidden = false })
  gFirstUseWizardActive = true
  gCurrentPageIndex = 1

  -- make a QR code for the serialnumber
  if wifi_available then
    gQrCode:show(enroll_url_string .. serial_number)
    gre.set_value("FirstUseWizard_Layer.QrCode_Page.QrCode_Canvas.grd_hidden", false)
  else
    gre.set_value("FirstUseWizard_Layer.QrCode_Page.QrCode_Canvas.grd_hidden", true) -- don't show QR code of the serialnumber, since serialnumber is unknown.
  end
end

-- FID 4436 removed this 5sec press action, due to customers unknowingly activating this.
--
--[[ press and hold outside wizard for 5s to activate demo mode ]]--
--
-- activate demo mode
--
function CBCOOK_FirstUseWizard_OnActivateDemoMode(mapargs)
--  FirstUseWizard_Close()

--  Settings:set("DEMO_MODE", 1)
--  Event:change_parameter("DEMO_MODE", Settings:get("DEMO_MODE"))
--  Event:get_recipes()
end

--[[ store location page ]]--

local STORE_LOCATION_MAX_LENGTH = 10

local gStoreLocation = ""
local gEditingStoreLocation = false

--
-- keyboard key press
--
function CBCOOK_FirstUseWizard_InputKeyEvent(mapargs)
  if not gEditingStoreLocation then return end

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    gStoreLocation = Utf8_RemoveLastCharacter(gStoreLocation)
  else
    -- append char
    local tmp = gStoreLocation..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= STORE_LOCATION_MAX_LENGTH then
      gStoreLocation = tmp
    end
  end

  gre.set_value("FirstUseWizard_Layer.EnterLocation_Page.location_ctrl.text", gStoreLocation.."|")
end

--
-- store location confirmed
--
function CBCOOK_FirstUseWizard_OnKeyboardConfirm(mapargs)
  if not gEditingStoreLocation then return end

  gEditingStoreLocation = false
  Settings:set("STORENUMBER_STR", gStoreLocation)
  Event:change_parameter_string("STORENUMBER_STR", gStoreLocation)

  gre.set_value("FirstUseWizard_Layer.EnterLocation_Page.location_ctrl.text", gStoreLocation)
end

--
-- edit store location cancelled
--
function CBCOOK_FirstUseWizard_OnKeyboardCancel(mapargs)
  if not gEditingStoreLocation then return end

  gEditingStoreLocation = false
  gStoreLocation = Settings:get_def("STORENUMBER_STR", "")

  gre.set_value("FirstUseWizard_Layer.EnterLocation_Page.location_ctrl.text", gStoreLocation)
end

--
-- begin editing store location
--
function CBCOOK_FirstUseWizard_OnEditStoreLocation(mapargs)
  if gEditingStoreLocation then return end

  gEditingStoreLocation = true
  gStoreLocation = Settings:get_def("STORENUMBER_STR", "")

  gre.set_value("FirstUseWizard_Layer.EnterLocation_Page.location_ctrl.text", gStoreLocation.."|")
end

--[[ editing date and time settings ]]--

local gActiveSettingControl = nil
local gActiveSetting = nil

--
-- modify setting using slider
--
local function FirstUseWizard_OnEditSlider(item, unit_key)
  local data = {}

  data["FirstUseWizard_Slider_Layer.Title_Text.text"] = i18n:get(item.param)
  data["FirstUseWizard_Slider_Layer.Integer_Group.Number_Text.text"] = string.format("%d", Settings:get_def(item.param, item.min))
  data["FirstUseWizard_Slider_Layer.Slider_Group.Interval0_Text.text"] = string.format("%d", item.min)
  for i = 1,7 do
    data["FirstUseWizard_Slider_Layer.Slider_Group.Interval"..i.."_Text.text"] = math.floor((item.min + ((i/8) * (item.max - item.min))) + 0.5)
  end
  data["FirstUseWizard_Slider_Layer.Slider_Group.Interval8_Text.text"] = string.format("%d", item.max)

  local min = 25
  local width = gre.get_value("FirstUseWizard_Slider_Layer.Slider_Group.Slider_Btn.grd_width") - 58
  local icon_center = 25
  local percentage = (Settings:get_def(item.param, item.min) - item.min) / (item.max - item.min)
  local position = min + (percentage * width) - icon_center
  data["FirstUseWizard_Slider_Layer.Slider_Group.Slider_Btn.x"] = position
  data["FirstUseWizard_Slider_Layer.Unit_Text.text"] = unit_key and i18n:get(unit_key) or ""

  gre.set_data(data)
  gre.set_layer_attrs("FirstUseWizard_Slider_Layer", { hidden = false })
end

--
-- modify setting using toggle
--
local function FirstUseWizard_OnEditToggle(item)
  local data = {}

  data["FirstUseWizard_Toggle_Layer.Title_Text.text"] = i18n:get(item.param)
  data["FirstUseWizard_Toggle_Layer.Swap_Left.text"] = i18n:get(item.param .. "." .. item.min)
  data["FirstUseWizard_Toggle_Layer.Swap_Right.text"] = i18n:get(item.param .. "." .. item.max)

  if (Settings:get(item.param) or 0) == 0 then
    data["FirstUseWizard_Toggle_Layer.toggled"] = 0
    data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 3
  else
    data["FirstUseWizard_Toggle_Layer.toggled"] = 1
    data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 57
  end

  gre.set_data(data)
  gre.set_layer_attrs("FirstUseWizard_Toggle_Layer", { hidden = false })
end

--
-- edit date/time settings
--
function CBCOOK_FirstUseWizard_OnEditDateTime(mapargs)
  Control_SetButtons(false, false, false, false, false, false, false)
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = true })

  local timeformat_is_ampm = Settings:get("TIME_FORMAT")
  local settings_map = {
    ["FirstUseWizard_Layer.SetDate_Page.year_text"    ] = "SET_YEAR",
    ["FirstUseWizard_Layer.SetDate_Page.month_text"   ] = "SET_MONTH",
    ["FirstUseWizard_Layer.SetDate_Page.day_text"     ] = "SET_DAY",
    ["FirstUseWizard_Layer.SetTime_Page.hours_text"   ] = "SET_HOUR",
    ["FirstUseWizard_Layer.SetTime_Page.minutes_text" ] = "SET_MINUTE",
    ["FirstUseWizard_Layer.SetTime_Page.timezone_text"] = "TIMEZONE_OFFSET",
    ["FirstUseWizard_Layer.AmPm_Page.format_text"     ] = "TIME_FORMAT"
  }

  if timeformat_is_ampm ~= 0 then -- 0 == 24hr format, 1 == 12hr AM/PM format
    settings_map["FirstUseWizard_Layer.SetTime_Page.hours_text"] = "SET_12HOUR"
    settings_map["FirstUseWizard_Layer.SetTime_Page.ampm_text" ] = "SET_AM_OR_PM"
  end

  gActiveSettingControl = mapargs.context_control
  local setting = settings_map[gActiveSettingControl]
  if not setting then return end
  gActiveSetting = FindSetting(setting)

  local settings_edit_map = {
    ["YEAR"     ] = { action = FirstUseWizard_OnEditSlider                                 },
    ["MONTH"    ] = { action = FirstUseWizard_OnEditSlider                                 },
    ["DAY"      ] = { action = FirstUseWizard_OnEditSlider                                 },
    ["HOUR"     ] = { action = FirstUseWizard_OnEditSlider, unit = "NLS_UNIT_HOURS_LONG"   },
    ["MINUTE"   ] = { action = FirstUseWizard_OnEditSlider, unit = "NLS_UNIT_MINUTES_LONG" },
    ["12HOUR"   ] = { action = FirstUseWizard_OnEditSlider, unit = "NLS_UNIT_HOURS_LONG"   },
    ["24HR_AMPM"] = { action = FirstUseWizard_OnEditToggle                                 },
    ["AM_OR_PM" ] = { action = FirstUseWizard_OnEditToggle                                 }
  }

  local begin_edit = settings_edit_map[gActiveSetting.type]
  if begin_edit then begin_edit.action(gActiveSetting, begin_edit.unit) end
end

--[[ toggle control ]]--

--
-- cancel editing using toggle
--
function CBCOOK_FirstUseWizard_Toggle_OnCancel()
  Control_SetButtons(true, true, true, true, true, true, false)
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = false })
  gre.set_layer_attrs("FirstUseWizard_Toggle_Layer", { hidden = true })
end

--
-- store value set with toggle
--
function CBCOOK_FirstUseWizard_Toggle_OnSave()
  local new_value = gre.get_value("FirstUseWizard_Toggle_Layer.toggled")
  Settings:set(gActiveSetting.param, new_value)
  gre.set_value(gActiveSettingControl..".value", i18n:get(gActiveSetting.param..".".. (new_value == 0 and gActiveSetting.min or gActiveSetting.max)))

  Control_SetButtons(true, true, true, true, true, true, false)
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = false })
  gre.set_layer_attrs("FirstUseWizard_Toggle_Layer", { hidden = true })
end

--
-- left toggle label pressed -> select left option
--
function CBCOOK_FirstUseWizard_Toggle_OnPressLeft()
  local data = {}
  data["FirstUseWizard_Toggle_Layer.toggled"   ] = 0
  data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 3
  gre.set_data(data)
end

--
-- right toggle label pressed -> select right option
--
function CBCOOK_FirstUseWizard_Toggle_OnPressRight()
  local data = {}
  data["FirstUseWizard_Toggle_Layer.toggled"   ] = 1
  data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 57
  gre.set_data(data)
end

--
-- swap toggle when it's pressed
--
--- @param gre#context mapargs
function CBCOOK_FirstUseWizard_Toggle_OnPress(mapargs)
  local data = {}

  local toggled = gre.get_value("FirstUseWizard_Toggle_Layer.toggled")
  toggled = 1 - toggled
  data["FirstUseWizard_Toggle_Layer.toggled"] = toggled

  if toggled == 0 then
    -- move button to the left
    data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 3
  else
    -- move button to the right
    data["FirstUseWizard_Toggle_Layer.Swap_Btn.x"] = 57
  end

  gre.set_data(data)
end

--[[ slider control ]]--

--
-- cancel editing using slider
--
function CBCOOK_FirstUseWizard_Slider_OnCancel()
  Control_SetButtons(true, true, true, true, true, true, false)
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = false })
  gre.set_layer_attrs("FirstUseWizard_Slider_Layer", { hidden = true })
end

--
-- store value set with slider
--
function CBCOOK_FirstUseWizard_Slider_OnSave()
  local new_value = tonumber(gre.get_value("FirstUseWizard_Slider_Layer.Integer_Group.Number_Text.text"))
  Settings:set(gActiveSetting.param, new_value)
  gre.set_value(gActiveSettingControl..".value", new_value)

  Control_SetButtons(true, true, true, true, true, true, false)
  gre.set_layer_attrs("FirstUseWizard_Layer", { hidden = false })
  gre.set_layer_attrs("FirstUseWizard_Slider_Layer", { hidden = true })
end

--
-- handle dragging slider around
--
function CBCOOK_FirstUseWizard_Slider_OnMotion(mapargs)
  local min_val = gActiveSetting.min
  local max_val = gActiveSetting.max

  local pressed = gre.get_value(mapargs.context_control..".slider_active") == 1

  if pressed then
    local min = 25
    local width = gre.get_value(mapargs.context_control..".grd_width") - 58
    local max = min + width
    local icon_center = 25

    local x = mapargs.context_event_data.x

    if x >= min and x <= max then
      local percentage = x > max and 1 or (x >= min and (x - min) / width or 0)
      local position = min + (percentage * width) - icon_center
      local value = math.floor(min_val + (percentage * (max_val - min_val)) + 0.5)

      gre.set_value(mapargs.context_control..".x", position)
      gre.set_value("FirstUseWizard_Slider_Layer.Integer_Group.Number_Text.text", string.format("%d", value))
    end
  end
end

--
-- handle label touch
--
function CBCOOK_FirstUseWizard_Slider_OnLabelPress(mapargs)
  local slider_info = gre.get_control_attrs("FirstUseWizard_Slider_Layer.Slider_Group.Slider_Bg", "x")
  local label_info = gre.get_control_attrs(mapargs.context_control, "x", "width")
  local label_center = label_info.x + (label_info.width / 2) - slider_info.x

  mapargs.context_control = "FirstUseWizard_Slider_Layer.Slider_Group.Slider_Btn"
  mapargs.context_event_data.x = math.floor(label_center)

  gre.set_value("FirstUseWizard_Slider_Layer.Slider_Group.Slider_Btn.slider_active", 1)
  CBCOOK_FirstUseWizard_Slider_OnMotion(mapargs)
  gre.set_value("FirstUseWizard_Slider_Layer.Slider_Group.Slider_Btn.slider_active", 0)
end
