----------------------
-- Help screen / Topic
----------------------

--[[ includes ]]--
require("Helpdir")
require("History")
require("Scrollbar")
require("Util")

local json = require("json")

--[[ local data ]]--

local TITLE_CONTROL = "HelpTopic_Layer.Title_Text"
local IMAGE_CONTROL = "HelpTopic_Layer.Image_Img"
local BODY_CONTROL = "HelpTopic_Layer.Body_Text"

local gHelpTopicActive = false
local gReturnCallback = nil
local gScrollbar = LayerScrollbar.new{layer="HelpTopic_Layer",scrollbar="HelpBackground_Layer.Scrollbar"}
local gLastImageFileName = nil
local gLastScaledImageName = nil

--[[ local functions ]]--

--
-- parse the help topic json file
--
local function parse_topic_json(json_file)
  local f = assert(io.open(json_file, "r"))
  local t = f:read("*all")
  f:close()

  local json_data = json.decode(t)

  local title = json_data.title or ""
  local image_file = json_data.image and help_path(json_data.image) or ""
  local image = file_exists(image_file) and image_file or nil
  local body = json_data.text or ""

  return title, image, body
end

--
-- show a help topic with optional image on the display
--
local function show_topic(title, image, body)
  local width = gre.get_layer_attrs("HelpTopic_Layer", "width").width
  local title_ypos = gre.get_value(TITLE_CONTROL..".grd_y")
  local title_dimensions = gre.rtext_text_extent(title, TITLE_CONTROL)
  local body_dimensions = gre.rtext_text_extent(body, BODY_CONTROL)
  local image_width, image_height = 0, 0
  local image_present = image ~= nil
  if image_present then image_width, image_height = png_dimensions(image) end
  image_present = image_present and image_width ~= nil and image_height ~= nil

  local scaled_image_width, scaled_image_height = 0, 0
  if image_present then
    if image_width > width then
      -- must rescale so that width fits
      scaled_image_width = width
      scaled_image_height = (width / image_width) * image_height
    else
      -- do not rescale
      scaled_image_width = image_width
      scaled_image_height = image_height
    end
  end

  local title_height = title_dimensions.height
  local image_xpos = (width - scaled_image_width) / 2
  local image_ypos = title_height + 10
  local body_ypos = image_ypos + scaled_image_height + 10
  local body_height = body_dimensions.height
  local total_height = body_ypos + body_height

  if image_present then -- FID4527 remember the filename of the help image, use it later to free it's memory.
    -- print("Help image:"..image.." W="..image_width.." x H="..image_height.."")
    gLastImageFileName = tostring(image)
    gLastScaledImageName = tostring(image) .. ":" .. tostring(scaled_image_width) .. "x" .. tostring(scaled_image_height)
  end

  local data = {}
  data[TITLE_CONTROL..".text"] = title
  data[TITLE_CONTROL..".grd_height"] = title_height
  data[TITLE_CONTROL..".grd_hidden"] = false
  data[IMAGE_CONTROL..".grd_hidden"] = not image_present
  data[IMAGE_CONTROL..".image"] = image_present and image or ""
  data[IMAGE_CONTROL..".grd_height"] = scaled_image_height
  data[IMAGE_CONTROL..".grd_width"] = scaled_image_width
  data[IMAGE_CONTROL..".grd_x"] = image_xpos
  data[IMAGE_CONTROL..".grd_y"] = image_ypos
  data[BODY_CONTROL..".text"] = body
  data[BODY_CONTROL..".grd_height"] = body_height
  data[BODY_CONTROL..".grd_y"] = body_ypos
  data[BODY_CONTROL..".grd_hidden"] = false
  gre.set_data(data)

  gScrollbar:init(total_height)
end

--[[ public functions ]]--

--
-- show a help topic
--
-- return_callback is called when control is returned using back button
--
function Help_ShowTopic(topic_json_file, return_callback)
  History:force_push("HELP_Screen")
  gHelpTopicActive = true
  gReturnCallback = return_callback

  show_topic(parse_topic_json(topic_json_file))

  gre.set_layer_attrs("HelpToc_Layer", { hidden = true })
  gre.set_layer_attrs("HelpTopic_Layer", { hidden = false, yoffset = 0 })
end

--[[ event handlers ]]--

--
-- synchronize scrollbar location to layer position
--
function CBHelp_Topic_SyncScrollbar(mapargs)
  gScrollbar:scroll()
end

--
-- back button in status bar pressed to exit current help topic screen
--
function CBHelp_Topic_OnBackPressed(mapargs)
  if gHelpTopicActive then
    --print("Memory Before garbage collect:", collectgarbage("count"))
    if gReturnCallback ~= nil then
      gReturnCallback()
      gReturnCallback = nil
    end
    local data = {} -- FID4527 remove any reference to the "help topic" image file from storyboard elements.
    data[IMAGE_CONTROL..".image"] = ""
    data[IMAGE_CONTROL..".grd_height"] = 0
    data[IMAGE_CONTROL..".grd_width"] = 0
    data[IMAGE_CONTROL..".grd_x"] = 0
    data[IMAGE_CONTROL..".grd_y"] = 0
    gre.set_data(data)

    gre.set_layer_attrs("HelpToc_Layer", { hidden = false })
    gre.set_layer_attrs("HelpTopic_Layer", { hidden = true })
    gHelpTopicActive = false

    -- FID4527 Force the last "help topic" image out of resource cache to prevent running out of memory during browsing through help pages.
    if gLastImageFileName ~= nil then
      -- print("==============================================")
      --show_cache("image")
      print("Releasing help image: " .. tostring(gLastScaledImageName))
      gre.dump_resource("image", gLastImageFileName) -- free the file: "image name" (non scaled image)
      gre.dump_resource("image", gLastScaledImageName) -- free the file: "image name"+":"+image_width+"x"+image_height (scaled image)
      gLastImageFileName = nil
      gLastScaledImageName = nil
      show_cache_memory_used("image")
    end
    -- collectgarbage("collect") -- force a LUA script garbage collection cycle (not relevant to this image memory OOM problem).
  end
end
