------------------------------------------------------------------------------
-- keep track of previous screens, for the back button and dragging the screen
------------------------------------------------------------------------------

--[[ screen history class implementation ]]--

local MAX_STACK_SIZE = 10    -- maximum number of screens in the history stack

-- history singleton class
History = {}
History.__index = History

History.stack = {}           -- screen history stack
History.home = "COOK_Screen" -- home screen
History.exclude = {          -- begin exclude list [
  "START_Welcome_Screen",    -- start frijado logo screen
  "START_Customer_Screen",   -- start customer logo screen
  "STANDBY_Screen",          -- standby screen
  "CLEAN_Screen",            -- clean screen
  "DOOROPEN_Screen",         -- door opened screen
  "PAUSE_Screen"             -- pause screen
}                            -- ] end exclude list
History.back = false         -- tracking back press, we do not want to push the screen to
                             -- the history if the gre.screenhide.pre event was caused by
                             -- the back button being pressed (this would create a screen
                             -- history loop resulting in the back button switching between
                             -- the current and last screens instead of being able to go
                             -- all the way back through the screen history)

--
-- push entry to end of stack, but keep stack size limited (pop old entries if needed)
--
local function stack_insert(self, screen_name)
  if #self.stack < MAX_STACK_SIZE then
    table.insert(self.stack, screen_name)
  else
    -- remove oldest entry
    table.remove(self.stack, 1)
    -- set first entry to home screen, but make sure it's not there twice
    if self.stack[2] ~= self.home then
      self.stack[1] = self.home
    else
      table.remove(self.stack, 1)
    end
    -- insert new entry
    table.insert(self.stack, screen_name)
  end
end

--
-- check if a screen is on the history exclude list
--
function History:on_exclude_list(screen_name)
  for _, screen in ipairs(self.exclude) do
    if screen == screen_name then
      return true
    end
  end
  return false
end

--
-- get the name of the previous screen
--
function History:peak()
  return self.stack[#self.stack] or self.home
end

--
-- remove duplicates from the top of the stack
--
local function pop_duplicates(self)
  local screen_name = self.stack[#self.stack] or self.home
  while #self.stack > 1 and self.stack[#self.stack-1] == screen_name do
    table.remove(self.stack)
  end
end

--
-- push a screen name on the history stack
-- call this function in gre.screenhide.pre
--
function History:push(screen_name)
  pop_duplicates(self)
  if not self.back and not self:on_exclude_list(screen_name) then
    if self.home == screen_name then self.stack = {} end
    if self.stack[#self.stack] ~= screen_name then stack_insert(self, screen_name) end
    gre.set_value("toScreen", screen_name)
  else
    self.back = false
  end
end

--
-- forcefully push a screen on the history stack without doing any checks
-- use this when you need to manually navigate through layers within a screen
--
function History:force_push(screen_name)
  stack_insert(self, screen_name)
end

--
-- switch to the previous screen and pop that name from the stack
-- call this function from back button press event
--
function History:pop()
  self.back = true
  gre.set_value("toScreen", table.remove(self.stack) or self.home)
  gre.send_event("switch_screen")
end

--[[ handle relevant events ]]--

--
-- application wide gre.screenhide.pre event handler
--
--- @param gre#context mapargs
function CB_OnScreenHide(mapargs)
  History:push(mapargs.context_screen)
end

--
-- back button press event handler
--
--- @param gre#context mapargs
function CB_OnBackPressed(mapargs)
  if mapargs.context_screen == "CR_Screen" then
    CR_OnBackPressed()
  else
    History:pop()
  end
end
