-----------------------------------
-- implementation of the log screen
-----------------------------------

require("Scrollbar")
require("History")
require("Time")
require("language")
require("Util")

local TABLE_LOG = "Log_Layer.Log_Table"                               -- log table
local SCROLLBAR = "Log_Layer.Scrollbar"                               -- log table scrollbar
local LOG_LAYER = "LOG_Screen.Log_Layer"                              -- log list screen
local LOGDETAIL_LAYER = "LOG_Screen.LogDetail_Layer"                  -- log detail screen

local gLogEntries = nil                                               -- list of log entries
local gLogEntry = nil                                                 -- active log entry (detail screen)
local gScrollbar = Scrollbar.new{table=TABLE_LOG,scrollbar=SCROLLBAR} -- settings menu scrollbar

--
-- error message
--
local function errorcode_to_message(code, devicenr)
  --return i18n:get("ERROR_"..code.."_"..devicenr)
  return i18n:get("ERROR_"..code)
end

--
-- error description
--
local function errorcode_to_description(code, devicenr)
  --return i18n:get("ERROR_"..code.."_"..devicenr.."_MSG")
  return i18n:get("ERROR_"..code.."_DESC")
end

--
-- error/info icon
--
local function errorcode_to_icon(code, active)
  return in_array(code, {10,12,16}) and image_path("icon-i.png") or (active and image_path("alarm-actief-2.png") or image_path("icon-!.png"))
end

--
-- get active/inactive image
--
local function icon_for_active(active)
  return active and image_path("CR-vinkje-klein-fel.png") or image_path("CR-vinkje-klein-grijs.png")
end

--
-- for debugging perpose print the contents of a received io_errorlog_reply event
--
--- @param gre#context mapargs
local function ErrorLog_Print(mapargs)  
  print("CBLOG_OnErrorLog #" .. tostring(#mapargs.context_event_data.code))
  
  for i = 1, #mapargs.context_event_data.code do
    local code   = mapargs.context_event_data.code[i]
    local active = mapargs.context_event_data.active[i] == 1
    local device = mapargs.context_event_data.device[i]
    local start  = mapargs.context_event_data.start[i]
    local stop   = mapargs.context_event_data.stop[i]
    
    local ts_start_date, ts_start_time = format_timestamp(start)
    local ts_end_date, ts_end_time = format_timestamp(stop)
    print("code   : " .. tostring(code))
    print("active : " .. tostring(active))
    print("device : " .. tostring(device))
    print("start  : " .. tostring(start))
    print("stop   : " .. tostring(stop))
    print("start  : " .. ts_start_date .. " " .. ts_start_time)
    print("stop   : " .. ts_end_date   .. " " .. ts_end_time)
  end
end

--
-- synchronize the scrollbar with the settings menu position
--
function CBLOG_SyncScrollbar()
  gScrollbar:scroll()
end

--
-- get the text width in a control (control is required to have font and font_size available)
--
local function text_width(text, control)
  local font = gre.get_value(control..".font")
  local font_size = gre.get_value(control..".font_size")
  return gre.get_string_size(font, font_size, text).width
end

--
-- handle log item pressed, transition to details screen
--
--- @param gre#context mapargs
function CBLOG_OnItemPressed(mapargs)
  if #gLogEntries == 0 then return end

  gLogEntry = mapargs.context_row
  local entry = gLogEntries[gLogEntry]

  History:force_push("LOG_Screen")

  gre.set_layer_attrs(LOG_LAYER, { hidden = true })
  gre.set_layer_attrs(LOGDETAIL_LAYER, { hidden = false })

  local ts_start_date, ts_start_time = format_date(entry.start), format_time(entry.start)
  local ts_end_date, ts_end_time = format_date(entry.stop), format_time(entry.stop)
  ts_start_date = ts_start_date.." - "
  ts_end_date = ts_end_date.." - "
  local ts_start_time_x = gre.get_value("LogDetail_Layer.Start_Timestamp.date_x") + text_width(ts_start_date, "LogDetail_Layer.Start_Timestamp")
  local ts_end_time_x = gre.get_value("LogDetail_Layer.End_Timestamp.date_x") + text_width(ts_end_date, "LogDetail_Layer.End_Timestamp")

  local data = {}
  data["LogDetail_Layer.LogEntry_Text.text"] = errorcode_to_message(entry.code, entry.device)
  data["LogDetail_Layer.Active_Image.image"] = icon_for_active(entry.active)
  data["LogDetail_Layer.Device_Number.text"] = entry.device
  data["LogDetail_Layer.Code_Number.text"] = entry.code
  data["LogDetail_Layer.Start_Timestamp.text_date"] = ts_start_date
  data["LogDetail_Layer.Start_Timestamp.text_time"] = ts_start_time
  data["LogDetail_Layer.Start_Timestamp.time_x"] = ts_start_time_x
  if entry.active then
    -- End_Timestamp is meaningless
    data["LogDetail_Layer.End_Timestamp.text_date"] = ""
    data["LogDetail_Layer.End_Timestamp.text_time"] = ""
    data["LogDetail_Layer.End_Timestamp.time_x"] = ""
  else
    data["LogDetail_Layer.End_Timestamp.text_date"] = ts_end_date
    data["LogDetail_Layer.End_Timestamp.text_time"] = ts_end_time
    data["LogDetail_Layer.End_Timestamp.time_x"] = ts_end_time_x
 end
  data["LogDetail_Layer.LogEntry_Description.text"] = string.format("(%s)", errorcode_to_description(entry.code, entry.device))
  gre.set_data(data)
end

--
-- forward touch event on overlaying filter/icon to table below
--
--- @param gre#context mapargs
function CBLOG_FowardOverlayTouch(mapargs)
  forward_touch_to_table(mapargs, TABLE_LOG, CBLOG_OnItemPressed)
end

--
-- handle back button press (if in details screen, go to list; if in list, go to previous screen)
--
function CBLOG_BackPress()
  if gLogEntry then
    gLogEntry = nil
    gre.set_layer_attrs(LOG_LAYER, { hidden = false })
    gre.set_layer_attrs(LOGDETAIL_LAYER, { hidden = true })
  end
end

--
-- prepare the screen
--
--- @param gre#context mapargs
function CBLOG_OnPrepare(mapargs)
  gLogEntries = {}
  gLogEntry = nil
  Event:get_errorlog()
  gre.set_value(TABLE_LOG..".grd_hidden", true)
  gre.set_value(SCROLLBAR..".grd_hidden", true)
  gre.set_table_attrs(TABLE_LOG, { rows = 0, yoffset = 0 })
  gre.set_layer_attrs(LOG_LAYER, { hidden = false })
  gre.set_layer_attrs(LOGDETAIL_LAYER, { hidden = true })
  
  Control_SetButtons(true, true, true, true, true, true, false)
end

--
-- received the error log with 'io_errorlog_reply' event
--
--- @param gre#context mapargs
function CBLOG_OnErrorLog(mapargs)
  local icon_x_pos = {
    [image_path("alarm-actief-2.png")] = 410,
    [image_path("icon-!.png")] = 410,
    [image_path("icon-i.png")] = 428,
  }

  local data = {}
  
  --ErrorLog_Print(mapargs)
  
  local code   = convert_to_table(mapargs.context_event_data.code)
  local active = convert_to_table(mapargs.context_event_data.active)
  local device = convert_to_table(mapargs.context_event_data.device)
  local start  = convert_to_table(mapargs.context_event_data.start)
  local stop   = convert_to_table(mapargs.context_event_data.stop)
  
  local lastChunck = false 
  
  for i = 1, #code do
    if code[i] > 0 then    
      table.insert(gLogEntries, {
        code   = code[i],
        active = active[i],
        device = device[i],
        start  = start[i],
        stop   = stop[i] 
      })

      local n = #gLogEntries
      
      data[TABLE_LOG..".date."..n..".1"] = format_date(start[i])
      data[TABLE_LOG..".timestamp."..n..".1"] = format_time(start[i])
      data[TABLE_LOG..".text."..n..".1"] = errorcode_to_message(code[i], device[i])
      data[TABLE_LOG..".image."..n..".1"] = errorcode_to_icon(code[i], active[i] == 1)
      data[TABLE_LOG..".icon_x."..n..".1"] = icon_x_pos[data[TABLE_LOG..".image."..n..".1"]]
    else
      -- for the last chunk has a 0-termination record  
      lastChunck = true
      break   
    end
  end
    
  if #gLogEntries == 0 and lastChunck then
    data[TABLE_LOG..".date.1.1"] = format_date()
    data[TABLE_LOG..".timestamp.1.1"] = format_time()
    data[TABLE_LOG..".text.1.1"] = i18n:get("NLS_NO_ITEMS")
    data[TABLE_LOG..".image.1.1"] = ""
  end
  
  data[TABLE_LOG..".grd_hidden"] = false
  gre.set_data(data)
   
  if lastChunck then    
    gre.set_table_attrs(TABLE_LOG , { rows = #gLogEntries or 1, yoffset = 0 })
    gScrollbar:init()
  end
end
