-----------------------------
-- language selection overlay
-----------------------------

require("events")
require("language")
require("Settings")
require("Scrollbar")

local LANGUAGE_LAYER = "Language_Layer"
local LANGUAGE_TABLE = LANGUAGE_LAYER..".languages_group.languages_table"
local LANGUAGE_TABLE_SCROLLBAR = LANGUAGE_LAYER..".Scrollbar"
local SLIDER_OVERLAY = LANGUAGE_LAYER..".languages_group.slider_overlay"
local LANGAUGE_KEY = "LANGUAGE"
local ACTIVE_TEXT_COLOR = 0xFFFFFF
local NORMAL_TEXT_COLOR = 0x000000
local ACTIVE_IMAGE_ALPHA = 255
local NORMAL_IMAGE_ALPHA = 0
local ACTIVE_SEGMENT_ALPHA = 255
local HIDDEN_SEGMENT_ALPHA = 0
local SLIDER_IMAGE_HEIGHT = 40

local gScrollbar = Scrollbar.new{table=LANGUAGE_TABLE,scrollbar=LANGUAGE_TABLE_SCROLLBAR}
local gActiveLanguage = 0
local gSelectedIndex = 0
local gSliderIndex = 0
local gSliderY = 0

--
-- update scrollbar slider location
--
function CBLanguage_SyncScrollbar()
  gScrollbar:scroll()
end

--
-- cancel language selection, close overlay
--
function CBLanguage_OnCancel(mapargs)
  gre.set_layer_attrs(LANGUAGE_LAYER, { hidden = true })
  gre.send_event("language_not_changed")
end

--
-- confirm language selection
--
function CBLanguage_OnConfirm(mapargs)
  local language = gre.get_value(LANGUAGE_TABLE..".language."..gSelectedIndex..".1")
  local id = i18n:name_to_id(language)
  if language ~= gActiveLanguage then
    i18n:load(language)
    Settings:set(LANGAUGE_KEY, id)
    Event:change_parameter(LANGAUGE_KEY, id)
  end
  gre.set_layer_attrs(LANGUAGE_LAYER, { hidden = true })
  gre.send_event("language_changed")
end

--
-- select a new language
--
function CBLanguage_OnSelect(mapargs)
  local i = mapargs.context_row
  local count = gre.get_table_attrs(LANGUAGE_TABLE, "rows").rows
  
  gre.set_value(SLIDER_OVERLAY..".grd_hidden", true)
  
  if i > 1 and i < count and i ~= gSelectedIndex then
    local data = {}
    data[LANGUAGE_TABLE..".text_color."..gSelectedIndex..".1"] = NORMAL_TEXT_COLOR
    data[LANGUAGE_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = NORMAL_IMAGE_ALPHA
    data[LANGUAGE_TABLE..".text_color."..i..".1"] = ACTIVE_TEXT_COLOR
    data[LANGUAGE_TABLE..".active_image_alpha."..i..".1"] = ACTIVE_IMAGE_ALPHA
    gre.set_data(data)
    gSelectedIndex = i
  end
end

--
-- leave slider area -> cancel slider
--
function CBLanguage_OnSliderOutbound(mapargs)  
  local data = {}
  data[SLIDER_OVERLAY..".grd_hidden"] = true
  data[LANGUAGE_TABLE..".text_color."..gSliderIndex..".1"] = NORMAL_TEXT_COLOR
  data[LANGUAGE_TABLE..".text_color."..gSelectedIndex..".1"] = ACTIVE_TEXT_COLOR
  data[LANGUAGE_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = ACTIVE_IMAGE_ALPHA
  gre.set_data(data)
end

--
-- release slider -> accept choice if within bounds, otherwise cancel
--
function CBLanguage_OnSliderRelease(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(LANGUAGE_TABLE, "rows").rows
  
  gSliderIndex = calculate_table_cell_index(LANGUAGE_TABLE, y)
  
  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBLanguage_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[SLIDER_OVERLAY..".grd_hidden"] = true
    data[LANGUAGE_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[LANGUAGE_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[LANGUAGE_TABLE..".active_image_alpha."..gSliderIndex..".1"] = ACTIVE_IMAGE_ALPHA
    gre.set_data(data)
    gSelectedIndex = gSliderIndex
  end
end

--
-- slider motion -> update highlighted item if within bounds, otherwise cancel
--
function CBLanguage_OnSliderMotion(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(LANGUAGE_TABLE, "rows").rows
  
  gSliderIndex = calculate_table_cell_index(LANGUAGE_TABLE, y)
  
  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBLanguage_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[LANGUAGE_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[LANGUAGE_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- grab slider button to start sliding
--
function CBLanguage_OnStartSlider(mapargs)
  local i = mapargs.context_row
  local x = mapargs.context_event_data.x
  local y = mapargs.context_event_data.y
  
  local slider_info = gre.get_control_attrs(SLIDER_OVERLAY, "x", "width")
  
  if i == gSelectedIndex and x >= slider_info.x and x <= (slider_info.x + slider_info.width) then
    gSliderIndex = i
    
    local data = {}
    data[LANGUAGE_TABLE..".active_image_alpha."..i..".1"] = NORMAL_IMAGE_ALPHA
    data[SLIDER_OVERLAY..".grd_hidden"] = false
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- populate language selection table
--
function CBLanguage_OnOverlayShow(mapargs)
  gActiveLanguage = i18n:language()
  
  gSliderY = gre.get_control_attrs(SLIDER_OVERLAY, "y").y
  
  local data = {}
  
  data[SLIDER_OVERLAY..".grd_hidden"] = true
  
  -- top segment of slider
  data[LANGUAGE_TABLE..".flag.1.1"] = ""
  data[LANGUAGE_TABLE..".language.1.1"] = ""
  data[LANGUAGE_TABLE..".active_image_alpha.1.1"] = NORMAL_IMAGE_ALPHA
  data[LANGUAGE_TABLE..".top_segment_alpha.1.1"] = ACTIVE_SEGMENT_ALPHA
  data[LANGUAGE_TABLE..".body_segment_alpha.1.1"] = HIDDEN_SEGMENT_ALPHA
  data[LANGUAGE_TABLE..".bottom_segment_alpha.1.1"] = HIDDEN_SEGMENT_ALPHA
  
  -- slider content: selectable languages
  local translations = i18n:translations()
  for i, language in ipairs(translations) do
    local index = i + 1
    local metaData = i18n:meta_data(language)
    data[LANGUAGE_TABLE..".flag."..index..".1"] = metaData.icon or ""
    data[LANGUAGE_TABLE..".language."..index..".1"] = language
    data[LANGUAGE_TABLE..".text_color."..index..".1"] = language == gActiveLanguage and ACTIVE_TEXT_COLOR or NORMAL_TEXT_COLOR
    data[LANGUAGE_TABLE..".active_image_alpha."..index..".1"] = language == gActiveLanguage and ACTIVE_IMAGE_ALPHA or NORMAL_IMAGE_ALPHA
    data[LANGUAGE_TABLE..".top_segment_alpha."..index..".1"] = HIDDEN_SEGMENT_ALPHA
    data[LANGUAGE_TABLE..".body_segment_alpha."..index..".1"] = ACTIVE_SEGMENT_ALPHA
    data[LANGUAGE_TABLE..".bottom_segment_alpha."..index..".1"] = HIDDEN_SEGMENT_ALPHA
    gSelectedIndex = language == gActiveLanguage and index or gSelectedIndex
  end
  
  -- bottom segment of slider
  data[LANGUAGE_TABLE..".flag."..(#translations + 2)..".1"] = ""
  data[LANGUAGE_TABLE..".language."..(#translations + 2)..".1"] = ""
  data[LANGUAGE_TABLE..".active_image_alpha."..(#translations + 2)..".1"] = NORMAL_IMAGE_ALPHA
  data[LANGUAGE_TABLE..".top_segment_alpha."..(#translations + 2)..".1"] = HIDDEN_SEGMENT_ALPHA
  data[LANGUAGE_TABLE..".body_segment_alpha."..(#translations + 2)..".1"] = HIDDEN_SEGMENT_ALPHA
  data[LANGUAGE_TABLE..".bottom_segment_alpha."..(#translations + 2)..".1"] = ACTIVE_SEGMENT_ALPHA
  
  gre.set_data(data)
  gre.set_table_attrs(LANGUAGE_TABLE, { yoffset = 0, rows = #translations + 2 })
  gre.set_layer_attrs(LANGUAGE_LAYER, { hidden = false })
  
  gScrollbar:init()
end
