----------------------------------------------
-- implementation of a reusable QR code widget
----------------------------------------------

local qrencode = require("qrencode")

local BACKGROUND_COLOR = 0xFFFFFF
local PIXEL_COLOR      = 0x000000
local MARGIN           = 0

-- QR code class
QrCodeWidget = {}
QrCodeWidget.__index = QrCodeWidget
QrCodeWidget.qr_data = nil
QrCodeWidget.data_string = nil

--
-- constructor
--
function QrCodeWidget.new(o)
  o = o or {}
  assert(o.canvas, "data field is required")
  setmetatable(o, QrCodeWidget)
  return o
end

--
-- generate a QR code for a data string
--
local function GenerateQrCode(data_string)
  if data_string == nil or data_string == "" then return nil end

  local ok, qr_data = qrencode.qrcode(data_string)
  if not ok then return nil end
  return qr_data
end

--
-- render a QR code in a canvas
--
local function RenderQrCode(cvs, qr_data)
  local cvs_dimensions = cvs:get_dimensions()
  local width = cvs_dimensions.width
  local height = cvs_dimensions.height
  local pixel_size = math.floor((width - (MARGIN * 2)) / #qr_data[1])
  local margin = (width - pixel_size * #qr_data) / 2
  
  for x = 1, #qr_data do
    local row = qr_data[x]
    for y = 1, #row do
      if row[y] > 0 then
        cvs:fill_rect(((x-1) * pixel_size) + margin , ((y-1) * pixel_size) + margin, (x * pixel_size) + margin, (y * pixel_size) + margin, PIXEL_COLOR)
      end
    end
  end
end

--
-- update the canvas to show a QR code for the provided data string
--
function QrCodeWidget:show(data_string)
  if data_string ~= self.data_string then
    self.qr_data = GenerateQrCode(data_string)
    self.data_string = data_string
  end
  
  local cvs = gre.get_canvas(self.canvas)
  cvs:fill(BACKGROUND_COLOR)
  
  if self.qr_data == nil then return end
  
  RenderQrCode(cvs, self.qr_data)
end
