------------------------------------------------------
-- screen with the list of all recipes in the database
------------------------------------------------------

require("ActiveRecipe")
require("Scrollbar")
require("Settings")
require("Dialog")
require("History")
require("Login_Layer")
require("RoleManager")
require("StartTime_Dialog")   -- for CBStartTime_SelectTime_OnCancel() & Recipe_GetEnteredStartTime) call

local TABLE_RECIPES = "REC_Layer.REC_Recipes_Table"                       -- recipes table (main control)
local SCROLLBAR = "REC_Layer.Scrollbar"                                   -- scrollbar for the recipes table

local gScrollbar = Scrollbar.new{table=TABLE_RECIPES,scrollbar=SCROLLBAR} -- scrollbar for the recipes table
local gSelectedRow = -1
local gReferenceCook = false
local gSelectingItem = false

--
-- start a recipe (by sending an internal event)
--
local function start_recipe(recipe, starttime)
  gre.send_event_data("start_recipe", "4u1 starttime 1u1 reference_cook 1s0 recipe", {starttime = starttime, reference_cook = (gReferenceCook and 1 or 0), recipe = recipe})
end

--
-- set Edit_Btn hidden attr
--
local function edit_btn_hide(iHide)
  local data = {}
  data["REC_Layer.REC_Edit_Btn.grd_hidden"] = iHide
  gre.set_data(data)
end

--
-- update scrollbar slider location
--
function CBREC_SyncScrollbar()
  gScrollbar:scroll()
end

--
-- initialize recipe table
--
function REC_InitRecipesTable(bSortByName)
  -- sort recipes as desired (alphabetically or custom order)
  local sorted = Recipes:sorted_names(bSortByName)

  -- check if there's any recipes available
  if #sorted > 0 then
    -- insert recipes into the table
    local data = {}
    local count = 0
    for i, name in ipairs(sorted) do
      local recipe = Recipes:get(name)
      local hours, minutes = calculate_time(recipe.steps)
      local active = (ActiveRecipe:is_cooking() or ActiveRecipe:is_scheduled()) and ActiveRecipe.name == name
      data[TABLE_RECIPES..".name."..i..".1"] = name
      data[TABLE_RECIPES..".icon."..i..".1"] = recipe.icon and icon_path(recipe.icon) or ""
      data[TABLE_RECIPES..".time."..i..".1"] = string.format("%01d:%02d", hours, minutes)
      data[TABLE_RECIPES..".active_alpha."..i..".1"] = active and 255 or 0
      data[TABLE_RECIPES..".selected_bg_alpha."..i..".1"] = 0
      data[TABLE_RECIPES..".selected_start_alpha."..i..".1"] = 0
      data[TABLE_RECIPES..".selected_text_color."..i..".1"] = 0xFFFFFF
      data[TABLE_RECIPES..".calibrated_alpha."..i..".1"] = (gReferenceCook and recipe.calibrated) and 255 or 0
      data[TABLE_RECIPES..".insert_line_alpha."..i..".1"] = 0
      count = count + 1
    end
    data[TABLE_RECIPES..".grd_hidden"] = false
    gre.set_table_attrs(TABLE_RECIPES, {rows = count, yoffset = 0})
    gre.set_data(data)
    gScrollbar:init()
  else
    -- hide table and scrollbar
    local data = {}
    data[TABLE_RECIPES..".grd_hidden"] = true
    data[SCROLLBAR..".grd_hidden"] = true
    gre.set_data(data)
  end
end

--
-- ask user to confirm delayed recipe start if a delayed recipe start was chosen.
--
local function REC_AskConfirmRecipeStart(recipe_name)
  local timestamp = Recipe_GetEnteredStartTime()

  if timestamp > 0 then
    local _, _, _, h, min, _ = utc_time(timestamp - now())
    local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
    dlg:set_message(string.format(i18n:get("NLS_CONFIRM_RECIPE_DELAY"), h, min))
    dlg:add_button(i18n:get("NLS_CANCEL"), function() Recipe_SetEnteredStartTime(0) end)
    dlg:add_button(i18n:get("NLS_CONTINUE"), function() Event:can_start_recipe(recipe_name) end)
    dlg:show()
  else
    Event:can_start_recipe(recipe_name)
  end
end
--
-- switch to load selection overlay
--
local function REC_ShowLoadSelection()
  if Settings:get_def("ASK_WEIGHT", 0) == 1 and not gReferenceCook then
    History:force_push("REC_Screen")

    local name = gre.get_value(TABLE_RECIPES..".name."..gSelectedRow..".1")
    local icon = gre.get_value(TABLE_RECIPES..".icon."..gSelectedRow..".1")

    local data = {}
    data["REC_Screen.REC_Layer.grd_hidden"] = true
    data["REC_Screen.RECSIZE_Layer.grd_hidden"] = false
    data["RECSIZE_Layer.Recipe_Text.text"] = name
    data["RECSIZE_Layer.Recipe_Img.image"] = icon
    gre.set_data(data)
  else
    if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
      local dlg = DialogBox.new(DIALOG_TYPE_WARNING)
      dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
      dlg:add_button(i18n:get("NLS_OK"))
      dlg:show()
    else
      local name = gre.get_value(TABLE_RECIPES..".name."..gSelectedRow..".1")
      REC_AskConfirmRecipeStart(name)
    end
  end
end

--
-- deselect currently selected recipe
--
function CBREC_DeselectRecipe(mapargs)
  if gSelectedRow ~= -1 then
    local data = {}
    data[TABLE_RECIPES..".selected_bg_alpha."..gSelectedRow..".1"] = 0
    data[TABLE_RECIPES..".selected_start_alpha."..gSelectedRow..".1"] = 0
    data[TABLE_RECIPES..".selected_text_color."..gSelectedRow..".1"] = 0xFFFFFF
    data["REC_Layer.REC_Edit_Btn.image"] = "images/icon-nieuw-recept-maken-x429-y92.png"
    gSelectedRow = -1
    gre.set_data(data)
  end
end

--
-- recipe selected, see if the touch was within bounds of the start button
--
--- @param gre#context mapargs
function CBREC_OnRecipeSelected(mapargs)
  gSelectingItem = true

  local i = mapargs.context_row
  local x_pos = gre.get_table_cell_attrs(TABLE_RECIPES, i, 1, "x").x
        x_pos = x_pos + gre.get_value(TABLE_RECIPES..".start_x_pos")
        x_pos = x_pos + gre.get_value(TABLE_RECIPES..".grd_x")
  local name = gre.get_value(TABLE_RECIPES..".name."..i..".1")
  local active = (ActiveRecipe:is_cooking() or ActiveRecipe:is_scheduled()) and ActiveRecipe.name == name

  local data = {}

  if i == gSelectedRow then
    -- second press on same row
    REC_ShowLoadSelection()
  else
    -- new row selected
    if gSelectedRow ~= -1 then
      data[TABLE_RECIPES..".selected_bg_alpha."..gSelectedRow..".1"] = 0
      data[TABLE_RECIPES..".selected_start_alpha."..gSelectedRow..".1"] = 0
      data[TABLE_RECIPES..".selected_text_color."..gSelectedRow..".1"] = 0xFFFFFF
    end
    data[TABLE_RECIPES..".selected_bg_alpha."..i..".1"] = 255
    data[TABLE_RECIPES..".selected_start_alpha."..i..".1"] = not active and 255 or 0
    data[TABLE_RECIPES..".selected_text_color."..i..".1"] = 0x32467D
    gSelectedRow = i
  end

  if gSelectedRow == -1 then
    data["REC_Layer.REC_Edit_Btn.image"] = "images/icon-nieuw-recept-maken-x429-y92.png"
  else
    data["REC_Layer.REC_Edit_Btn.image"] = not active and "images/icon-nieuw-recept-zonder+_x429-y92.png" or ""
  end
  gre.set_data(data)
end

--
-- recipe no longer running, remove active icon
--
function CBREC_OnRecipeStopped(mapargs)
  local data = {}

  local count = gre.get_table_attrs(TABLE_RECIPES, "rows").rows
  for i=1,count do
    data[TABLE_RECIPES..".active_alpha."..i..".1"] = 0
    data[TABLE_RECIPES..".selected_start_alpha."..i..".1"] = i == gSelectedRow and 255 or 0
  end

  gre.set_data(data)
end

--
-- handle table touch to determine press outside element
--
function CBREC_OnTableTouch(mapargs)
  if not gSelectingItem then
    CBREC_DeselectRecipe(mapargs)
  end

  gSelectingItem = false
end

--
-- edit a recipe
--
function CBREC_OnEditRecipe(mapargs)
  if gSelectedRow == -1 then
    -- create new recipe
    gre.set_value("toScreen", "CR_Screen")
    gre.send_event("switch_screen")
    CBCR_NewRecipe()
  else
    -- edit recipe
    local name = gre.get_value(TABLE_RECIPES..".name."..gSelectedRow..".1")

    if ActiveRecipe:is_cooking() and (ActiveRecipe.name == name) then
      -- cannot edit recipes while this recipe is running
      --local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
      --dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
      --dlg:add_button(i18n:get("NLS_OK"))
      --dlg:show()
    else
      gre.send_event_data("edit_recipe", "1s0 recipe", {recipe = name})
    end
  end
end

--
-- request editing recipe
--
function CBREC_OnRequestEditRecipe(mapargs)
  -- check if currently active role is allowed to edit/create recipe
  if Settings:get_def("ENDUSER_EDIT", 1) == 0 and not RoleManager:can_access(MANAGER_ROLE) then
    LoginDialog_Init("REC_Screen", MANAGER_ROLE)
  else
    CBREC_OnEditRecipe(mapargs)
  end
end

--
-- disable recipes button
--
function CBREC_OnScreenShow()
  gSelectedRow = -1

  -- show "recipe calibration" icon if needed, or show "delay recipe start" icon if needed.
  local data = {}
  local delayed_recipe_enabled = Settings:get("DELAYED_RECIPE_START")
  data["REC_Layer.REC_ReferenceCook_Img.grd_hidden"] = not gReferenceCook
  data["REC_Layer.REC_Edit_Btn.grd_hidden"] = false --Settings:get("COMMISSION_TIME") == 0
  if delayed_recipe_enabled == 1 then
    -- show "delayed recipe start" icon in the place where the "recipe calibration" icon normally sits.
    data["REC_Screen.StartRecipe_Overlay.grd_hidden"] = gReferenceCook
    if ActiveRecipe:is_scheduled() then
      Recipe_SetEnteredStartTime(ActiveRecipe:scheduled_time())
    else
      Recipe_SetEnteredStartTime(0)
    end
    CBRecipe_StartTime_Prepare()
  else
    data["REC_Screen.StartRecipe_Overlay.grd_hidden"] = true
    Recipe_SetEnteredStartTime(0)
  end
  gre.set_data(data)

  Control_SetButtons(true, true, true, true, true, true, false)

  REC_InitRecipesTable(false)
end

--
-- open recipes screen for reference cook
--
function REC_SetReferenceCook()
  gReferenceCook = true
end

--
-- leaving recipes screen
--
function CBREC_OnScreenHide()
  gReferenceCook = false
end

--
-- start with full load
--
function CBREC_OnPlayFull()
  if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
    local dlg = DialogBox.new(DIALOG_TYPE_WARNING)
    dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  else
    local name = gre.get_value("RECSIZE_Layer.Recipe_Text.text")
    REC_AskConfirmRecipeStart(name)
  end
end

--
-- start with half load
--
function CBREC_OnPlayHalf()
  if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
    local dlg = DialogBox.new(DIALOG_TYPE_WARNING)
    dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  else
    local name = gre.get_value("RECSIZE_Layer.Recipe_Text.text")
    REC_AskConfirmRecipeStart(name)
  end
end

--
-- start with quarter load
--
function CBREC_OnPlayQuarter()
  if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
    local dlg = DialogBox.new(DIALOG_TYPE_WARNING)
    dlg:set_message(i18n:get("NLS_RECIPE_READONLY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  else
    local name = gre.get_value("RECSIZE_Layer.Recipe_Text.text")
    REC_AskConfirmRecipeStart(name)
  end
end

--
-- handle back button press (go to previous menu, or if in top stack to previous screen)
--
function CBREC_BackPress()
  local data = {}
  data["REC_Screen.REC_Layer.grd_hidden"] = false
  data["REC_Screen.RECSIZE_Layer.grd_hidden"] = true
  gre.set_data(data)
end

--
-- received reply from application if recipe can be started
--
function CBREC_OnCanStartReply(mapargs)
  local reply = mapargs.context_event_data.reply
  local info = split_string(mapargs.context_event_data.info, DELIMITER_TOKEN)
  local name = info[1]
  local message = #info > 1 and info[2] or nil
  local starttime = Recipe_GetEnteredStartTime()

  if reply == -1 then
    -- error
    local dlg = DialogBox.new(DIALOG_TYPE_ERROR)
    dlg:set_message(i18n:get(message))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  elseif reply == 0 then
    -- start recipe
    start_recipe(name, starttime)
  elseif reply == 1 then
    -- display information message, then start recipe
    local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
    dlg:set_message(i18n:get(message))
    dlg:set_image(dlg:get_image(message)) -- NLS_REMOVE_DETERGENT has an image
    dlg:add_button(i18n:get("NLS_CANCEL"))
    dlg:add_button(i18n:get("NLS_CONTINUE"), function() start_recipe(name, starttime) end)
    dlg:show()
  end
end

--
-- commissioning state changed, (dis)allow editing/creating of recipes
--
function CBREC_OnCommissioningChanged(mapargs)
  --if mapargs.context_event_data.id == "COMMISSION_TIME" then
  --  gre.set_value("REC_Layer.REC_Edit_Btn.grd_hidden", Settings:get("COMMISSION_TIME") == 0)
  --end
end
