-------------------------------------
-- local copy of the recipes database
-------------------------------------

require("Config")
require("events")

--[[ recipes database ]]--

-- recipes database singleton
Recipes = {}
Recipes.__index = Recipes

Recipes.db = {}           -- table to store the recipes in
Recipes.firstuse_db = {}  -- table to store first use recipes in

--
-- get the number of recipes in the database
--
function Recipes:getn(include_first_use)
  if include_first_use then
    return table_count(self.db) + table_count(self.firstuse_db)
  else
    return table_count(self.db)
  end
end

--
-- get a recipe from the database
--
function Recipes:get(name)
  return self.db[name] or self.firstuse_db[name]
end

--
-- get a first-use recipe from the database
--
function Recipes:first_use_recipe()
  for name, recipe in pairs(self.firstuse_db) do
    return name, recipe
  end
  return nil, nil
end

--
-- check if a recipe name is already in use
--
function Recipes:is_name_taken(name)
  return self.db[name] ~= nil or self.firstuse_db[name] ~= nil
end

--
-- delete a recipe from the database
--
function Recipes:delete(name)
  self.db[name] = nil
end

--
-- add a recipe to the database
--
function Recipes:put(name, recipe, first_use)
  if first_use then
    self.firstuse_db[name] = recipe
  else
    self.db[name] = recipe
  end
end

--
-- update a recipe in the database
--
function Recipes:update(name, newname, recipe)
  if name ~= newname then
    self.db[name] = nil
  end
  self.db[newname] = recipe
end

--
-- add a step to a recipe in the database
--
function Recipes:add_step(name, step, first_use)
  if first_use then
    table.insert(self.firstuse_db[name].steps, step)
  else
    table.insert(self.db[name].steps, step)
  end
end

--
-- sort recipes alphabetically
--
local function sort_by_name(db)
  local sorted_names = {}
  for name, _ in pairs(db) do
    table.insert(sorted_names, name)
  end
  table.sort(sorted_names)
  return sorted_names
end

--
-- sort recipes by sort index
--
local function sort_by_index(db)
  local sorted = {}
  for name, entry in pairs(db) do
    table.insert(sorted, {name = name, sort_index = entry.sort_index})
  end
  table.sort(sorted, function(a, b) return a.sort_index < b.sort_index end)

  local sorted_names = {}
  for _, entry in ipairs(sorted) do
    table.insert(sorted_names, entry.name)
  end
  return sorted_names
end

--
-- get a list with the names of the recipes in the database, sorted alphabetically or by sort index
--
function Recipes:sorted_names(bSortByName)
  if bSortByName then
    return sort_by_name(self.db)
  else
    return sort_by_index(self.db)
  end
end

--
-- update the sort order by providing a list of recipe names in the new sort order
--
function Recipes:update_sort_order(sorted_names)
  for index, name in ipairs(sorted_names) do
    self.db[name].sort_index = index
  end
  Event:update_recipe_sort_order(sorted_names)
end

--
-- check if any recipe in the database is using the icon
--
function Recipes:has_icon(icon)
  for _, recipe in pairs(self.db) do
    if recipe.icon == icon then
      return true
    end
  end
  return false
end

--[[ events to construct the recipes database ]]--

local gRecipesCount = -1            -- number of recipes to be constructed
local gActiveRecipe = nil           -- name of the recipe under construction
local gActiveRecipeIsFirstUse = nil -- is the recipe under construction a first-use recipe?
local gRecipeSteps = -1             -- total number of steps in the recipe under construction

--
-- event to initialize recipes database received (io_recipelist)
--
--- @param gre#context mapargs
function CB_InitRecipesDB(mapargs)
  gRecipesCount = mapargs.context_event_data.count
  gActiveRecipe = nil
  gActiveRecipeIsFirstUse = nil
  gRecipeSteps = -1
  
  if gRecipesCount == 0 then
    -- no recipes in the database, so we are done
    gre.send_event("recipes_init_done")
  end
end

--
-- event with recipe data received (io_recipe)
--
--- @param gre#context mapargs
function CB_OnRecipe(mapargs)
  assert(gRecipesCount ~= -1, "DB not init.")
  
  gRecipeSteps = mapargs.context_event_data.steps
  
  local first_use = mapargs.context_event_data.first_use == 1
  local sort_index = mapargs.context_event_data.sort_index
  local calibrated = mapargs.context_event_data.calibrated == 1

  -- data payload is a concatenation of the recipe name and icon
  local split = split_string(mapargs.context_event_data.data, DELIMITER_TOKEN)
  local name = split[1]
  local icon = split[2]

  gActiveRecipe = name
  gActiveRecipeIsFirstUse = first_use
  Recipes:put(gActiveRecipe, { icon = icon, sort_index = sort_index, calibrated = calibrated, steps = {} }, first_use)
end

--
-- event with steps of the recipe that is currently under construction( io_recipe_step)
--
--- @param gre#context mapargs
function CB_OnRecipeStep(mapargs)
  assert((gActiveRecipe ~= nil) and (gRecipeSteps ~= -1), "Recipe not init.")
  
  local data = mapargs.context_event_data
  Recipes:add_step(gActiveRecipe, {
    name    = data.name,                             -- step name
    temp    = data.temp / 10.0,                      -- step temperature (in °C)
    time    = data.time,                             -- step time (in minutes)
    preheat = data.params[1] ~= 0 and true or false, -- is this the preheating step? [0-1]
    hold    = data.params[2] ~= 0 and true or false, -- is this the holding step at the end of the recipe to keep the oven warm? [0-1]
    probe   = data.params[3],                        -- core probe enabled [0-1]
    fan     = data.params[4],                        -- fan speed [0-2] or [0-4] or [0-5]
    steam   = data.params[5],                        -- steam [0-3] off or *, **, ***
    vent    = data.params[6],                        -- vent/exhaust valve: 0 (closed), 1 (open), 5 (open for last 5 minutes)
    drying  = data.params[7] / 2,                    -- drying: 0-9.5 minutes in steps of 0.5, so [0-19]
    custom  = data.params[8] ~= 0 and true or false, -- 0 = default step name, 1 = custom step name
    notify  = data.params[9] ~= 0 and true or false, -- is this a notification step?
  }, gActiveRecipeIsFirstUse)
  
  -- check if we received all steps for the active recipe yet
  if gRecipeSteps == #Recipes:get(gActiveRecipe).steps then
    gRecipeSteps = -1
    
    -- check if we received all recipes yet
    if gRecipesCount == Recipes:getn(true) then
      gre.send_event("recipes_init_done")
    end
  end
end

--
-- event to refresh recipe database (io_refesh_recipe)
--
--- @param gre#context mapargs
function CB_OnRefreshRecipe(mapargs)
  Recipes.firstuse_db = {}
  Recipes.db = {}
  Event:get_recipes()
end
