--------------------------------------------------------------------------
-- Role manager implementation, determines access to manager functionality
--------------------------------------------------------------------------

require("Settings")

local MAX_ATTEMPTS = 3      -- maximum amount of elevation attempts before cooldown
local COOLDOWN_TIME = 30    -- cooldown time before next attempt after max attempts
local END_USER = "END_USER" -- end user role name
local MANAGER = "MANAGER"   -- manager role name
local SERVICE = "SERVICE"   -- service role name

END_USER_ROLE = END_USER
MANAGER_ROLE = MANAGER
SERVICE_ROLE = SERVICE

--[[ role manager class implementation ]]--

-- role manager singleton
RoleManager = {}
RoleManager.__index = RoleManager

RoleManager.role = END_USER -- default role, standard end user
RoleManager.attempts = 0    -- number of failed elevation attempts
RoleManager.cooldown = nil  -- cooldown timer when max elevation attempts are exceeded
RoleManager.countdown = 0   -- cooldown timer countdown

-- available roles
local gRoles = {
  [END_USER] = { [END_USER] = true                                     }, -- end user can only access end user roles
  [MANAGER ] = { [END_USER] = true, [MANAGER] = true                   }, -- manager can access both end user and manager roles
  [SERVICE ] = { [END_USER] = true, [MANAGER] = true, [SERVICE] = true }  -- service can access all roles
}

--
-- get active role
--
function RoleManager:get_role()
  return self.role
end

--
-- get list of roles that can be used
--
function RoleManager:get_roles()
  local roles = {}
  for role in gRoles[self.roles] do
    table.insert(roles, role)
  end
  return roles
end

--
-- check if we are allowed to perform tasks that require role
--
function RoleManager:can_access(role)
  return gRoles[self.role][role] or false
end

--
-- check if a pin code is required for a role
--
function RoleManager:is_pin_required(role)
  if role == END_USER then
    return false
  elseif role == MANAGER then
    return Settings:get("MANAGER_PIN") ~= 0
  else
    return true
  end
end

--
-- request to elevate to a new role
--
function RoleManager:request_elevation(role, pin)
  if self.cooldown then
    gre.send_event_data("elevation_cooldown", "2u1 wait", { wait = self.countdown })
  else
    Event:request_elevation(role, pin)
  end
end

--
-- let the application know we are actively excercising our rights, so it doesn't auto log off
--
function RoleManager:kick()
  Event:kick_rolemanager()
end

--
-- on elevation granted
--
function RoleManager:on_elevation_granted(role)
  self.role = role
  self.attempts = 0
  gre.send_event("elevation_granted")
end

--
-- on cooldown timer tick
--
local function cooldown_counter()
  if RoleManager.countdown == 0 then
    gre.timer_clear_interval(RoleManager.cooldown)
    RoleManager.cooldown = nil
  else
    RoleManager.countdown = RoleManager.countdown - 1
  end
end

--
-- on elevation denied
--
function RoleManager:on_elevation_denied()
  self.attempts = self.attempts + 1
  if self.attempts >= MAX_ATTEMPTS then
    self.countdown = COOLDOWN_TIME
    self.cooldown = gre.timer_set_interval(cooldown_counter, 1000)
    gre.send_event("elevation_attempts")
  else
    gre.send_event("elevation_denied")
  end
end

--[[ handle relevant events ]]--

--
-- on elevation granted event
--
--- @param gre#context mapargs
function CB_OnElevationGranted(mapargs)
  RoleManager:on_elevation_granted(mapargs.context_event_data.role)
end

--
-- on elevation denied event
--
function CB_OnElevationDenied()
  RoleManager:on_elevation_denied()
end
