-------------------------------------------
-- implementation of the screenshots screen
-------------------------------------------

require("Config")
require("Dialog")
require("events")
require("History")
require("Time")
require("Utf8")

--[[ Screenshot screen ]]--

local NOTE_MAX_CHARACTERS = 1000

local gScreenshot = nil
local gEditingNote = false
local gNoteText = ""
local gOldNoteText = ""

--
-- keyboard button pressed
--
function CBSCRN_InputKeyEvent(mapargs)
  if not gEditingNote then return end

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    gNoteText = Utf8_RemoveLastCharacter(gNoteText)
  else
    -- append char
    local tmp = gNoteText..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= NOTE_MAX_CHARACTERS then
      gNoteText = tmp
    end
  end

  local data = {}
  data["RemarkText_Layer.note_text.text"] = gNoteText.."|"
  data["Screenshot_Layer_v2.Remark_Ctrl.characters_used"] = string.format("%d/%d", #gNoteText, NOTE_MAX_CHARACTERS)
  gre.set_data(data)
end

--
-- finish editing note
--
function CBSCRN_OnKeyboardConfirm()
  gEditingNote = false

  gre.set_value("RemarkText_Layer.note_text.text", gNoteText)
  gre.set_layer_attrs("RemarkText_Layer", { yoffset = 0 })
end

--
-- stop editing note
--
function CBSCRN_OnKeyboardCancel()
  gEditingNote = false

  gNoteText = gOldNoteText
  gre.set_value("RemarkText_Layer.note_text.text", gNoteText)
  gre.set_layer_attrs("RemarkText_Layer", { yoffset = 0 })
end

--
-- begin editing the note
--
function CBSCRN_StartEditNote()
  if not gEditingNote then
    gEditingNote = true

    gNoteText = gre.get_value("RemarkText_Layer.note_text.text")
    gOldNoteText = gNoteText

    gre.set_value("RemarkText_Layer.note_text.text", "|")
    gre.set_layer_attrs("Keyboard_Layer_v2", { hidden = false })
  end
end

--
-- send the screenshot and optional note to service
--
function CBSCRN_SendToService()
  Event:send_screenshot(gNoteText)
  History:pop()
end

--
-- prepare the screenshot screen for display
--
function CBSCRN_Prepare()
  gEditingNote = false
  gNoteText = ""
  gOldNoteText = ""

  local time = local_time()
  local date_format = format_date(time)
  local time_format, ampm = format_time(time)

  local data = {}
  data["RemarkText_Layer.note_text.text"] = gNoteText
  data["Screenshot_Layer_v2.Remark_Ctrl.characters_used"] = string.format("%d/%d", 0, NOTE_MAX_CHARACTERS)
  data["Screenshot_Layer_v2.Preview_Ctrl.image"] = gScreenshot
  data["Screenshot_Layer_v2.Timestamp_Text.text"] = date_format.." "..time_format..(ampm or "")
  gre.set_data(data)

  gre.set_layer_attrs("SCRN_Screen.RemarkText_Layer", { yoffset = 0 })

  Control_SetButtons(true, true, true, true, true, true, false)
end

--[[ Make screenshot dialog ]]--

--
-- initiate making a screenshot
--
function CBSCRN_MakeScreenshot()
  gre.set_layer_attrs("TakeScreenshot_Layer", { hidden = true })
  gre.set_layer_attrs("Loading_Layer", { hidden = false }) -- change from false to true prevent hourglass in screenshot!

  gre.timer_set_timeout(function() Event:make_screenshot() end, DISPLAY_UPDATE_DELAY_MS)
end

--
-- initiate making a screenshot from the application
--
function CBSCRN_ForceScreenshot()
  Event:make_screenshot()
end

--[[ Taking a screenshot ]]--

--
-- taking screenshot finished (success or failure)
--
--- @param gre#context mapargs
function CBSCRN_OnScreenShotreply(mapargs)
  -- remove screenshot from cache because screenshot always uses the same name,
  -- if it is not removed an old screenshot will be shown when a new one is taken
  if gScreenshot ~= nil then gre.dump_resource("image", gScreenshot) end

  gre.set_layer_attrs("Loading_Layer", { hidden = true })

  if mapargs.context_event_data.ok == 1 then
    gScreenshot = mapargs.context_event_data.file
    if mapargs.context_screen == "SCRN_Screen" then
      CBSCRN_Prepare()
    else
      gre.send_event("screenshot_screen")
    end
  else
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get("NLS_SCREENSHOT_FAILED"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  end
end

--
-- sending screenshot to service finished (success or failure)
--
--- @param gre#context mapargs
function CNSCRN_OnScreenShotSendReply(mapargs)
  if mapargs.context_event_data.ok == 0 then
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get("NLS_SCREENSHOT_SEND_FAILED"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  end
end
