--------------------------------------------------------------
-- progress dialog for async actions (currenty load/store USB)
--------------------------------------------------------------

require("Dialog")
require("events")
require("language")

--[[ action progress dialog ]]--

local ACTION_PROGRESS_STATE_CONFIRM = 0
local ACTION_PROGRESS_STATE_START   = 1
local ACTION_PROGRESS_STATE_BUSY    = 2
local ACTION_PROGRESS_STATE_DONE    = 3
local ACTION_PROGRESS_STATE_ERROR   = 4

local gActionName = nil
local gConfirmAction = nil
local gParameterValue = nil
local gActionProgressState = nil

--
-- populate action progress dialog controls
--
local function ActionProgress_Populate(title_nls_id, message_nls_id, next_nls_id, show_back, show_next)
  local data = {}
  data["ActionProgress_Dialog.title_text.text"] = i18n:get(title_nls_id)
  data["ActionProgress_Dialog.value_ctrl.text"] = gParameterValue
  data["ActionProgress_Dialog.value_ctrl.grd_hidden"] = message_nls_id ~= nil
  data["ActionProgress_Dialog.message_ctrl.text"] = (message_nls_id ~= nil) and i18n:get(message_nls_id) or ""
  data["ActionProgress_Dialog.message_ctrl.grd_hidden"] = message_nls_id == nil
  data["ActionProgress_Dialog.arrow_next.text"] = i18n:get(next_nls_id)
  data["ActionProgress_Dialog.arrow_back.grd_hidden"] = not show_back
  data["ActionProgress_Dialog.arrow_next.alpha"] = show_next and 255 or 0
  gre.set_data(data)
end

--
-- progress title depends for the action_id-prefix 
--
local function ActionProgress_ProgressTitle(action_id)
    local title_nls_id
    if starts_with(action_id, "SAVE") then
      title_nls_id  = "NLS_SAVING"
    else
      title_nls_id  = "NLS_LOADING"
    end 
    return title_nls_id
end

--
-- show message page (confirm or error)
--
local function ActionProgress_ShowMessage(message_nls_id)
  if gActionProgressState == ACTION_PROGRESS_STATE_CONFIRM and message_nls_id == "NLS_FS_ERR_DUPLICATED" then
    ActionProgress_Populate(gActionName, gActionName..".CONFIRM", "NLS_START", true, true)
    gActionProgressState = ACTION_PROGRESS_STATE_START
  else
    ActionProgress_Populate("NLS_LOADING", message_nls_id, "NLS_DONE", false, true)
    gActionProgressState = ACTION_PROGRESS_STATE_DONE
  end
end


--
-- show done page (ready)
--
local function ActionProgress_ShowDone(action_id)
  if gActionProgressState == ACTION_PROGRESS_STATE_CONFIRM then
    Event:action(gActionName, gParameterValue)
    gActionProgressState = ACTION_PROGRESS_STATE_BUSY
  else
    ActionProgress_Populate(ActionProgress_ProgressTitle(action_id), nil, "NLS_DONE", false, true)
    gActionProgressState = ACTION_PROGRESS_STATE_DONE
  end
end


--
-- show busy page (loading/saving)
--
local function ActionProgress_ShowBusy(action_id)
  if gActionProgressState ~= ACTION_PROGRESS_STATE_CONFIRM then
       
    ActionProgress_Populate(ActionProgress_ProgressTitle(action_id), nil, "NLS_BUSY", false, false)
    gActionProgressState = ACTION_PROGRESS_STATE_BUSY
  end
end

--
-- show start page (start)
--
local function ActionProgress_ShowStart()  
  ActionProgress_Populate(gActionName, nil, "NLS_START", true, true)
  gActionProgressState = (gConfirmAction == nil) and ACTION_PROGRESS_STATE_START or ACTION_PROGRESS_STATE_CONFIRM
end

--
-- start/ready button pressed
--
function CBSET_ActionProgress_OnButtonPress()
  if gActionProgressState == ACTION_PROGRESS_STATE_CONFIRM then
    -- execute the confirmation action
    Event:action(gConfirmAction, gParameterValue)
  elseif gActionProgressState == ACTION_PROGRESS_STATE_START then
    -- execute the action
    Event:action(gActionName, gParameterValue)
  elseif gActionProgressState == ACTION_PROGRESS_STATE_DONE then
    -- finished, close dialog
    gre.set_layer_attrs("ActionProgress_Dialog", { hidden = true })
  end
end

--
-- show the progress dialog
--
function CBSET_ActionProgress_OnShowOverlay(action_name, confirm_action, parameter_value)
  gActionName = action_name
  gConfirmAction = confirm_action
  gParameterValue = parameter_value

  ActionProgress_ShowStart()
  
  gre.set_layer_attrs("ActionProgress_Dialog", { hidden = false })
end

--[[ action progress events ]]--

local gSupportedParameters = {
  "EXIST_HACCP_USB",
  "EXIST_PARAM_USB",
  "EXIST_RECIPE_USB",
  "SAVE_HACCP_USB",  
  "SAVE_PARAM_USB",
  "SAVE_RECIPE_USB",
  "LOAD_PARAM_USB",
  "LOAD_RECIPE_USB",
  "UPDATE_RECIPE_USB"
}

--
-- check if action progress dialog or standard text dialog should be shown
--
local function ActionProgress_DialogSupported(action_id)
  return in_array(action_id, gSupportedParameters)
--     and gre.get_layer_attrs("ActionProgress_Dialog", "hidden").hidden == false
end

--
-- show a text based dialog
--
local function ActionProgress_ShowTextDialog(type, text_nls_id)
  local dlg = DialogBox.new(type)  
  dlg:set_message(i18n:get(text_nls_id))
  dlg:add_button(i18n:get("NLS_OK"))
  dlg:show()
end

--
-- show error dialog
--
local function ActionProgress_OnStateError(action_id, message_nls_id)
  if ActionProgress_DialogSupported(action_id) then
    ActionProgress_ShowMessage(message_nls_id)
  else
    gre.set_layer_attrs("ActionProgress_Dialog", { hidden = true })
    ActionProgress_ShowTextDialog(DIALOG_TYPE_ERROR, message_nls_id or "NLS_ACTION_NOT_SCHED")
  end
end

--
-- show action completed
--
local function ActionProgress_OnStateDone(action_id, message_nls_id)
  if ActionProgress_DialogSupported(action_id) then
    ActionProgress_ShowDone(action_id)
  else
    ActionProgress_ShowTextDialog(DIALOG_TYPE_INFORMATION, message_nls_id or "NLS_ACTION_COMPLETED")
  end
end

--
-- show action is busy
--
local function ActionProgress_OnStateBusy(action_id, message_nls_id)
  if ActionProgress_DialogSupported(action_id) then
    ActionProgress_ShowBusy(action_id)
  else
    ActionProgress_ShowTextDialog(DIALOG_TYPE_INFORMATION, message_nls_id or "NLS_ACTION_SCHED")
  end
end

--
-- io_action_state event received
--
-- 1u1 state = { [0] = scheduled, [1] = ready, [2] = error }
-- 1s0 data = action parameter id + optional message NLS id
--
function CBSET_ActionProgress_OnState(mapargs)
  local state_cb_map = {
    [0] = ActionProgress_OnStateBusy,
    [1] = ActionProgress_OnStateDone,
    [2] = ActionProgress_OnStateError
  }
  local callback = state_cb_map[mapargs.context_event_data.state]
  if callback ~= nil then
    local reply = split_string(mapargs.context_event_data.data, DELIMITER_TOKEN)
    callback(reply[1], reply[2])
  end
end
