-----------------------------------------------------------------------
-- implementation of the settings screen which is in the form of a menu
-- tree with action leaves with control layers to modify parameters
-----------------------------------------------------------------------

require("bit32")
require("Dialog")
require("Settings")
require("Scrollbar")
require("Utf8")

local TABLE_SETTINGS = "SET_Menu_Layer.SET_Settings_Table"     -- settings table
local SCROLLBAR = "SET_Menu_Layer.Scrollbar"                   -- settings table scrollbar
local LANGUAGE_TABLE = "SET_Language_Layer.SET_Language_Table" -- language selection table
local LANGUAGE_SCROLLBAR = "SET_Language_Layer.Scrollbar"      -- language selection table scrollbar

gCurrentMenu = nil                                             -- currently active/displayed (sub)menu
gLastRow = nil                                                 -- id of currently active menu
gYOffsetStack = {}                                             -- stack with yoffset
gOldYOffset = 0                                                -- remember scroll position

local gScrollbar = Scrollbar.new{table=TABLE_SETTINGS,scrollbar=SCROLLBAR}              -- settings menu scrollbar

-- list of settings submenu types
local gSettingsLayers = {
  "SET_Menu_Layer",        -- submenu
  "Language_Layer",        -- language selection
  "SET_Toggle_Layer",      -- toggle control
  "SET_Pin_Layer",         -- pincode
  "SET_Integer_Layer",     -- integer
  "SET_Slider_Layer",      -- number in range
  "SET_MultiToggle_Layer", -- toggle with many options
  "SET_Text_Layer",        -- free text input
  "SET_BitTable_Layer"     -- bitmap
}

--
-- synchronize the scrollbar with the settings menu position
--
function CBSET_SyncScrollbar()
  gScrollbar:scroll()
end

--
-- initialize the settings screen
--
--- @param gre#context mapargs
function CBSET_PrepareScreen(mapargs)
  Control_SetButtons(true, true, true, true, true, true, false)
  gFromScreen = gre.get_value("prevScreen")
  gMenuStack = {}
  gYOffsetStack = {}
  gOldYOffset = 0
  PopulateMenu()
  LiveVars_CurMenu_Enter()
end

--
-- exit the settings screen
--
function CBSET_ExitScreen()
  Event:iotest_end()
end

--
-- force refresh the settings screen
--
function CBSET_ForceRefresh()
  History:push("SET_Screen")
  gMenuStack = {}
  gYOffsetStack = {}
  gOldYOffset = 0
  PopulateMenu()
  LiveVars_CurMenu_Enter()
end

--
-- display the (sub)menu or a parameter edit page
--
function CBSET_ToggleMenu()
  local activelayer = gre.get_value("SET_Screen.activeLayer")

  for _, layer in pairs(gSettingsLayers) do
    local hidden = true
    if layer == activelayer then
      hidden = false
    end
    gre.set_layer_attrs(layer, { hidden = hidden })
  end

  if activelayer == "SET_Menu_Layer" then
    gScrollbar:init()
  end
end

--
-- menu item pressed event
--
--- @param gre#context mapargs
function CBSET_MenuPress(mapargs)
  -- notify application that we are excercising our rights
  RoleManager:kick()

  -- store scroll position
  local scroll_position = gre.get_table_attrs(TABLE_SETTINGS, "yoffset").yoffset

  -- perform action associated with menu entry (submenu, parameter, action)
  gLastRow = gre.get_value(TABLE_SETTINGS..".index."..mapargs.context_row..".1")
  local entry = gCurrentMenu.items and gCurrentMenu.items[gLastRow] or nil
  if not entry then
    print("Missing menu entry: ["..gLastRow.."]")
    return
  end
  local entry_lambda = function()
    if entry.type == "menu" then
      if entry.key == "IO_TEST_MENU" and SettingsAreReadOnly() then return end
      table.insert(gMenuStack, entry)
    end
    table.insert(gYOffsetStack, scroll_position)
    entry.action()
    LiveVars_CurMenu_Enter()
  end
  if not entry.role or RoleManager:can_access(entry.role) then
    entry_lambda()
  else
    LoginDialog_Init("SET_Screen", entry.role, entry_lambda)
  end
end

--
-- shade and icon are partially covering the menu table, forward touch events in this overlapping area to the table
--
--- @param gre#context mapargs
function CBSET_ForwardOverlayTouch(mapargs)
  forward_touch_to_table(mapargs, TABLE_SETTINGS, CBSET_MenuPress)
end

--
-- handle back button press (go to previous menu, or if in top stack to previous screen)
--
function CBSET_BackPress()
  if #gMenuStack > 1 then
    LiveVars_CurMenu_Leave()
    table.remove(gMenuStack, #gMenuStack)
    gOldYOffset = table.remove(gYOffsetStack) or 0
    PopulateMenu(true)
    LiveVars_CurMenu_Enter()
  end
end

--
-- save a parameter
--
-- parameter: the settings menu parameter object
-- newvalue : new value for the parameter
-- issetting: is parameter an element of the settings table
--
local function save_parameter(parameter, newvalue)
  -- update local
  Settings:set(parameter.param, newvalue)

  -- update remote
  if parameter.isLive then
    Event:live_set(parameter.param, newvalue)
  else
    Event:change_parameter(parameter.param, newvalue)
  end
end

--
-- save button pressed on parameter toggle screen
--
function CBSET_OnSaveParameter()
  gLastValue = Settings:get(gCurrentMenu.items[gLastRow].param)
  gNewValue = gre.get_value("SET_Toggle_Layer.toggled")

  if gNewValue ~= gLastValue then
    save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
  end

  CBSET_BackPress()
end

--
-- left toggle label pressed -> select left option
--
function CBSET_ToggleLeft()
  local data = {}
  data["SET_Toggle_Layer.toggled"   ] = 0
  data["SET_Toggle_Layer.Swap_Btn.x"] = 3
  gre.set_data(data)
end

--
-- right toggle label pressed -> select right option
--
function CBSET_ToggleRight()
  local data = {}
  data["SET_Toggle_Layer.toggled"   ] = 1
  data["SET_Toggle_Layer.Swap_Btn.x"] = 57
  gre.set_data(data)
end

--
-- swap toggle when it's pressed
--
--- @param gre#context mapargs
function CBSET_TogglePress(mapargs)
  local data = {}

  local toggled = gre.get_value("SET_Toggle_Layer.toggled")
  toggled = 1 - toggled
  data["SET_Toggle_Layer.toggled"] = toggled

  if toggled == 0 then
    -- move button to the left
    data["SET_Toggle_Layer.Swap_Btn.x"] = 3
  else
    -- move button to the right
    data["SET_Toggle_Layer.Swap_Btn.x"] = 57
  end

  gre.set_data(data)
end

--
-- shade and icon are partially covering the bit field table, forward touch events in this overlapping area to the table
--
--- @param gre#context mapargs
function CBSET_BitTableForwardOverlayTouch(mapargs)
  forward_touch_to_table(mapargs, "SET_BitTable_Layer.Bit_Table", CBSET_OnBitTablePress)
end

--
-- save button pressed on pincode edit screen
--
function CBSET_OnSavePincode()
  if string.len(gLastValue) == 4 then
    gNewValue = tonumber(gLastValue)
    save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
    CBSET_BackPress()
  else
    gre.set_value("SET_Pin_Layer.feedback_text.text", i18n:get("NLS_PINCODE_LENGTH"))
  end
end

--
-- use keyboard to edit pincode
--
--- @param gre#context mapargs
local function Pin_InputKeyEvent(mapargs)
  local data = {}
  local len = string.len(gLastValue)

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    gLastValue = string.sub(gLastValue, 1, #gLastValue-1)
    data["SET_Pin_Layer.pincode_group.pincode_text_1.text"] = len > 1 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_2.text"] = len > 2 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_3.text"] = len > 3 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_4.text"] = ""
  elseif mapargs.context_event_data.code >= string.byte("0") and mapargs.context_event_data.code <= string.byte("9") and len < 4 then
    -- numeric key pressed
    local digit = string.char(mapargs.context_event_data.code)
    gLastValue = gLastValue..digit
    data["SET_Pin_Layer.pincode_group.pincode_text_1.text"] = len >= 0 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_2.text"] = len >= 1 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_3.text"] = len >= 2 and "X" or ""
    data["SET_Pin_Layer.pincode_group.pincode_text_4.text"] = len >= 3 and "X" or ""
  end

  gre.set_data(data)
end

--
-- save integer parameter after updating
--
function CBSET_OnSaveInteger()
  gre.animation_trigger("hide_numkeyb")

  gLastValue = Settings:get(gCurrentMenu.items[gLastRow].param)
  if gCurrentMenu.items[gLastRow].convert then gLastValue = gCurrentMenu.items[gLastRow].convert(gLastValue, true) end

  local text_value = gre.get_value("SET_Integer_Layer.Integer_Group.Number_Text.text")
  gNewValue = (text_value ~= "" and text_value ~= INFINITY_SYMBOL) and tonumber(text_value) or 0

  if gNewValue ~= gLastValue then
    if gCurrentMenu.items[gLastRow].convert then gNewValue = gCurrentMenu.items[gLastRow].convert(gNewValue, false) end
    gNewValue = math.max(gCurrentMenu.items[gLastRow].min, math.min(gNewValue, gCurrentMenu.items[gLastRow].max))
    save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
  end

  CBSET_BackPress()
end

--
-- increment or decrement an
--
local function IncrementDecrementInteger(increment, zero_is_infinity)
  local min = gCurrentMenu.items[gLastRow].min or 0
  local max = gCurrentMenu.items[gLastRow].max or 65535

  if gCurrentMenu.items[gLastRow].convert then
    min = gCurrentMenu.items[gLastRow].convert(min, true)
    max = gCurrentMenu.items[gLastRow].convert(max, true)
  end

  local text_value = gre.get_value("SET_Integer_Layer.Integer_Group.Number_Text.text")
  local value = (text_value ~= "" and text_value ~= INFINITY_SYMBOL) and tonumber(text_value) or 0
  local new_value = value + increment

  if zero_is_infinity == 1 and value == 0 then
    new_value = increment < 0 and max or (min + increment)
  elseif new_value < min then
    new_value = max
  elseif new_value > max then
    new_value = min
  end

  text_value = (new_value == 0 and zero_is_infinity == 1) and INFINITY_SYMBOL or string.format("%d", new_value)
  gre.set_value("SET_Integer_Layer.Integer_Group.Number_Text.text", text_value)
end

--
-- decrement an integer parameter
--
function CBSET_DecrementInteger(mapargs)
  IncrementDecrementInteger(-1 * mapargs.step_size, mapargs.zero_is_infinity)
end

--
-- increment an integer parameter
--
function CBSET_IncrementInteger(mapargs)
  IncrementDecrementInteger(mapargs.step_size, mapargs.zero_is_infinity)
end

--
-- use keyboard to edit integer
--
--- @param gre#context mapargs
local function Int_InputKeyEvent(mapargs)
  local value = gre.get_value("SET_Integer_Layer.Integer_Group.Number_Text.text")

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    value = string.sub(value, 1, #value-1)
  elseif mapargs.context_event_data.code >= string.byte("0") and mapargs.context_event_data.code <= string.byte("9") then
    -- numeric key pressed
    value = value..string.char(mapargs.context_event_data.code)
  end

  gre.set_value("SET_Integer_Layer.Integer_Group.Number_Text.text", value ~= "" and math.min(tonumber(value), gCurrentMenu.items[gLastRow].max or 65535) or "")
end

--
-- save slider parameter after updating
--
function CBSET_OnSaveSlider()
  gLastValue = Settings:get(gCurrentMenu.items[gLastRow].param)
  if gCurrentMenu.items[gLastRow].convert then gLastValue = gCurrentMenu.items[gLastRow].convert(gLastValue, true) end
  gNewValue = tonumber(gre.get_value("SET_Slider_Layer.Integer_Group.Number_Text.text"))

  if gNewValue ~= gLastValue then
    if gCurrentMenu.items[gLastRow].convert then gNewValue = gCurrentMenu.items[gLastRow].convert(gNewValue, false) end
    save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
  end

  CBSET_BackPress()
end

--
-- dragging around slider
--
--- @param gre#context mapargs
function CBSET_OnSlider(mapargs)
  local min_val = gCurrentMenu.items[gLastRow].min
  local max_val = gCurrentMenu.items[gLastRow].max

  if gCurrentMenu.items[gLastRow].convert then
    min_val = gCurrentMenu.items[gLastRow].convert(min_val, true)
    max_val = gCurrentMenu.items[gLastRow].convert(max_val, true)
  end

  local pressed = gre.get_value(mapargs.context_control..".slider_active") == 1

  if pressed then
    local min = 25
    local width = gre.get_value(mapargs.context_control..".grd_width") - 58
    local max = min + width
    local icon_center = 25

    local x = mapargs.context_event_data.x

    if x >= min and x <= max then
      local percentage = x > max and 1 or (x >= min and (x - min) / width or 0)
      local position = min + (percentage * width) - icon_center
      local value = math.floor(min_val + (percentage * (max_val - min_val)) + 0.5)

      gre.set_value(mapargs.context_control..".x", position)
      gre.set_value("SET_Slider_Layer.Integer_Group.Number_Text.text", string.format("%d", value))
    end
  end
end

--
-- slider label pressed
--
--- @param gre#context mapargs
function CBSET_OnSliderLabel(mapargs)
  local slider_info = gre.get_control_attrs("SET_Slider_Layer.Slider_Group.Slider_Bg", "x")
  local label_info = gre.get_control_attrs(mapargs.context_control, "x", "width")
  local label_center = label_info.x + (label_info.width / 2) - slider_info.x

  mapargs.context_control = "SET_Slider_Layer.Slider_Group.Slider_Btn"
  mapargs.context_event_data.x = math.floor(label_center)

  gre.set_value("SET_Slider_Layer.Slider_Group.Slider_Btn.slider_active", 1)
  CBSET_OnSlider(mapargs)
  gre.set_value("SET_Slider_Layer.Slider_Group.Slider_Btn.slider_active", 0)
end

--
-- save multitoggle parameter after updating
--
function CBSET_OnSaveMultiToggle()
  local item = gCurrentMenu.items[gLastRow]
  gLastValue = Settings:get(item.param)

  local index = gre.get_value("SET_MultiToggle_Layer.Slider_Ctrl.active_option")
  local label = (index == 1) and "Item_Label" or ("Item_Label_"..index)
  local active_item = "SET_MultiToggle_Layer.SliderOptions_Group."..label

  if item.type == "MULTITOGGLE" then
    gNewValue = gre.get_value(active_item..".text")
    if gNewValue ~= gLastValue then
      local parameter = gCurrentMenu.items[gLastRow].param
      Settings:set(parameter, gNewValue)
      Event:change_parameter_string(parameter, gNewValue)
    end
  else
    gNewValue = gre.get_value(active_item..".options_index")
    if gNewValue ~= gLastValue then
      save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
    end
  end

  CBSET_BackPress()
end

--
-- select multitoggle item by touching the item label
--
--- @param gre#context mapargs
function CBSET_OnMultiToggleLabelTouch(mapargs)
  local oldoption = gre.get_value("SET_MultiToggle_Layer.Slider_Ctrl.active_option")
  local option = gre.get_value(mapargs.context_control..".id")

  if option ~= oldoption then
    local option_height = 34
    local inter_option_height = 12
    local oldkey = "SET_MultiToggle_Layer.SliderOptions_Group.Item_Label" .. (oldoption > 1 and "_"..oldoption or "")
    local newkey = "SET_MultiToggle_Layer.SliderOptions_Group.Item_Label" .. (option > 1 and "_"..option or "")

    local data = {}
    data["SET_MultiToggle_Layer.Slider_Ctrl.active_option"] = option
    data["SET_MultiToggle_Layer.Slider_Ctrl.slider_y"] = 2 + ((option - 1) * (option_height + inter_option_height))
    data[oldkey..".color"] = 0xFFFFFF
    data[newkey..".color"] = 0xB1D5CF
    gre.set_data(data)
  end
end

--
-- select multitoggle item by touching the toggle bar
--
--- @param gre#context mapargs
function CBSET_OnMultiToggleTouch(mapargs)
  local y = mapargs.context_event_data.y

  local numoptions = #gCurrentMenu.items[gLastRow].options
  local option_height = 34
  local inter_option_height = 12
  local ypos = gre.get_value("SET_MultiToggle_Layer.Slider_Ctrl.grd_y")

  local option = 0
  for i = 1, numoptions do
    local option_y1 = ypos + ((i-1) * (option_height + inter_option_height))
    local option_y2 = option_y1 + option_height
    if y >= option_y1 and y <= option_y2 then option = i break end
  end

  local oldoption = gre.get_value("SET_MultiToggle_Layer.Slider_Ctrl.active_option")
  if option > 0 and option ~= oldoption then
    local oldkey = "SET_MultiToggle_Layer.SliderOptions_Group.Item_Label" .. (oldoption > 1 and "_"..oldoption or "")
    local newkey = "SET_MultiToggle_Layer.SliderOptions_Group.Item_Label" .. (option > 1 and "_"..option or "")

    local data = {}
    data["SET_MultiToggle_Layer.Slider_Ctrl.active_option"] = option
    data["SET_MultiToggle_Layer.Slider_Ctrl.slider_y"] = 2 + ((option - 1) * (option_height + inter_option_height))
    data[oldkey..".color"] = 0xFFFFFF
    data[newkey..".color"] = 0xB1D5CF
    gre.set_data(data)
  end
end

--
-- save text parameter after updating
--
function CBSET_OnSaveText()
  gLastValue = Settings:get(gCurrentMenu.items[gLastRow].param)
  gNewValue = gre.get_value("SET_Text_Layer.Text_Ctrl.text")

  if gNewValue ~= gLastValue then
    local parameter = gCurrentMenu.items[gLastRow].param
    Settings:set(parameter, gNewValue)
    Event:change_parameter_string(parameter, gNewValue)
  end

  CBSET_BackPress()
end

--
-- display keyboard to edit text
--
function CBSET_OnTextEdit()
  gre.set_layer_attrs("Keyboard_Layer_v2", {hidden=false})
end

--
-- use keyboard to edit text
--
--- @param gre#context mapargs
local function Text_InputKeyEvent(mapargs)
  local text = gre.get_value("SET_Text_Layer.Text_Ctrl.text")

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    text = Utf8_RemoveLastCharacter(text)
  else
    -- character, append to text
    local tmp = text..Utf8_FromUcs2(mapargs.context_event_data.code)
    if Utf8_StrLen(tmp) <= gCurrentMenu.items[gLastRow].max then
      text = tmp
    end
  end

  gre.set_value("SET_Text_Layer.Text_Ctrl.text", text)
end

--
-- toggle bit table entry
--
function CBSET_OnBitTablePress(mapargs)
  local entry = gCurrentMenu.items[gLastRow]
  if entry.isReadOnly then return end

  local index = mapargs.context_row
  local bit_index = gre.get_value("SET_BitTable_Layer.Bit_Table.index."..index..".1")
  local new_value = 1 - bit32.extract(gNewValue, bit_index)
  gNewValue = bit32.replace(gNewValue, new_value, bit_index)
  gre.set_value("SET_BitTable_Layer.Bit_Table.toggle_pos."..index..".1", new_value == 1 and 409 or 352)

  if entry.isLive then Event:live_set(entry.param, gNewValue) end
end

--
-- save bit table
--
function CBSET_OnSaveBitTable()
  local entry = gCurrentMenu.items[gLastRow]
  gLastValue = Settings:get(entry.param) or 0

  if not entry.isLive and not entry.isReadOnly and gNewValue ~= gLastValue then
    save_parameter(gCurrentMenu.items[gLastRow], gNewValue)
  end

  CBSET_BackPress()
end

--
-- keyboard event
--
--- @param gre#context mapargs
function CBSET_InputKeyEvent(mapargs)
  if ENTERFILENAME_DIALOG_ACTIVE then
    CBSET_EnterFileName_InputKeyEvent(mapargs)
  else
    local keyboard_map = {
      ["pincode"] = Pin_InputKeyEvent,
      ["integer"] = Int_InputKeyEvent,
      ["text"   ] = Text_InputKeyEvent
    }

    local keyboard_handler = keyboard_map[gMenuStack[#gMenuStack]]
    if keyboard_handler then keyboard_handler(mapargs) end
  end
end

--
-- keyboard cancel button
--
function CBSET_OnKeyboardCancel(mapargs)
  if ENTERFILENAME_DIALOG_ACTIVE then
    CBSET_EnterFileName_OnKeyboardCancel(mapargs)
  else
    CBSET_BackPress()
  end
end

--
-- keyboard ok button
--
function CBSET_OnKeyboardConfirm(mapargs)
  if ENTERFILENAME_DIALOG_ACTIVE then
    CBSET_EnterFileName_OnKeyboardConfirm(mapargs)
  else
    local keyboard_map = {
      ["pincode"] = CBSET_OnSavePincode,
      ["integer"] = CBSET_OnSaveInteger,
      ["text"   ] = CBSET_OnSaveText
    }

    local keyboard_handler = keyboard_map[gMenuStack[#gMenuStack]]
    if keyboard_handler then keyboard_handler(mapargs) end
  end
end

--
-- handle livevar_changed event
--
--- @param gre#context mapargs
function CBSET_OnLiveVar_Changed(mapargs)
  local id = mapargs.context_event_data.id

  if id == "CLEANING_SUPPORTED" then
    gre.send_event("refresh_settings_menu")
  else
    RefreshSubMenuItem(id)
  end
end

--
-- handle io_refresh_parameter event
--
function CBSET_OnRefreshParameters()
  RefreshMenu()
end
