-----------------------------------------
-- implementation of a reusable scrollbar
-----------------------------------------

--[[ TABLE SCROLLBAR ]]--

-- scrollbar class
Scrollbar = {}
Scrollbar.__index = Scrollbar

-- Scrollbar.table = "{$screen:table_control}"
-- Scrollbar.scrollbar = "{$screen:scrollbar_layer}"

--
-- create a new scrollbar instance (required table id and scrollbar id)
--
function Scrollbar.new(o)
  assert(o.table and o.scrollbar, "table and scrollbar ids required")
  
  o = o or {}
  setmetatable(o, Scrollbar)
  return o
end

--
-- initialize the scrollbar (calculate slider size/position)
--
function Scrollbar:init()
  local tableInfo = gre.get_table_attrs(self.table, "height", "rows", "yoffset")
  local cellInfo = gre.get_table_cell_attrs(self.table, 1, 1, "height")
  local scrollbarInfo = gre.get_control_attrs(self.scrollbar..".Background_Img", "height", "y")
  local sliderTopInfo = gre.get_control_attrs(self.scrollbar..".Slider_Top_Img", "height")
  local sliderBottomInfo = gre.get_control_attrs(self.scrollbar..".Slider_Bottom_Img", "height")
  
  self.totalheight = cellInfo.height * tableInfo.rows
  self.visiblepercentage = tableInfo.height / self.totalheight
  self.tableheight = tableInfo.height
  self.scrollbarheight = scrollbarInfo.height
  self.scrollbarpos = scrollbarInfo.y
  self.slidertopheight = sliderTopInfo.height
  
  if self.visiblepercentage < 1 then
    -- scale and position scrollbar according to visible entries
    self.sliderheight = (self.visiblepercentage * (scrollbarInfo.height - 8)) - (2 * self.slidertopheight)
    if self.sliderheight < 1 then self.sliderheight = 1 end
    local positionPercent = (-1 * tableInfo.yoffset) / (self.totalheight - self.tableheight)
    self.totalsliderheight = self.sliderheight + self.slidertopheight + sliderBottomInfo.height
    
    local sliderPos = self.scrollbarpos + (positionPercent * ((self.scrollbarheight - 8) - self.totalsliderheight)) + self.slidertopheight + 4
    local sliderTopPos = sliderPos - self.slidertopheight
    local sliderBottomPos = sliderPos + self.sliderheight - 1
    
    local data = {}
    data[self.scrollbar..".grd_hidden"] = false
    data[self.scrollbar..".Slider_Body_Img.grd_height"] = self.sliderheight
    data[self.scrollbar..".Slider_Body_Img.grd_y"] = sliderPos
    data[self.scrollbar..".Slider_Top_Img.grd_y"] = sliderTopPos
    data[self.scrollbar..".Slider_Bottom_Img.grd_y"] = sliderBottomPos
    gre.set_data(data)
    
    self.hidden = false
  else
    -- all entries visible, no scrollbar needed
    local data = {}
    data[self.scrollbar..".grd_hidden"] = true
    gre.set_data(data)
    self.hidden = true
  end
end

--
-- move the scrollbar slider to the correct position
--
function Scrollbar:scroll()
  if self.hidden then return end

  local tableInfo = gre.get_table_attrs(self.table, "yoffset")

  local positionPercent = (-1 * tableInfo.yoffset) / (self.totalheight - self.tableheight)
  local sliderPos = self.scrollbarpos + (positionPercent * ((self.scrollbarheight - 8) - self.totalsliderheight)) + self.slidertopheight + 4
  local sliderTopPos = sliderPos - self.slidertopheight
  local sliderBottomPos = sliderPos + self.sliderheight - 1

  local data = {}
  data[self.scrollbar..".Slider_Body_Img.grd_y"] = sliderPos
  data[self.scrollbar..".Slider_Top_Img.grd_y"] = sliderTopPos
  data[self.scrollbar..".Slider_Bottom_Img.grd_y"] = sliderBottomPos
  gre.set_data(data)
end

--[[
--
-- update table yoffset to match scrollbar position
--
local function do_scrollbar_work(self, new_y_pos)
  if not self.slideractive then return end
  
  local current_yoffset = gre.get_table_attrs(self.table, "yoffset").yoffset
  local new_slider_percentage = math.max(0, math.min(1, new_y_pos / self.scrollbarheight))
  local new_yoffset = new_slider_percentage * self.totalheight
  
  gre.set_table_attrs(self.table, { yoffset = new_yoffset })
end

--
-- scroll the table by touching the scrollbar
-- handle: gre.press, gre.release, gre.outbound, gre.motion
--
--- @param gre#context mapargs
function Scrollbar:on_action(mapargs)
  if mapargs.context_event == "gre.press" then
    self.slideractive = true
  elseif mapargs.context_event == "gre.release" then
    self.slideractive = false
  elseif mapargs.context_event == "gre.outbound" then
    self.slideracative = false
  elseif mapargs.context_event == "gre.motion" then
    do_scrollbar_work(self, mapargs.context_event_data.y)
  else
    -- ignore other events
  end
end
--]]

--[[ LAYER SCROLLBAR ]]--

-- scrollbar class
LayerScrollbar = {}
LayerScrollbar.__index = LayerScrollbar

-- LayerScrollbar.layer = "{$screen:scrolling_layer}"
-- LayerScrollbar.scrollbar = "{$screen:scrollbar_layer}"
-- LayerScrollbar.height = 800

--
-- create a new scrollbar instance (required layer id and scrollbar id)
--
function LayerScrollbar.new(o)
  assert(o.layer and o.scrollbar, "layer and scrollbar ids required")
  
  o = o or {}
  setmetatable(o, LayerScrollbar)
  return o
end

--
-- initialize the scrollbar (calculate slider size/position)
--
function LayerScrollbar:init(height)
  local layerInfo = gre.get_layer_attrs(self.layer, "height", "yoffset")
  local scrollbarInfo = gre.get_control_attrs(self.scrollbar..".Background_Img", "height", "y")
  local sliderTopInfo = gre.get_control_attrs(self.scrollbar..".Slider_Top_Img", "height")
  local sliderBottomInfo = gre.get_control_attrs(self.scrollbar..".Slider_Bottom_Img", "height")
  
  self.height = height
  self.visiblepercentage = layerInfo.height / self.height
  self.scrollbarheight = scrollbarInfo.height
  self.scrollbarpos = scrollbarInfo.y
  self.slidertopheight = sliderTopInfo.height
  
  if self.visiblepercentage < 1 then
    -- scale and position scrollbar according to visible entries
    self.sliderheight = (self.visiblepercentage * (scrollbarInfo.height - 8)) - (2 * self.slidertopheight)
    if self.sliderheight < 1 then self.sliderheight = 1 end
    local positionPercent = (-1 * layerInfo.yoffset) / (self.height - layerInfo.height)
    self.totalsliderheight = self.sliderheight + self.slidertopheight + sliderBottomInfo.height
    
    local sliderPos = self.scrollbarpos + (positionPercent * ((self.scrollbarheight - 8) - self.totalsliderheight)) + self.slidertopheight + 4
    local sliderTopPos = sliderPos - self.slidertopheight
    local sliderBottomPos = sliderPos + self.sliderheight - 1
    
    local data = {}
    data[self.scrollbar..".grd_hidden"] = false
    data[self.scrollbar..".Slider_Body_Img.grd_height"] = self.sliderheight
    data[self.scrollbar..".Slider_Body_Img.grd_y"] = sliderPos
    data[self.scrollbar..".Slider_Top_Img.grd_y"] = sliderTopPos
    data[self.scrollbar..".Slider_Bottom_Img.grd_y"] = sliderBottomPos
    gre.set_data(data)
    
    self.hidden = false
  else
    -- all entries visible, no scrollbar needed
    local data = {}
    data[self.scrollbar..".grd_hidden"] = true
    gre.set_data(data)
    self.hidden = true
  end
end

--
-- move the scrollbar slider to the correct position
--
function LayerScrollbar:scroll()
  if self.hidden then return end
  
  local layerInfo = gre.get_layer_attrs(self.layer, "yoffset", "height")

  local positionPercent = (-1 * layerInfo.yoffset) / (self.height - layerInfo.height)
  local sliderPos = self.scrollbarpos + (positionPercent * ((self.scrollbarheight - 8) - self.totalsliderheight)) + self.slidertopheight + 4
  local sliderTopPos = sliderPos - self.slidertopheight
  local sliderBottomPos = sliderPos + self.sliderheight - 1

  local data = {}
  data[self.scrollbar..".Slider_Body_Img.grd_y"] = sliderPos
  data[self.scrollbar..".Slider_Top_Img.grd_y"] = sliderTopPos
  data[self.scrollbar..".Slider_Bottom_Img.grd_y"] = sliderBottomPos
  gre.set_data(data)
end
