------------------------------------------------------
-- local copy of the configuration parameters database
-- both as keys with values and in a hierarchical tree
------------------------------------------------------

require("SubMenu")

local json = require("json")

-- settings singleton
Settings = {}
Settings.__index = Settings

Settings.db = {}        -- database of parameters and their values

--
-- read a setting from the database
--
-- note : 0 when parameters does not exist
-- as firmware may not a send 0 value parameter
function Settings:get(key)
  return self.db[key] or 0
end

--
-- read a setting from the database
--
-- return iDefault when parameters does not exist
--
function Settings:get_def(key, iDefault)
  local value = self.db[key]
  if value ~= nil then
    return value
  else
    return iDefault
  end
end

--
-- set a setting in the database
--
function Settings:set(key, value)
  self.db[key] = value
end

--
-- get a list with keys of the settings in the database, sorted alphabetically
--
function Settings:sorted_keys()
  local sorted = {}
  for name, _ in pairs(self.db) do
    table.insert(sorted, name)
  end
  table.sort(sorted)
  return sorted
end

--
-- for debugging print the contents of the setting database
--
function Settings:sorted_print()
  local keys = Settings:sorted_keys()
  for _, key in ipairs(keys) do
    print(key, Settings:get(key))
  end
end

-- top level menu
mainMenu = {
  key = "Settings", items = nil
}

gMenuStack = {mainMenu} -- menu stack

-- dead entry in menu
local function nullTableEnd()
  table.remove(gYOffsetStack)
  return false
end

-- display value as text
local function defaultDisplay(item)
  local value = Settings:get(item.param)
  if value ~= nil then
    return { value = tostring(value) }
  else
    return { value = "" }
  end
end

-- don't display value
local function noDisplay(item)
  return {}
end

-- mapping of parameter GIDs to menu actions
local gGIDs = {
  ["TEXT"] = {
    action = EditText,
    format = defaultDisplay,
    min = 0, max = 10
  },
  ["CLOUD_TEXT"] = {
    action = EditText,
    format = noDisplay,
    min = 0, max = 64
  },
  ["PINCODE"] = {
    action = EditPincode,
    format = noDisplay,
    min = 0, max = 9999
  },
  ["TEMPERATURE_UNIT"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "C", max = "F"
  },
  ["LIT_GAL"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "LIT", max = "GAL"
  },
  ["24HR_AMPM"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "24HR", max = "AMPM"
  },
  ["AM_OR_PM"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "AM", max = "PM"
  },
  ["DMY_MDY"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "DMY", max = "MDY"
  },
  ["NO_1x_CONT"] = {
    action = EditMultiToggle,
    format = MultiToggleDisplay,
    options = {"NO", "1x", "CONT"}
  },
  ["TEMPERATURE"] = {
    action = function() EditSlider(Temperature.isC() and "NLS_UNIT_C_LONG" or "NLS_UNIT_F_LONG") end,
    format = function(value) return DisplayNumber(value, Temperature.isC() and "NLS_UNIT_C_SHORT" or "NLS_UNIT_F_SHORT") end,
    convert = function(value, from_internal)
      if from_internal then
        local temperature = Temperature.new({temp=(value/10.0)})
        return temperature:get()
      else
        local temperature = Temperature.new({temp=0})
        temperature:set(value)
        return temperature:getC() * 10.0
      end
    end,
    min = 0, max = 2500
  },
  ["TEMPERATURE_OFFSET"] = {
    action = function() EditSlider(Temperature.isC() and "NLS_UNIT_C_LONG" or "NLS_UNIT_F_LONG") end,
    format = function(value) return DisplayNumber(value, Temperature.isC() and "NLS_UNIT_C_SHORT" or "NLS_UNIT_F_SHORT") end,
    convert = function(value, from_internal)
      if from_internal then
        local temperature = Temperature.new({temp=(value/10.0)})
        return temperature:getOffset()
      else
        local temperature = Temperature.new({temp=0})
        temperature:setOffset(value)
        return temperature:getOffsetC() * 10.0
      end
    end,
    min = -300, max = 300
  },
  ["NO_YES_TIMED"] = {
    action = EditMultiToggle,
    format = MultiToggleDisplay,
    options = {"NO", "YES", "TIMED"}
  },
  ["NO_YES"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "NO", max = "YES"
  },
  ["INTEGER"] = {
    action = EditNumber,
    format = DisplayNumber,
    min = 0, max = 65535
  },
  ["LITER"] = {
    action = function() EditSlider(Volume.isL() and "NLS_UNIT_LITER_LONG" or "NLS_UNIT_GALLON_LONG") end,
    format = function(value) return DisplayNumber(value, Volume.isL() and "NLS_UNIT_LITER_SHORT" or "NLS_UNIT_GALLON_SHORT") end,
    convert = function(value, from_internal)
      if from_internal then
        local volume = Volume.new({v=value})
        return volume:get()
      else
        local volume = Volume.new({v=0})
        volume:set(value)
        return volume:getL()
      end
    end,
    min = 0, max = 65535
  },
  ["FILTER_CAPACITY"] = {
    action = function() EditNumber(Volume.isL() and "NLS_UNIT_LITER_LONG" or "NLS_UNIT_GALLON_LONG", Volume.isL() and 50 or 100, true) end,
    format = function(value) return DisplayNumber(value, Volume.isL() and "NLS_UNIT_LITER_SHORT" or "NLS_UNIT_GALLON_SHORT", true) end,
    convert = function(value, from_internal)
      if from_internal then
        local volume = Volume.new({v=value})
        return volume:get()
      else
        local volume = Volume.new({v=0})
        volume:set(value)
        return volume:getL()
      end
    end,
    min = 0, max = 65535
  },
  ["PERCENT"] = {
    action = function() EditSlider("NLS_UNIT_PERCENT_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_PERCENT_SHORT") end,
    min = 0, max = 100
  },
  ["LANGUAGE"] = {
    action = SelectLanguage,
    format = DisplayLanguage
  },
  ["FACTOR"] = {
    action = EditSlider,
    format = DisplayNumber,
    min = 0, max = 100
  },
  ["YEAR"] = {
    action = EditSlider,
    format = DisplayNumber,
    min = 2000, max = 2050
  },
  ["MONTH"] = {
    action = EditSlider,
    format = DisplayNumber,
    min = 1, max = 12
  },
  ["DAY"] = {
    action = EditSlider,
    format = DisplayNumber,
    min = 1, max = 31
  },
  ["HOUR"] = {
    action = function() EditSlider("NLS_UNIT_HOURS_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_HOURS_SHORT") end,
    min = 0, max = 23
  },
  ["12HOUR"] = {   -- 12 hours used when AM/PM is format selected.
    action = function() EditSlider("NLS_UNIT_HOURS_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_HOURS_SHORT") end,
    min = 1, max = 12
  },
  ["MINUTE"] = {
    action = function() EditSlider("NLS_UNIT_MINUTES_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_MINUTES_SHORT") end,
    min = 0, max = 60
  },
  ["SECOND"] = {
    action = function() EditSlider("NLS_UNIT_SECONDS_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_SECONDS_SHORT") end,
    min = 0, max = 60
  },
  ["SECONDS_AS_HOURS"] = {
    action = function() EditSlider("NLS_UNIT_HOURS_LONG") end,
    format = function(value) return DisplayNumberScaled(value, 3600, "NLS_UNIT_HOURS_SHORT") end,
    min = 0, max = 0xFFFFFFFF
  },
  ["NO_3WAY_4F_4"] = {
    action = EditMultiToggle,
    format = MultiToggleDisplay,
    options = {"NO", "3WAY", "4F", "4"}
  },
  ["NO_DEEP_SHORT"] = {
    action = EditMultiToggle,
    format = MultiToggleDisplay,
    options = {"NO", "DEEP", "SHORT" }
  },
  ["DECISECOND"] = {
    action = function() EditSlider("NLS_UNIT_DECISECONDS_LONG") end,
    format = DisplayNumber,
    min = 0, max = 600
  },
  ["DECI"] = {
    action = function() EditSlider("NLS_UNIT_DECI_LONG") end,
    format = DisplayNumber,
    min = 0, max = 10
  },
  ["OPEN_PROG"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "OPEN", max = "PROG"
  },
  ["R6_OUTPUT"] = {
    action = PopulateToggle,
    format = ToggleDisplay,
    min = "BACKFLUSH_VALVE", max = "HEATER_SAFETY"
  },
  ["VOLTAGE"] = {
    action = function() EditSlider("NLS_UNIT_VOLTAGE_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_VOLTAGE_SHORT") end,
    min = 0, max = 330
  },
  ["DEVICE_TYPE"] = {
    action = EditMultiToggle,
    format = function(value) return { value = i18n:get(value.param .. "." .. Settings:get_def(value.param, 0)) } end,
    options = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", "17"},
    --options = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18"}, --TODO:FID4423 add OVEN_LDR_9S_AC_USA as "Device Type" number 18.
    filter = nil
  },

  ["MACHINE_MODEL"] = {
    action = nullTableEnd
  },
  ["TIMESTAMP"] = {
    action = nullTableEnd,
    format = DisplayTimeStamp
  },
  ["ACT"] = {
    action = PerformAction,
    format = noDisplay
  },
  ["ELEVATE"] = {
    action = ElevateRole,
    format = noDisplay
  },
  ["REFERENCE_COOK"] = {
    action = ReferenceCook,
    format = noDisplay
  },
  ["BITTABLE"] = {
    action = EditBitTable,
    format = noDisplay,
    update = UpdateBitTable
  },
  ["MULTITOGGLE"] = {
    action = EditGenericMultiToggle,
    format = defaultDisplay
  },
  ["STHACCPUSB"] = {
    action = StoreHACCPToUsb,
    format = noDisplay
  },
  ["LDPARUSB"] = {
    action = LoadSettingsFromUsb,
    format = noDisplay
  },
  ["STPARUSB"] = {
    action = StoreSettingsToUsb,
    format = noDisplay
  },
  ["LDRECUSB"] = {
    action = LoadRecipesFromUsb,
    format = noDisplay
  },
  ["STRECUSB"] = {
    action = StoreRecipesToUsb,
    format = noDisplay
  },
  ["MILLIVOLT"] = {
    action = function() EditSlider("NLS_UNIT_MILLIVOLT_LONG") end,
    format = function(value) return DisplayNumber(value, "NLS_UNIT_MILLIVOLT_SHORT") end,
    min = 0, max = 10000
  },
  ["RSSI_DECIBEL"] = {
    action = EditNumber,
    format = function(value) return DisplayNegativeNumber(value, "NLS_UNIT_DECIBEL_SHORT") end,
    min = 0, max = 110
  }
}

--
-- get the type info
--
function get_type_meta_data(setting_type)
  return gGIDs[setting_type]
end

--
-- parse a json condition and convert is to a lambda
--
local function parse_condition(condition)
  local no_filter = function() return true end
  if condition == nil or condition.Parameter == nil or condition.Operator == nil or condition.Values == nil then return no_filter end
  local function_map = {
    ["IN"]           = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values >= 1 and in_array(value, values) end,
    ["NOT IN"]       = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values >= 1 and not in_array(value, values) end,
    ["IN RANGE"]     = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 2 and value >= values[1] and value <= values[2] end,
    ["NOT IN RANGE"] = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 2 and (value < values[1] or value > values[2]) end,
    ["=="]           = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value == values[1] end,
    ["!="]           = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value ~= values[1] end,
    ["<"]            = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value < values[1] end,
    ["<="]           = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value <= values[1] end,
    [">"]            = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value > values[1] end,
    [">="]           = function(x, values) local value = Settings:get_def(x, nil) return value ~= nil and #values == 1 and value >= values[1] end
  }
  local f = function_map[condition.Operator]
  local values = convert_to_table(condition.Values)
  if f ~= nil then return function() return f(condition.Parameter, values) end end
  return no_filter
end

--
-- find the submenu with key in our menu tree
--
local function findMenu(menu, key)
  local result = nil
  for _, item in pairs(menu.items or {}) do
    if item.key == key then
      result = item
      break
    elseif item.type == "menu" and item.items ~= nil then
      result = findMenu(item, key)
      if result ~= nil then break end
    end
  end
  return result
end

--
-- insert a submenu in the settings menu
--
local function insertMenu(entry)
  local items = {}

  -- if submenu entry already exists, use that one, this
  -- allows items to appear in multiple menus
  for _, item in ipairs(entry.Ref) do
    table.insert(items, findMenu(mainMenu, item) or {
      key = item,
      action = nullTableEnd
    })
  end

  if entry.ID == "TOP" then
    mainMenu.items = items
  else
    local menu = findMenu(mainMenu, entry.ID)
    if menu then
      menu.type = "menu"
      menu.action = PopulateMenu
      menu.items = items
      menu.role = entry.Role
      menu.condition = parse_condition(entry.Condition)
    else
      print("No such menu ["..entry.ID.."]")
    end
  end
end

--
-- find a the parameter with key in the parsed json
--
local function findParameter(parameters, key)
  local result = nil
  for _, item in pairs(parameters) do
   if item.ID == key then
     result = item
     break
   end
  end
  return result
end

--
-- check if parameter is read-only
--
local function isReadOnly(parameter)
  return string.find(parameter.Type or "", "[AW]") == nil
end

--
-- check if parameter is a live variable
--
local function isLive(parameter)
  return string.find(parameter.Type or "", "L") ~= nil
end

--
-- insert a parameter in the settings menu
--
local function insertParameter(parameters, entry)
  local menu = findMenu(mainMenu, entry.ID)
  local parameter = findParameter(parameters, entry.Ref)

  if not menu or not parameter then
    print("No such parameter ["..entry.Ref.."]")
    return
  end

  if gGIDs[parameter.GID] then
    menu.type = parameter.GID
    menu.action = gGIDs[parameter.GID].action or nullTableEnd
    menu.update = gGIDs[parameter.GID].update or function() end
    menu.format = gGIDs[parameter.GID].format or defaultDisplay
    menu.convert = gGIDs[parameter.GID].convert or nil
    menu.filter = gGIDs[parameter.GID].filter or nil
    if parameter.Min ~= nil then
      menu.min = tonumber(parameter.Min)
    else
      menu.min = gGIDs[parameter.GID].min
    end
    if parameter.Max ~= nil then
      menu.max = tonumber(parameter.Max)
    else
      menu.max = gGIDs[parameter.GID].max
    end
    menu.options = parameter.options or gGIDs[parameter.GID].options

    if (parameter.GID == "INTEGER" or parameter.GID == "LITER")
      and menu.min ~= nil and menu.max ~= nil and (menu.max - menu.min) <= 500 then
      menu.action = EditSlider
    end
  else
    print("Unhandled parameter type ["..parameter.GID.."]")
    menu.action = nullTableEnd
    menu.format = defaultDisplay
  end

  menu.isReadOnly = isReadOnly(parameter)
  menu.isLive = isLive(parameter)

  if menu.isReadOnly and parameter.GID ~= "BITTABLE" then
    menu.action = nullTableEnd
  end

  if parameter.GID == "ACT" then
    menu.prompt = parameter.Prompt
  end

  menu.condition = parse_condition(entry.Condition)
  menu.param = parameter.ID
  menu.role = parameter.Role
end

--
-- initialize the settings menu structure
--
local function initSettingsMenu()
  local f = assert(io.open(config_path("42smenu.json"), "r"))
  local t = f:read("*all")
  f:close()

  local json_content = json.decode(t)

  for _, entry in ipairs(json_content.menu) do
    if entry.Type == "COLLECTION" then
      insertMenu(entry)
    elseif entry.Type == "PARAMETER" then
      insertParameter(json_content.parameters, entry)
    else
      error("Invalid menu entry ["..entry.Type.."]")
    end
  end
end

--
-- load language set in settings as active display language
-- fall to default id and name
--
local function loadLanguage()
  i18n:load(i18n:id_to_name(Settings:get("LANGUAGE") or 0) or "English")
end

--
-- recursive helper function to find a setting in the menu
--
local function recurseFindSetting(menu, key)
  for _, entry in ipairs(menu.items or {}) do
    if entry.type == "menu" then
      local result = recurseFindSetting(entry, key)
      if result then return result end
    elseif entry.param == key then
      return entry
    end
  end
  return nil
end

--
-- find a setting in the settings menu
--
function FindSetting(key)
  return recurseFindSetting(mainMenu, key)
end

--
-- settings event received, which contains (key, value) pairs
--
--- @param gre#context mapargs
function CB_OnSettings(mapargs)
  for i, id in ipairs(split_string(mapargs.context_event_data.ids, DELIMITER_TOKEN)) do
    Settings:set(id, mapargs.context_event_data.values[i])
  end

  -- when last chunk
  if mapargs.context_event_data.last ~= 0 then
    loadLanguage()
    initSettingsMenu()
    gre.send_event("settings_init_done")

    -- Settings:sorted_print()

  end -- last chunk
end

--
-- settings event received, which contains a key and string value
--
function CB_OnSettingsString(mapargs)
  local data = split_string(mapargs.context_event_data.data, DELIMITER_TOKEN, true)

  local id = data[1]
  local value = data[2]
  if value == nil then value = "" end
  Settings:set(id, value)
end

--
-- event to refresh parameter database  (io_refesh_parameter)
--
--- @param gre#context mapargs
function CB_OnRefreshParameter(mapargs)
  Settings.db = {}
  Event:get_parameters()
end

--
-- device type options received
--
function CB_OnParameterOptions(mapargs)
  local reply = split_string(mapargs.context_event_data.options, DELIMITER_TOKEN)

  local parameter = reply[1]
  if parameter ~= "DEVICE_TYPE" then return end

  local options = {}
  for i=2,#reply do
    table.insert(options, reply[i])
  end

  gGIDs["DEVICE_TYPE"].filter = function(item) return in_array(item, options) end
end
