----------------------------------------------
-- Schedule cleaning or recipe program start
----------------------------------------------

require("language")
require("Settings")
require("Time")

local CLEANING_SET_STARTTIME_DIALOG = "StartCleaning_Overlay.selecttime_dialog"
local CLEANING_STARTTIME_TEXT = "StartCleaning_Overlay.starttime_ctrl.text"

local RECIPE_SET_STARTTIME_DIALOG = "StartRecipe_Overlay.selecttime_dialog"
local RECIPE_STARTTIME_TEXT = "StartRecipe_Overlay.starttime_ctrl.text"

local SET_STARTTIME_DIALOG = CLEANING_SET_STARTTIME_DIALOG
local STARTTIME_TEXT = CLEANING_STARTTIME_TEXT

local gCleaningStartTime = 0
local gRecipeStartTime = 0
local gPrevRecipeStartTime = 0  -- FID4193: Previous inputted recipe start time is used as intial recipe start time with the next input action.
local gbAmPm = false
local gbIsPm = false
local gSlidingY = 0
local gSlidingHours = false
local gSlidingMinutes = false

--
-- get the time at which the cleaning program should start
--
function Cleaning_GetStartingTime()
  return gCleaningStartTime
end

-- set the time at which the cleaning should start to (0 = no delay)
--
function Cleaning_SetStartingTime(start_time)
  gCleaningStartTime = start_time
  local timestamp = 0
  local ampm = 0

  if gCleaningStartTime == 0 then
    gre.set_value(CLEANING_STARTTIME_TEXT, "")
  else
    timestamp, ampm = format_time(gCleaningStartTime)
    if ampm then timestamp = string.format("%s %s", timestamp, ampm) end
    gre.set_value(CLEANING_STARTTIME_TEXT, timestamp)
  end
end

--
-- get the time at which the recipe should start
--
function Recipe_GetEnteredStartTime()
  return gRecipeStartTime
end

-- set the time at which the recipe should start (0 = no delay)
--
function Recipe_SetEnteredStartTime(start_time)
  gRecipeStartTime = start_time
  local timestamp = 0
  local ampm = 0
  local data = {}

  if gRecipeStartTime == 0 then
    gre.set_value(RECIPE_STARTTIME_TEXT, "")
  else
    timestamp, ampm = format_time(gRecipeStartTime)
    if ampm then timestamp = string.format("%s %s", timestamp, ampm) end
    gre.set_value(RECIPE_STARTTIME_TEXT, timestamp)
  end
  gre.set_data(data)
end

--
-- Select the correct starttime dialog layer depening on which screen we are on.
---
function Select_StartTime_Dialog(used_screen, reset_start_time)
  if used_screen == "CLEANING_Screen" then
    SET_STARTTIME_DIALOG = CLEANING_SET_STARTTIME_DIALOG
    STARTTIME_TEXT = CLEANING_STARTTIME_TEXT
    if reset_start_time then
      Cleaning_SetStartingTime(0)
    end
  elseif used_screen == "REC_Screen" then
    SET_STARTTIME_DIALOG = RECIPE_SET_STARTTIME_DIALOG
    STARTTIME_TEXT = RECIPE_STARTTIME_TEXT
    if reset_start_time then
      Recipe_SetEnteredStartTime(0)
    end
  end
end

--
-- press event on hours or minutes
-- make the respective control active for time scrolling
--
function CBStartTime_SelectTime_OnPress(mapargs)
  Select_StartTime_Dialog(mapargs.context_screen, false)
  gSlidingY = tonumber(mapargs.context_event_data.y)
  gSlidingHours = false
  gSlidingMinutes = false

  local hours = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.hours"))
  local minutes = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.minutes"))

  local data = {}

  data[SET_STARTTIME_DIALOG..".time_text.hours_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.hours_more"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes_more"] = ""

  if mapargs.context_control == SET_STARTTIME_DIALOG..".slide_control_hours" then
    gSlidingHours = true
    local min_hours = gbAmPm and 1 or 0
    local max_hours = gbAmPm and 12 or 23
    data[SET_STARTTIME_DIALOG..".time_text.hours_less"] = string.format("%02d", hours > min_hours and hours - 1 or max_hours)
    data[SET_STARTTIME_DIALOG..".time_text.hours_more"] = string.format("%02d", hours < max_hours and hours + 1 or min_hours)
  elseif mapargs.context_control == SET_STARTTIME_DIALOG..".slide_control_minutes" then
    gSlidingMinutes = true
    local max_minutes = 59
    data[SET_STARTTIME_DIALOG..".time_text.minutes_less"] = string.format("%02d", minutes >= 1 and minutes - 1 or max_minutes)
    data[SET_STARTTIME_DIALOG..".time_text.minutes_more"] = string.format("%02d", minutes < max_minutes and minutes + 1 or 0)
  end

  gre.set_data(data)
end

--
-- release event on hours or minutes
-- depending on the y-coordinate compared to the y-coordinate from the press event, scroll up or down
--
function CBStartTime_SelectTime_OnRelease(mapargs)
  Select_StartTime_Dialog(mapargs.context_screen, false)
  local y = tonumber(mapargs.context_event_data.y)
  local hours = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.hours"))
  local minutes = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.minutes"))

  local data = {}

  data[SET_STARTTIME_DIALOG..".time_text.hours_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.hours_more"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes_more"] = ""

  if mapargs.context_control == SET_STARTTIME_DIALOG..".slide_control_hours" and gSlidingHours then
    local old_hours = hours

    local bScrollingDown = y >= (gSlidingY + 5)
    local bScrollingUp = y <= (gSlidingY - 5)
    if bScrollingDown then
      hours = hours - 1
    elseif bScrollingUp then
      hours = hours + 1
    end

    local min_hours = gbAmPm and 1 or 0
    local max_hours = gbAmPm and 12 or 23
    if hours < min_hours then
      hours = max_hours
    end
    if hours > max_hours then
      hours = min_hours
    end

    if gbAmPm then
      if gbIsPm and old_hours == 11 and hours == 12 then gbIsPm = false         -- 11pm -> 12 => am
      elseif gbIsPm and old_hours == 12 and hours == 11 then gbIsPm = false     -- 12pm -> 11 => am
      elseif not gbIsPm and old_hours == 11 and hours == 12 then gbIsPm = true  -- 11am -> 12 => pm
      elseif not gbIsPm and old_hours == 12 and hours == 11 then gbIsPm = true  -- 12am -> 11 => pm
      end
    end

    data[SET_STARTTIME_DIALOG..".time_text.hours"] = string.format("%02d", hours)
    data[SET_STARTTIME_DIALOG..".time_text.hours_less"] = string.format("%02d", hours > min_hours and hours - 1 or max_hours)
    data[SET_STARTTIME_DIALOG..".time_text.hours_more"] = string.format("%02d", hours < max_hours and hours + 1 or min_hours)
    data[SET_STARTTIME_DIALOG..".ampm_text.text"] = gbAmPm and i18n:get(gbIsPm and "NLS_PM" or "NLS_AM") or ""
  elseif mapargs.context_control == SET_STARTTIME_DIALOG..".slide_control_minutes" and gSlidingMinutes then
    if y >= (gSlidingY + 5) then
      minutes = minutes - 1
    elseif y <= (gSlidingY - 5) then
      minutes = minutes + 1
    end

    local max_minutes = 59
    if minutes < 0 then minutes = max_minutes end
    if minutes > max_minutes then minutes = 0 end

    data[SET_STARTTIME_DIALOG..".time_text.minutes"] = string.format("%02d", minutes)
    data[SET_STARTTIME_DIALOG..".time_text.minutes_less"] = string.format("%02d", minutes >= 1 and minutes - 1 or max_minutes)
    data[SET_STARTTIME_DIALOG..".time_text.minutes_more"] = string.format("%02d", minutes < max_minutes and minutes + 1 or 0)
  end

  gre.set_data(data)
end

--
-- confirm selection of edited starting time
--
function CBStartTime_SelectTime_OnOk(mapargs)
  Select_StartTime_Dialog(mapargs.context_screen, false)
  local new_hours = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.hours"))
  local new_minutes = tonumber(gre.get_value(SET_STARTTIME_DIALOG..".time_text.minutes"))

  local epoch_time = now()
  local _, _, _, h, min = utc_time(epoch_time)
  local offset = Settings:get("TIMEZONE_OFFSET") or 0

  if gbAmPm then
    if not gbIsPm and new_hours == 12 then new_hours = 0 end
    if gbIsPm and new_hours < 12 then new_hours = new_hours + 12 end
  end

  local base_epoch_time = epoch_time - (h * 3600) - (min * 60)
  local new_epoch_time = base_epoch_time + ((new_hours - offset) * 3600) + (new_minutes * 60)

  if new_epoch_time < (epoch_time - 60) then
    -- if selected time lies in the past, find the next time it occurs
    new_epoch_time = new_epoch_time + ((gbAmPm and 12 or 24) * 3600)
  end

  local timestamp = 0
  local ampm = 0
  if mapargs.context_screen == "CLEANING_Screen" then
    Cleaning_SetStartingTime(new_epoch_time)
  elseif mapargs.context_screen == "REC_Screen" then
    Recipe_SetEnteredStartTime(new_epoch_time)
    gPrevRecipeStartTime = new_epoch_time
  end
end

--
-- cancel selection of cleaning/recipe start time -> reset start time to 0
--
function CBStartTime_SelectTime_OnCancel(mapargs)
  Select_StartTime_Dialog(mapargs.context_screen, true)
end

--
-- show cleaning/recipe start time in selection dialog
--
function CBStartTime_SelectTime_OnShow(mapargs)
  Select_StartTime_Dialog(mapargs.context_screen, false)
  gSlidingY = 0
  gSlidingHours = false
  gSlidingMinutes = false
  gbAmPm = Settings:get_def("TIME_FORMAT", 0) == 1
  gbIsPm = false

  local utctime = 0
  if mapargs.context_screen == "CLEANING_Screen" then
    utctime = (gCleaningStartTime == 0) and now() or gCleaningStartTime
  elseif mapargs.context_screen == "REC_Screen" then
    if gRecipeStartTime ~= 0 then
      utctime = gRecipeStartTime
    elseif gPrevRecipeStartTime ~= 0 then
      utctime = gPrevRecipeStartTime
    else
      utctime = now()
    end
  end

  local _, _, _, h, min, _ = local_time(utctime)
  local ampm = ""

  if gbAmPm then
    ampm = i18n:get("NLS_AM")
    if h == 0 then
      h = 12
      ampm = i18n:get("NLS_AM")
    elseif h == 12 then
      ampm = i18n:get("NLS_PM")
      gbIsPm = true
    elseif h > 12 then
      h = h - 12
      ampm = i18n:get("NLS_PM")
      gbIsPm = true
    end
  end

  local data = {}

  data[SET_STARTTIME_DIALOG..".time_text.hours_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.hours"] = string.format("%02d", h)
  data[SET_STARTTIME_DIALOG..".time_text.hours_more"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes_less"] = ""
  data[SET_STARTTIME_DIALOG..".time_text.minutes"] = string.format("%02d", min)
  data[SET_STARTTIME_DIALOG..".time_text.minutes_more"] = ""
  data[SET_STARTTIME_DIALOG..".ampm_text.grd_hidden"] = not gbAmPm
  data[SET_STARTTIME_DIALOG..".ampm_text.text"] = ampm
  data[SET_STARTTIME_DIALOG..".grd_hidden"] = false

  gre.set_data(data)
end

--
-- prepare the cleaning start time selection dialog
-- this function should be called each time the cleaning program selection overlay is shown
--
function CBCleaning_StartTime_Prepare()
  Cleaning_SetStartingTime(0)
  local data = {}
  data[CLEANING_SET_STARTTIME_DIALOG..".grd_hidden"] = true
  gre.set_data(data)
end

--
-- prepare the recipe start time selection dialog
-- this function should be called each time the recipe program selection overlay is shown
--
function CBRecipe_StartTime_Prepare()
  local data = {}
  data[RECIPE_SET_STARTTIME_DIALOG..".grd_hidden"] = true
  gre.set_data(data)
end
