---------------------------------------------------------------
-- implementation of the possible subtypes of the settings tree
---------------------------------------------------------------

require("bit32")
require("Scrollbar")
require("events")

local TABLE_SETTINGS = "SET_Menu_Layer.SET_Settings_Table" -- settings table

gLastValue = nil                                           -- previous value of the parameter

--[[ General ]]--

--
-- settings are readonly while a recipe or cleaning program is running
--
function SettingsAreReadOnly()
  if ActiveRecipe:is_cooking() or ActiveCleanProg:is_cleaning() then
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get("NLS_SETTINGS_READONLY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
    return true
  else
    return false
  end
end

--[[ Submenu ]]--

--
-- menu visitor, returns index of last entry + 1
--
local function VisitMenu(menu_lambda, parameter_lambda)
  local i = 1
  for index, entry in ipairs(gCurrentMenu.items or {}) do
    if entry.condition() then
      if entry.type == "menu" then
        menu_lambda(index, i, entry)
        i = i + 1
      elseif not entry.role or RoleManager:can_access(entry.role) then
        parameter_lambda(index, i, entry)
        i = i + 1
      end
    end
  end
  return i
end

--
-- activate a submenu
--
function PopulateMenu(back)
  local data = {}

  if #gMenuStack < 1 then
    gCurrentMenu = mainMenu
    table.insert(gMenuStack, mainMenu)
  else
    gCurrentMenu = gMenuStack[#gMenuStack]
    if not back then History:force_push("SET_Screen") end
  end

  local menu_lambda = function(parameter_index, menu_index, entry)
    data[TABLE_SETTINGS..".key."..menu_index..".1"] = i18n:get(entry.key)
    data[TABLE_SETTINGS..".alpha."..menu_index..".1"] = 255
    data[TABLE_SETTINGS..".value."..menu_index..".1"] = ""
    data[TABLE_SETTINGS..".timestamp_value_date."..menu_index..".1"] = ""
    data[TABLE_SETTINGS..".timestamp_value_time."..menu_index..".1"] = ""
    data[TABLE_SETTINGS..".index."..menu_index..".1"] = parameter_index
  end

  local parameter_lambda = function(parameter_index, menu_index, entry)
    if not entry.param then print(table_print(entry)) end
    data[TABLE_SETTINGS..".key."..menu_index..".1"] = i18n:get(entry.param)
    data[TABLE_SETTINGS..".alpha."..menu_index..".1"] = 0
    if entry.format then
      local display_value = entry.format(entry)
      data[TABLE_SETTINGS..".value."..menu_index..".1"] = if_not_nil_else(display_value.value, "")
      data[TABLE_SETTINGS..".timestamp_value_date."..menu_index..".1"] = if_not_nil_else(display_value.date, "")
      data[TABLE_SETTINGS..".timestamp_value_time."..menu_index..".1"] = if_not_nil_else(display_value.time, "")
    else
      data[TABLE_SETTINGS..".value."..menu_index..".1"] = tostring(Settings:get(entry.param) or 0)
    end
    data[TABLE_SETTINGS..".index."..menu_index..".1"] = parameter_index
  end

  local i = VisitMenu(menu_lambda, parameter_lambda)
  if i == 1 then
    data[TABLE_SETTINGS..".key."..i..".1"] = i18n:get("NLS_NO_ITEMS")
    data[TABLE_SETTINGS..".alpha."..i..".1"] = 0
    data[TABLE_SETTINGS..".value."..i..".1"] = ""
    data[TABLE_SETTINGS..".timestamp_value_date."..i..".1"] = ""
    data[TABLE_SETTINGS..".timestamp_value_time."..i..".1"] = ""
    data[TABLE_SETTINGS..".index."..i..".1"] = 0
    i = i + 1
  end

  gre.set_table_attrs(TABLE_SETTINGS, { rows = i - 1, yoffset = back and gOldYOffset or 0 })

  data["SET_Screen.activeLayer"] = "SET_Menu_Layer"
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--
-- refresh the value of a parameter item corresponding to iParamID
--
function RefreshSubMenuItem(iParamID)
  local menu_active = (gre.get_value("SET_Screen.activeLayer") == "SET_Menu_Layer")
  if menu_active then
    local data = {}
    local menu_lambda = function(parameter_index, menu_index, entry) end
    local parameter_lambda = function(parameter_index, menu_index, entry)
      if entry.param == iParamID then
        if entry.format then
          local display_value = entry.format(entry)
          data[TABLE_SETTINGS..".value."..menu_index..".1"] = if_not_nil_else(display_value.value, "")
          data[TABLE_SETTINGS..".timestamp_value_date."..menu_index..".1"] = if_not_nil_else(display_value.date, "")
          data[TABLE_SETTINGS..".timestamp_value_time."..menu_index..".1"] = if_not_nil_else(display_value.time, "")
        else
          data[TABLE_SETTINGS..".value."..menu_index..".1"] = tostring(Settings:get(entry.param) or 0)
        end
      end
    end
    VisitMenu(menu_lambda, parameter_lambda)
    gre.set_data(data)
  else
    local item = gCurrentMenu.items and gCurrentMenu.items[gLastRow] or nil
    if item ~= nil and item.isLive and item.param == iParamID then
      item.update()
    end
  end
end

--
-- refresh all currently visible values
--
function RefreshMenu()
  local menu_active = (gre.get_value("SET_Screen.activeLayer") == "SET_Menu_Layer")
  if menu_active then
    local data = {}
    local menu_lambda = function(parameter_index, menu_index, entry) end
    local parameter_lambda = function(parameter_index, menu_index, entry)
      if entry.format then
        data[TABLE_SETTINGS..".value."..menu_index..".1"] = entry.format(entry)
      else
        data[TABLE_SETTINGS..".value."..menu_index..".1"] = tostring(Settings:get(entry.param) or 0)
      end
    end
    VisitMenu(menu_lambda, parameter_lambda)
    gre.set_data(data)
  else
    local item = gCurrentMenu.items and gCurrentMenu.items[gLastRow] or nil
    if item ~= nil then item.update() end
  end
end

--[[ Simple toggle control ]]--

--
-- display a toggle parameter
--
function ToggleDisplay(item)
  assert(item.min and item.max, "Missing options: [" .. item.param .. "]")
  local value = Settings:get(item.param) or 0
  return { value = i18n:get(item.param .. (value == 0 and "." .. item.min or "." .. item.max)) }
end

--
-- activate a toggle parameter
--
function PopulateToggle()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]

  gLastValue = Settings:get(item.param) or 0
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "toggle")

  data["SET_Toggle_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_Toggle_Layer.Swap_Left.text"] = i18n:get(item.param .. "." .. item.min)
  data["SET_Toggle_Layer.Swap_Right.text"] = i18n:get(item.param .. "." .. item.max)

  if gLastValue == 0 then
    data["SET_Toggle_Layer.toggled"] = 0
    data["SET_Toggle_Layer.Swap_Btn.x"] = 3
  else
    data["SET_Toggle_Layer.toggled"] = 1
    data["SET_Toggle_Layer.Swap_Btn.x"] = 57
  end

  data["SET_Screen.activeLayer"] = "SET_Toggle_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--[[ Language selection ]]--

local ACTIVE_LANG_IMG = image_path("parlijst-schuifknop-y148.png")

--
-- display active language parameter
--
function DisplayLanguage(item)
  return { value = i18n:id_to_name(Settings:get(item.param)) }
end

--
-- language selection screen
--
function SelectLanguage()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local item = gCurrentMenu.items[gLastRow]
  gLastValue = Settings:get(item.param) or 0
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "language_select")

  CBLanguage_OnOverlayShow()

  History:force_push("SET_Screen")
  gre.set_value("SET_Screen.activeLayer", "Language_Layer")
  gre.send_event("swap_menu")
end

--[[ Pincode edit control ]]--

--
-- pincode edit screen
--
function EditPincode()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]

  gLastValue = ""
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "pincode")

  data["SET_Pin_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_Pin_Layer.feedback_text.text"] = ""
  data["SET_Pin_Layer.pincode_group.pincode_text_1.text"] = ""
  data["SET_Pin_Layer.pincode_group.pincode_text_2.text"] = ""
  data["SET_Pin_Layer.pincode_group.pincode_text_3.text"] = ""
  data["SET_Pin_Layer.pincode_group.pincode_text_4.text"] = ""

  data["SET_Screen.activeLayer"] = "SET_Pin_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--[[ Number edit control (numeric keyboard) ]]--

-- display value as number
function DisplayNumber(item, unit_key, zero_is_infinity)
  local value = Settings:get(item.param) or 0
  if value == 0 and zero_is_infinity then
    return { value = INFINITY_SYMBOL }
  else
    if item.convert then value = item.convert(value, true) end
    if unit_key then
      return { value = string.format("%d%s", value, i18n:get(unit_key)) }
    else
      return { value = tostring(value) }
    end
  end
end

-- display value as negative number for RSSI decibel
function DisplayNegativeNumber(item, unit_key)
  local value = Settings:get(item.param) or 0
  if value == 0 then
    return { value = string.format("--") }
  else
    if item.convert then value = item.convert(value, true) end
    if unit_key then
      return { value = string.format("-%d%s", value, i18n:get(unit_key)) }
    else
      return { value = tostring(value) }
    end
  end
end

-- display value as number, but scale it first
function DisplayNumberScaled(item, scale_factor, unit_key)
  local value = math.floor(((Settings:get(item.param) or 0) / scale_factor) + 0.5)
  if item.convert then value = item.convert(value, true) end
  if unit_key then
    return { value = string.format("%d%s", value, i18n:get(unit_key)) }
  else
    return { value = tostring(value) }
  end
end

--
-- number (integer) edit screen
--
function EditNumber(unit_key, step_size, zero_is_infinity)
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]

  gLastValue = Settings:get(item.param) or 0
  if item.convert then gLastValue = item.convert(gLastValue, true) end
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "integer")

  data["SET_Integer_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_Integer_Layer.Integer_Group.Number_Text.text"] = ((gLastValue == 0) and zero_is_infinity) and INFINITY_SYMBOL or string.format("%d", gLastValue)
  data["SET_Integer_Layer.Unit_Text.text"] = unit_key and i18n:get(unit_key) or ""
  data["SET_Integer_Layer.step_group.step_size"] = step_size or 0
  data["SET_Integer_Layer.step_group.grd_hidden"] = step_size == 0
  data["SET_Integer_Layer.zero_is_infinity"] = zero_is_infinity and 1 or 0

  data["SET_Screen.activeLayer"] = "SET_Integer_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--[[ Number edit control (slider) ]]--

--
-- number (range) edit screen
--
function EditSlider(unit_key)
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]

  gLastValue = Settings:get(item.param) or 0
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "slider")

  local item_min = item.min
  local item_max = item.max
  if item.convert then
    gLastValue = item.convert(gLastValue, true)
    item_min = item.convert(item_min, true)
    item_max = item.convert(item_max, true)
  end

  data["SET_Slider_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_Slider_Layer.Integer_Group.Number_Text.text"] = string.format("%d", gLastValue)
  data["SET_Slider_Layer.Slider_Group.Interval0_Text.text"] = string.format("%d", item_min)
  for i = 1,7 do
    data["SET_Slider_Layer.Slider_Group.Interval"..i.."_Text.text"] = math.floor((item_min + ((i/8) * (item_max - item_min))) + 0.5)
  end
  data["SET_Slider_Layer.Slider_Group.Interval8_Text.text"] = string.format("%d", item_max)

  local min = 25
  local width = gre.get_value("SET_Slider_Layer.Slider_Group.Slider_Btn.grd_width") - 58
  local icon_center = 25
  local percentage = (gLastValue - item_min) / (item_max - item_min)
  local position = min + (percentage * width) - icon_center
  data["SET_Slider_Layer.Slider_Group.Slider_Btn.x"] = position
  data["SET_Slider_Layer.Unit_Text.text"] = unit_key and i18n:get(unit_key) or ""

  data["SET_Screen.activeLayer"] = "SET_Slider_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--[[ Toggle control with many options (radio group) ]]--

local MTOPTIONS_GRP = "SET_MultiToggle_Layer.SliderOptions_Group"
local gMultiToggleOptionIds = {}

--
-- display selected toggle option
--
function MultiToggleDisplay(item)
  assert(item.options, "Missing options: [" .. item.param .. "]")
  local value = Settings:get(item.param) or 0
  return { value = i18n:get(item.param .. "." .. item.options[value + 1]) }
end

--
-- toggle control with many options
--
function EditMultiToggle()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]
  local no_nls = item.type == "MULTITOGGLE"

  assert(item.options, "missing options")

  gLastValue = Settings:get(item.param) or 0
  if no_nls then
    gLastValue = index_of(gLastValue, item.options) or 1
    gLastValue = gLastValue - 1
  end

  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "multitoggle")

  local numoptions = #item.options
  local option_height = 34 -- was 40        ( should match the one in SET_Screen.lua )
  local inter_option_height = 12 -- was 20  ( should match the one in SET_Screen.lua )
  local ypos = gre.get_value("SET_MultiToggle_Layer.Slider_Ctrl.grd_y")

  data["SET_MultiToggle_Layer.Title_Text.text"] = i18n:get(item.param)

  for i, _ in pairs(gMultiToggleOptionIds) do
    data[MTOPTIONS_GRP..".Item_Label_"..i..".grd_hidden"] = true
  end

  local active_option = 1
  local index = 1
  for options_index = 1, numoptions do
    if not item.filter or item.filter(item.options[options_index]) then
      if index > 1 and not gMultiToggleOptionIds[index] then
        gre.clone_object(MTOPTIONS_GRP..".Item_Label", "Item_Label_"..index, MTOPTIONS_GRP)
        gMultiToggleOptionIds[index] = true
      end

      local label = (index == 1) and ".Item_Label" or (".Item_Label_"..index)

      if no_nls then
        data[MTOPTIONS_GRP..label..".text"] = item.options[options_index]
      else
        data[MTOPTIONS_GRP..label..".text"] = i18n:get(item.param .. "." .. item.options[options_index])
      end
      data[MTOPTIONS_GRP..label..".color"] = 0xFFFFFF
      data[MTOPTIONS_GRP..label..".id"] = index
      data[MTOPTIONS_GRP..label..".options_index"] = options_index - 1

      if index > 1 then
        data[MTOPTIONS_GRP..label..".grd_y"] = ypos + ((index - 1) * (option_height + inter_option_height))
        data[MTOPTIONS_GRP..label..".grd_hidden"] = false
      end

      if gLastValue + 1 == options_index then
        active_option = index
      end
      index = index + 1
    end
  end

  index = index - 1
  local height = (index * option_height) + ((index - 1 ) * inter_option_height) + 4 -- 2 pixels padding at top and bottom
  data["SET_MultiToggle_Layer.Slider_Ctrl.grd_height"] = height
  data["SET_MultiToggle_Layer.Slider_Ctrl.center_part_height"] = height - 44

  local active_key = MTOPTIONS_GRP .. ".Item_Label" .. (active_option > 1 and "_"..active_option or "")
  data[active_key..".color"] = 0xB1D5CF
  data["SET_MultiToggle_Layer.Slider_Ctrl.slider_y"] = 2 + ((active_option - 1) * (option_height + inter_option_height))
  data["SET_MultiToggle_Layer.Slider_Ctrl.active_option"] = active_option

  data["SET_Screen.activeLayer"] = "SET_MultiToggle_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--
-- toggle control with many variable options, options are requested from the application first
--
function EditGenericMultiToggle()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  Event:parameter_options(gCurrentMenu.items[gLastRow].param)
end

--
-- toggle options received from application
--
function CBSET_OnMultiToggleOptions(mapargs)
  local reply = split_string(mapargs.context_event_data.options, DELIMITER_TOKEN)
  local item = gCurrentMenu.items[gLastRow]

  local parameter = reply[1]
  if parameter ~= item.param then
    table.remove(gYOffsetStack)
    return
  end

  item.options = {}
  for i=2,#reply do
    table.insert(item.options, reply[i])
  end

  EditMultiToggle()
end

--[[ Simple text edit control (text field and keyboard) ]]--

--
-- text edit screen
--
function EditText()
  if SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  local data = {}
  local item = gCurrentMenu.items[gLastRow]

  gLastValue = Settings:get_def(item.param, "")
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "text")

  data["SET_Text_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_Text_Layer.Text_Ctrl.text"] = gLastValue

  data["SET_Screen.activeLayer"] = "SET_Text_Layer"
  History:force_push("SET_Screen")
  gre.set_data(data)
  gre.send_event("swap_menu")
end

--[[ display a TimeStamp (time_t live vars) ]]--

-- display value as yyyyMMdd HH:mm
function DisplayTimeStamp(item)
  local result
  local value = Settings:get(item.param) or 0
  if value > 0 then
    return { date = format_date(value), time = format_time(value) }
  else
    return { value = "N/A" }
  end
end

--[[ Action parameter (send event to execute the action) ]]--

--
-- activate an action parameter
--
function PerformAction()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  local param = gCurrentMenu.items[gLastRow].param
  local prompt = gCurrentMenu.items[gLastRow].prompt
  if prompt then
    local dlg = DialogBox.new(DIALOG_TYPE_CONFIRMATION)
    dlg:set_message(i18n:get(prompt))
    dlg:add_button(i18n:get("NLS_NO"))
    dlg:add_button(i18n:get("NLS_YES"), function() Event:action(param) end)
    dlg:show()
  else
    Event:action(param)
  end
end

--[[ Enter pincode to elevate user role ]]--

--
-- activate a role elevation parameter
--
function ElevateRole()
  table.remove(gYOffsetStack)

  local param = gCurrentMenu.items[gLastRow].param
  local roles = {
    ["ELEVATE_MANAGER"] = MANAGER_ROLE,
    ["ELEVATE_SERVICE"] = SERVICE_ROLE
  }
  LoginDialog_Init("SET_Screen", roles[param] or END_USER_ROLE)
end

--[[ Reference cook action ]]--

--
-- open recipes list for reference cook
--
function ReferenceCook()
  if SettingsAreReadOnly() then return end

  REC_SetReferenceCook()
  gre.set_value("toScreen", "REC_Screen")
  gre.send_event("switch_screen")
end

--[[ Bit table ]]--

local BIT_TABLE = "SET_BitTable_Layer.Bit_Table"
local BIT_TABLE_SCROLLBAR = "SET_BitTable_Layer.Scrollbar"

local gBitTableToggleOffPos = 352
local gBitTableToggleOnPos = 409
local gBitTableScrollbar = Scrollbar.new{table=BIT_TABLE,scrollbar=BIT_TABLE_SCROLLBAR}

--
-- open bit table layer and send io_bittable_get event
--
function EditBitTable()
  local item = gCurrentMenu.items[gLastRow]

  if not item.isReadOnly and SettingsAreReadOnly() then
    table.remove(gYOffsetStack)
    return
  end

  gLastValue = Settings:get(item.param) or ""
  gCurrentMenu = gMenuStack[#gMenuStack]
  table.insert(gMenuStack, "text")

  local data = {}
  data["SET_Screen.activeLayer"] = "SET_BitTable_Layer"
  data["SET_BitTable_Layer.Title_Text.text"] = i18n:get(item.param)
  data["SET_BitTable_Layer.button_group.grd_hidden"] = item.isLive
  data["SET_BitTable_Layer.Toggle_Header.grd_hidden"] = item.isReadOnly
  data[BIT_TABLE..".grd_height"] = item.isLive and 635 or 575
  data[BIT_TABLE..".grd_hidden"] = true
  data[BIT_TABLE..".toggle_alpha"] = not item.isReadOnly and 255 or 0
  data[BIT_TABLE_SCROLLBAR..".Background_Img.grd_height"] = item.isLive and 635 or 575
  data[BIT_TABLE_SCROLLBAR..".grd_hidden"] = true
  gre.set_data(data)
  gre.set_table_attrs(BIT_TABLE, { rows = 0, yoffset = 0 })

  History:force_push("SET_Screen")
  gre.send_event("swap_menu")

  Event:get_bittable_format(item.param)
end

--
-- io_bittable_reply event received
--
function CBSET_InitBitTable(mapargs)
  local item = gCurrentMenu.items[gLastRow]
  gLastValue = Settings:get(item.param) or 0
  gNewValue = gLastValue
  local bit_table = split_string(mapargs.context_event_data.reply, DELIMITER_TOKEN, true)

  local param = bit_table[1]
  if param ~= item.param then return end

  local data = {}
  local index = 1
  for i=2,#bit_table do
    if bit_table[i] ~= "" then
      local bit_index = i - 2
      local is_active = bit32.extract(gLastValue, bit_index) == 1
      data[BIT_TABLE..".text."..index..".1"] = i18n:get(bit_table[i])
      data[BIT_TABLE..".index."..index..".1"] = bit_index
      data[BIT_TABLE..".toggle_pos."..index..".1"] = is_active and gBitTableToggleOnPos or gBitTableToggleOffPos
      data[BIT_TABLE..".ro_active_alpha."..index..".1"] = (item.isReadOnly and is_active) and 255 or 0
      data[BIT_TABLE..".ro_inactive_alpha."..index..".1"] = (item.isReadOnly and not is_active) and 255 or 0
      index = index + 1
    end
  end
  data[BIT_TABLE..".grd_hidden"] = (index == 1)
  gre.set_data(data)
  gre.set_table_attrs(BIT_TABLE, { rows = (index - 1) })

  gBitTableScrollbar:init()
end

--
-- live variable changed, update bittable
--
function UpdateBitTable()
  local item = gCurrentMenu.items[gLastRow]
  gLastValue = Settings:get(item.param) or 0
  gNewValue = gLastValue
  local count = gre.get_table_attrs(BIT_TABLE, "rows").rows

  local data = {}
  for i=1,count do
    local bit_index = gre.get_value(BIT_TABLE..".index."..i..".1")
    local is_active = bit32.extract(gLastValue, bit_index) == 1
    data[BIT_TABLE..".toggle_pos."..i..".1"] = is_active and gBitTableToggleOnPos or gBitTableToggleOffPos
    data[BIT_TABLE..".ro_active_alpha."..i..".1"] = (item.isReadOnly and is_active) and 255 or 0
    data[BIT_TABLE..".ro_inactive_alpha."..i..".1"] = (item.isReadOnly and not is_active) and 255 or 0
  end
  gre.set_data(data)
end

--
-- update scrollbar position
--
function CBSET_SyncBitTableScrollbar()
  gBitTableScrollbar:scroll()
end

--[[ Loading and storing settings to/from USB ]]--

--
-- load settings from a usb stick
--
function LoadSettingsFromUsb()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  Event:list_dir(settings_dir_usb())
end

--
-- store settings on a usb stick
--
function StoreSettingsToUsb()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  CBSET_EnterFileName_OnShowOverlay("P", 7, function(filename)
    CBSET_ActionProgress_OnShowOverlay(gCurrentMenu.items[gLastRow].param, "EXIST_PARAM_USB", filename..".json")
  end)
end

--
-- received list of settings files on usb stick
--
function CBSET_OnSettingsListDirUsb(files)
  -- strip '.json' from file names
  local file_names = {}
  for _, file in ipairs(files) do
    if string.sub(file, 1, 1) == "P" then
      table.insert(file_names, string.sub(file, 1, -6))
    end
  end

  if #file_names > 0 then
    CBSET_SelectFile_OnShowOverlay(file_names, function(filename)
      CBSET_ActionProgress_OnShowOverlay(gCurrentMenu.items[gLastRow].param, nil, filename..".json")
    end)
  else
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get("NLS_USB_NO_DIRECTORY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  end
end

--[[ Loading and storing recipes to/from USB ]]--

--
-- load recipes from a usb stick
--
function LoadRecipesFromUsb()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  Event:list_dir(recipes_dir_usb(), 1)
end

--
-- store settings on a usb stick
--
function StoreRecipesToUsb()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  CBSET_EnterFileName_OnShowOverlay("R", 7, function(directory)
    CBSET_ActionProgress_OnShowOverlay(gCurrentMenu.items[gLastRow].param, "EXIST_RECIPE_USB", directory)
  end)
end

--
-- received list of recipes directories on usb stick
--
function CBSET_OnRecipesListDirUsb(folders)
  -- remove current (.) and previous (..) folders from list
  local dirs = {}
  for _, dir in ipairs(folders) do
    if dir ~= "." and dir ~= ".." and string.sub(dir, 1, 1) == "R" then
      table.insert(dirs, dir)
    end
  end

  if #dirs > 0 then
    CBSET_SelectFile_OnShowOverlay(dirs, function(directory)
      CBSET_ActionProgress_OnShowOverlay(gCurrentMenu.items[gLastRow].param, nil, directory)
    end)
  else
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get("NLS_USB_NO_DIRECTORY"))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  end
end

--
-- store HACCP logging on a usb stick
--
function StoreHACCPToUsb()
  table.remove(gYOffsetStack)
  if SettingsAreReadOnly() then return end

  CBSET_ActionProgress_OnShowOverlay(gCurrentMenu.items[gLastRow].param, "EXIST_HACCP_USB", "LOGGING")

end
