----------------------------------------------------------------------------
-- implementation of temperature class to handle conversions between C and F
----------------------------------------------------------------------------

--[[ conversion functions to convert between C and F ]]--

-- convert celcius to fahrenheit
function CtoF(c)
  return ((c + 40) * 1.8) - 40
end

-- convert fahrenheit to celcius
function FtoC(f)
  return ((f + 40) / 1.8) - 40
end

-- convert an offset in celcius to an offset in fahrenheit
function offsetCtoF(c)
  return (c * 1.8)
end

-- convert an offset in fahrenheit to an offset in celcius
function offsetFtoC(f)
  return (f / 1.8)
end

-- convert celcius to kelvin
function CtoK(c)
  return c + 273
end

-- convert kelvin to celcius
function KtoC(k)
  return k - 273
end

-- convert fahrenheit to kelvin
function FtoK(f)
  return CtoK(FtoC(f))
end

-- convert kelvin to fahrenheit
function KtoF(k)
  return CtoF(KtoC(k))
end

--[[ Temperature class implementation ]]--

local TEMP_UNIT_KEY = "TEMPERATURE_UNIT" -- temperature unit key in the settings database
local TEMP_UNIT_C = 0                    -- 0 means celcius
local TEMP_UNIT_F = 1                    -- 1 means fahrenheit

-- Temperature class
Temperature = {}
Temperature.__index = Temperature

-- Temperature.temp = 0

--
-- constructor, create a new Temperature instance
--
function Temperature.new(o)
  o = o or { temp = 0 }
  setmetatable(o, Temperature)
  return o
end

--
-- check if temperature unit is C or F (returns true for C, false for F)
--
function Temperature.isC()
  return Settings:get(TEMP_UNIT_KEY) == TEMP_UNIT_C
end

--
-- get the correct temperature unit symbol
--
function Temperature.unit(display_unit)
  return DEGREE_SYMBOL .. (display_unit and (Temperature.isC() and "C" or "F") or "")
end

--
-- set the temperature value in C
--
function Temperature:setC(c)
  self.temp = c
end

--
-- set the temperature value in F
--
function Temperature:setF(f)
  self.temp = FtoC(f)
end

--
-- set the temperature value in K
--
function Temperature:setK(k)
  self.temp = KtoC(k)
end

--
-- get the temperature in C
--
function Temperature:getC()
  return self.temp
end

--
-- get the temperature in F
--
function Temperature:getF()
  return CtoF(self.temp)
end

--
-- get the temperature in K
--
function Temperature:getK()
  return CtoK(self.temp)
end

--
-- set the temperature value, unit (C or F) depends on settings
--
function Temperature:set(temp)
  if Temperature.isC() then
    self:setC(temp)
  else
    self:setF(temp)
  end
end

--
-- get the temperature in the unit from the settings
--
function Temperature:get()
  if Temperature.isC() then
    return self:getC()
  else
    return self:getF()
  end
end

--
-- set the temperature offset value in C
--
function Temperature:setOffsetC(c)
  self.temp = c
end

--
-- set the temperature offset value in F
--
function Temperature:setOffsetF(f)
  self.temp = offsetFtoC(f)
end

--
-- get the temperature offset in C
--
function Temperature:getOffsetC()
  return self.temp
end

--
-- get the temperature offset in F
--
function Temperature:getOffsetF()
  return offsetCtoF(self.temp)
end

--
-- set the temperature offset value, unit (C or F) depends on settings
--
function Temperature:setOffset(temp)
  if Temperature.isC() then
    self:setOffsetC(temp)
  else
    self:setOffsetF(temp)
  end
end

--
-- get the temperature in the unit from the settings
--
function Temperature:getOffset()
  if Temperature.isC() then
    return self:getOffsetC()
  else
    return self:getOffsetF()
  end
end

--
-- get the temperature in the unit from the settings rounded to the nearest decimal
--
function Temperature:get_rounded()
  return math.floor(self:get() + 0.5)
end

--
-- convert the temperature to a text string
--
function Temperature:to_string(display_unit)
  return string.format("%d%s", self:get_rounded(), Temperature.unit(display_unit))
end
