-------------------------
-- time utility functions
-------------------------

local DSEC = 24 * 60 * 60      -- seconds in a day
local YSEC = 365 * DSEC        -- seconds in a year
local LSEC = YSEC + DSEC       -- seconds in a leap year
local FSEC = (4 * YSEC) + DSEC -- seconds in a 4-year interval
local BASE_YEAR = 1970         -- 1970 is the base year

local gDays   = { -1, 30, 58, 89, 119, 150, 180, 211, 242, 272, 303, 333, 364 } -- month days sum (-1) in a normal year
local gLpDays = { -1, 30, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 } -- month days sum (-1) in a leap year

local gNow = 0

--
-- set the current time
--
function set_time(epoch_time)
  gNow = epoch_time
end

--
-- get seconds since epoch
--
function epoch_time()
  return os.time()
end

--
-- get seconds since epoch
--
function now()
  return gNow or os.time()
end

--
-- get time in: years, months, days, hours, minutes, seconds
--
function utc_time(seconds_since_epoch)
  local y, j, m, d, h, min, s
  local mdays = gDays
  
  s = seconds_since_epoch
  
  -- year
  y = math.floor(s / FSEC)
  s = s - (y * FSEC)
  y = (y * 4) + BASE_YEAR      -- 1970, 1974, 1978, ...
  if s >= YSEC then
    y = y + 1                  -- 1971, 1975, 1979, ...
    s = s - YSEC
    
    if s >= YSEC then
      y = y + 1                -- 1972, 1976, 1980, ... (leap years)
      s = s - YSEC
      
      if s >= LSEC then
        y = y + 1              -- 1971, 1975, 1979, ...
        s = s - LSEC
      else
        mdays = gLpDays        -- leap year
      end
    end
  end
  
  -- month
  j = math.floor(s / DSEC)
  s = s - (j * DSEC)
  local m = 1
  while mdays[m] < j do m = m + 1 end
  m = m - 1
  
  -- day
  d = j - mdays[m]
  
  -- hours, minutes, seconds
  h = math.floor(s / 3600)
  s = s - (h * 3600)
  min = math.floor(s / 60)
  s = s - (min * 60)
  
  return y, m, d, h, min, s
end

--
-- day number from date, month is zero base
--
local function count_days(y, m, d)
  local mm = math.mod(math.mod(m, 12) + 10, 12)
  local yy = y + math.floor(m / 12) - math.floor(mm / 10)
  return (365 * yy) + math.floor(yy / 4) - math.floor(yy / 100) + math.floor(yy / 400)
         + math.floor(((mm * 306) + 5) / 10) + d - 307
end

--
-- convert timestamp in utc time to epoch time
--
function utc_to_epoch(y, m, d, h, min, s)
  return ((count_days(y, m - 1, d) - count_days(BASE_YEAR, 0, 1)) * DSEC) + ((((h * 60) + min) * 60) + s)
end

--
-- convert timestamp in local time to epoch time
--
function local_to_epoch(y, m, d, h, min, s)
  return utc_to_epoch(y, m, d, h, min, s) - (Settings:get_def("TIMEZONE_OFFSET", 0) * 3600)
end

--
-- as utc_time get time in: years, months, days, hours, minutes, seconds
-- but corrected for the local timezone offset   
--
function local_time(seconds_since_epoch)
  local offset = Settings:get("TIMEZONE_OFFSET") or 0
  local seconds_local = (seconds_since_epoch or now()) + (offset * 3600) 
  return utc_time(seconds_local)
end

--
-- format utc_time in seconds since epoch as local_time in 24h or am-pm
--
function format_time(seconds_since_epoch)
  local _, _, _, h, min, _ = local_time(seconds_since_epoch or now())
  local timestamp = string.format("%01d:%02d", h, min)
  local ampm = nil
  if Settings:get("TIME_FORMAT") == 1 then
    ampm = i18n:get("NLS_AM")
    if h == 0 then
      h = 12
      ampm = i18n:get("NLS_AM")
    elseif h == 12 then
      ampm = i18n:get("NLS_PM")
    elseif h > 12 then
      h = h - 12
      ampm = i18n:get("NLS_PM")
    else
      ampm = i18n:get("NLS_AM")
    end
    timestamp = string.format("%01d:%02d", h, min)
  end
  return timestamp, ampm
end

--
-- format utc_time in seconds since epoch as local date in DMY or MDY
--
function format_date(seconds_since_epoch)
  local y, m, d, _, _, _ = local_time(seconds_since_epoch or now())
  if Settings:get("DATE_FORMAT") == 0 then
    return string.format("%01d/%01d/%04d", d, m, y)
  else
    return string.format("%01d/%01d/%04d", m, d, y)
  end
end

--
-- get a timestamp
--
function make_timestamp(seconds_since_epoch)
  return string.format("%04d%02d%02dT%02d%02d%02d", utc_time(seconds_since_epoch or epoch_time()))
end
