-------------------------
-- some utility functions
-------------------------

--[[ application utility functions ]]--

--
-- return value depending on whether it is nil or not
--
function if_not_nil_else(value, nil_value)
  if value ~= nil then
    return value
  else
    return nil_value
  end
end

--
-- convert array of recipe steps time from minutes to total hours and minutes
--
function calculate_time(steps, offset)
  local hours = 0
  local minutes = offset or 0

  for _, step in pairs(steps) do
    minutes = minutes + step.time
  end

  while minutes >= 60 do
    hours = hours + 1
    minutes = minutes - 60
  end

  return hours, minutes
end

--
-- convert array of recipe steps time from minutes tot total hours, minutes and seconds
--
function calculate_time_seconds(steps, offset)
  local hours = 0
  local minutes = 0
  local seconds = offset or 0

  for _, step in pairs(steps) do
    seconds = seconds + (step.time * 60)
  end

  while seconds >= 60 do
    minutes = minutes + 1
    seconds = seconds - 60
  end

  while minutes >= 60 do
    hours = hours + 1
    minutes = minutes - 60
  end

  return hours, minutes, seconds
end

--
-- forward an overlay touch event to the table below it
--
--- @param gre#context mapargs
function forward_touch_to_table(mapargs, table, callback)
  -- get required information to calculate active cell
  local overlay_attrs = gre.get_control_attrs(mapargs.context_control, "x", "y")
  local table_attrs = gre.get_table_attrs(table, "x", "y", "height", "width", "yoffset", "rows")
  local cell_attrs = gre.get_table_cell_attrs(table, 1, 1, "height")

  -- convert global coordinates to table coordinates
  local x = mapargs.context_event_data.x - table_attrs.x
  local y = mapargs.context_event_data.y - table_attrs.y - table_attrs.yoffset

  -- if touch event within table bounds continue
  if x >= 0 and x <= table_attrs.width and y >= (-1 * table_attrs.yoffset) and y <= (table_attrs.height - table_attrs.yoffset) then
    for i = 1, table_attrs.rows do
      local cell_y = (i - 1) * cell_attrs.height
      if y >= cell_y and y <= (cell_y + cell_attrs.height) then
        callback({context_row = i})
        break
      end
    end
  end
end

--
-- calculate the table cell index at a given y-coordinate
--
--- @param gre#context mapargs
function calculate_table_cell_index(table, touch_y)
  -- get required information to calculate active cell
  local table_attrs = gre.get_table_attrs(table, "x", "y", "height", "width", "yoffset", "rows")
  local cell_attrs = gre.get_table_cell_attrs(table, 1, 1, "height")

  -- convert global coordinates to table coordinates
  local y = touch_y - table_attrs.y - table_attrs.yoffset

  -- if touch event within table bounds continue
  if y >= (-1 * table_attrs.yoffset) and y <= (table_attrs.height - table_attrs.yoffset) then
    for i = 1, table_attrs.rows do
      local cell_y = (i - 1) * cell_attrs.height
      if y >= cell_y and y <= (cell_y + cell_attrs.height) then
        return i
      end
    end
  end

  return -1
end

--[[ table utility functions ]]--

--
-- check if a variable is a table
--
function is_table(object)
  return type(object) == "table"
end

--
-- convert an object to a table
--
function convert_to_table(object)
  return is_table(object) and object or { object }
end

--
-- count the number of elements in a table
--
function table_count(table)
  local count = 0
  for _ in pairs(table) do
    count = count + 1
  end
  return count
end

--
-- make a deep copy of a table
--
function table_copy(orig)
  local orig_type = type(orig)
  local copy
  if orig_type == 'table' then
    copy = {}
    for orig_key, orig_value in pairs(orig) do
      copy[table_copy(orig_key)] = table_copy(orig_value)
    end
    setmetatable(copy, table_copy(getmetatable(orig)))
  else -- number, string, boolean, etc
    copy = orig
  end
  return copy
end

--
-- convert a table to its string representation
--
function table_print(tt, indent, done)
  done = done or {}
  indent = indent or 0
  if type(tt) == "table" then
    local sb = {}
    for key, value in pairs(tt) do
      table.insert(sb, string.rep(" ", indent)) -- indent it
      if type(value) == "table" and not done[value] then
        done[value] = true
        if "number" ~= type(key) then
          table.insert(sb, key.." = ")
        end
        table.insert(sb, "{\n");
        table.insert(sb, table_print(value, indent + 2, done))
        table.insert(sb, string.rep(" ", indent)) -- indent it
        table.insert(sb, "}\n");
      elseif "number" == type(key) then
        table.insert(sb, string.format("\"%s\"\n", tostring(value)))
      else
        table.insert(sb, string.format("%s = \"%s\"\n", tostring(key), tostring(value)))
      end
    end
    return table.concat(sb)
  else
    return tt.."\n"
  end
end

--
-- check if an item exists in a table
--
function in_array(item, array)
  for _, v in pairs(array) do
    if v == item then return true end
  end
  return false
end

--
-- get the item index in a table
--
function index_of(item, array)
  for i, v in ipairs(array) do
    if v == item then return i end
  end
  return 0
end

--[[ string utility functions ]]--

--
-- split a string and return a table with the resulting substrings
--
function split_string(str, delimiter, allowempty)
  local result = {}
  for match in string.gmatch(str, string.format("[^%s]+", delimiter)) do
    if match ~= nil and (allowempty or match ~= "") then
      table.insert(result, match)
    end
  end
  return result
end

--
-- check if string X starts with string Y
--
function starts_with(str, start)
   return str:sub(1, #start) == start
end

--[[ file utility functions ]]--

--
-- check if a file exists
-- return true if the file can be opened for reading, return false if not
--
function file_exists(filepath)
  local f, e = io.open(filepath, "r")
  if f then f:close() return true else return false end
end

--
-- show the cache of given pool
-- pool_name    The name of the resource pool: "image" or "font"
--
function show_cache(pool_name)
    print("show_cache of " .. pool_name )
    local data = gre.walk_pool(pool_name)
    local total = 0
    for k,v in pairs(data) do
        print("  ".. tostring(k) .. "=" .. tostring(v))
        total = total + v
    end
    print("Cache total =" .. tostring(total))
    print("")
end

--
-- show the cache memory being used by a given pool
-- pool_name    The name of the resource pool: "image" or "font"
--
function show_cache_memory_used(pool_name)
  local data = gre.walk_pool(pool_name)
  local total = 0
  for k,v in pairs(data) do
      total = total + v
  end
  print(" ".. tostring(pool_name) .. " cache memory used:" .. tostring(total) .. " bytes")
end

--
-- show the memory statistic
--  useless on target (and even on windows)
--
function show_memstat()
    print("show_memstat")
    local data = gre.env("mem_stats")

    for k,v in pairs(data) do
        print("  ".. tostring(k) .. "=" .. tostring(v))
    end
    print("")
end

--
-- show the contents of the settings database
--
function show_settings()
  print("show_settings")
  Settings:sorted_print()
  print("")
end
--
-- on io_debug_show event
--
-- @param gre#context mapargs
function CB_OnDebugShow(mapargs)
   print("==============================================")
   show_cache("font")
   show_cache("image")
   print("==============================================")
   show_settings()
   print("==============================================")
end

--[[ image utilities ]]--

--
-- convert a 32-bit in big-endian format from a byte array to an integer
--
local function uint32be_from_bytes(s, p)
    local a, b, c, d = s:byte(p, p + 3)
    return (a * 0x1000000) + (b * 0x10000) + (c * 0x100) + d
end

--
-- get the width and height of a png file
--
function png_dimensions(image_file)
  -- try opening the image file
  local file, _ = io.open(image_file, "rb")
  if not file then return nil, nil end

  -- check if the file is a png file
  local header = file:read(256)
  if not header then file:close() return nil, nil end
  local ok, err = file:seek("set")
  if not ok then file:close() return nil, nil end
  if not header:find("^\137PNG\13\10\26\10") then file:close() return nil, nil end

  -- try reading the width and height from its header
  ok, err = file:seek("set", 12)
  if not ok then file:close() return nil, nil end
  local buf = file:read(4)
  if not buf or buf:len() ~= 4 or buf ~= "IHDR" then file:close() return nil, nil end
  buf = file:read(8)
  if not buf or buf:len() ~= 8 then file:close() return nil, nil end
  local width, height = uint32be_from_bytes(buf, 1), uint32be_from_bytes(buf, 5)
  file:close()
  return width, height
end
