-------------------------------------------------------------------------
-- implementation of volume class to handle conversions between L and Gal
-------------------------------------------------------------------------

--[[ conversion functions to convert between L and Gal ]]--

local L_TO_GAL_FACTOR = 0.2641720523581 -- Gal = L * factor

-- convert liters to gallons
function LToGal(l)
  return l * L_TO_GAL_FACTOR
end

-- convert gallons to liters
function GalToL(gal)
  return gal / L_TO_GAL_FACTOR
end

--[[ Volume class implementation ]]--

local VOLUME_UNIT_KEY = "VOLUME_UNIT"   -- volume unit key in the settings database
local VOLUME_UNIT_L = 0                 -- 0 means liter
local VOLUME_UNIT_GAL = 1               -- 1 means gallon

-- Volume class
Volume = {}
Volume.__index = Volume

-- Volume.v = 0

--
-- constructor, create a new Volume instance
--
function Volume.new(o)
  o = o or { v = 0 }
  setmetatable(o, Volume)
  return o
end

--
-- check if volume unit is L or Gal (returns true for L, false for Gal)
--
function Volume.isL()
  return Settings:get(VOLUME_UNIT_KEY) == VOLUME_UNIT_L
end

--
-- set the volume value in L
--
function Volume:setL(l)
  self.v = l
end

--
-- set the volume value in Gal
--
function Volume:setGal(gal)
  self.v = GalToL(gal)
end

--
-- get the volume in L
--
function Volume:getL()
  return self.v
end

--
-- get the volume in Gal
--
function Volume:getGal()
  return LToGal(self.v)
end

--
-- set the volume value, unit (L or Gal) depends on settings
--
function Volume:set(v)
  if Volume.isL() then
    self:setL(v)
  else
    self:setGal(v)
  end
end

--
-- get the volume in the unit from the settings
--
function Volume:get()
  if Volume.isL() then
    return self:getL()
  else
    return self:getGal()
  end
end
