-------------------------
-- wifi network selection
-------------------------

require("Dialog")
require("events")
require("language")
require("RoleManager")
require("Scrollbar")
require("Utf8")
require("Util")

-- TODO [EventGenerator]
-- local event_generator = require("EventGenerator")

local NLS_CUSTOM_SSID_KEY = "NLS_SSID_CUSTOM"
local NLS_NO_WIFI_SELECTED_KEY = "NLS_NO_WIFI_SELECTED"
local NLS_WIFI_STORED_KEY = "NLS_WIFI_STORED"
local WIFI_LAYER = "WifiNetworkSelection_Layer"
local SSID_TABLE = WIFI_LAYER..".ssids_group.ssids_table"
local SSID_TABLE_SCROLLBAR = WIFI_LAYER..".Scrollbar"
local SLIDER_OVERLAY = WIFI_LAYER..".ssids_group.slider_overlay"
local SSIDPOPUP_LAYER = "CustomSsid_Layer"
local PASSWORDPOPUP_LAYER = "EnterPassword_Layer"
local KEYBOARD_LAYER = "Keyboard_Layer_v2"
local ACTIVE_TEXT_COLOR = 0xFFFFFF
local NORMAL_TEXT_COLOR = 0x000000
local ACTIVE_IMAGE_ALPHA = 255
local NORMAL_IMAGE_ALPHA = 0
local ACTIVE_SEGMENT_ALPHA = 255
local HIDDEN_SEGMENT_ALPHA = 0
local SLIDER_IMAGE_HEIGHT = 40
local ALIGN_CENTER = 2
local ALIGN_RIGHT = 3

local gScrollbar = Scrollbar.new{table=SSID_TABLE,scrollbar=SSID_TABLE_SCROLLBAR}
local gActiveLanguage = 0
local gSelectedIndex = 0
local gSliderIndex = 0
local gSliderY = 0
local gSsidCount = 0
local gSsidList = {}
local gAttemptedToConnect = false
local gEnteringCustomSsid = false
local gEnteringPassword = false
local gCustomSsid = ""

local NLS_SCANNING_WIFI_KEY = "NLS_SCANNING_WIFI"
local LOADING_TEXT_LIST = {
  "<p style=\"font-size:24px; text-align:center; color:white\">%s<span style=\"font-size:36px\">.</span>..</p>",
  "<p style=\"font-size:24px; text-align:center; color:white\">%s.<span style=\"font-size:36px\">.</span>.</p>",
  "<p style=\"font-size:24px; text-align:center; color:white\">%s..<span style=\"font-size:36px\">.</span></p>"
}
local gLoadingTextIndex = 1

--
-- password - center if text fits, right align if text does not fit
--
local function UpdatePasswordAlignment()
  local password = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.text")
  local font = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.font")
  local size = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.size")
  local width = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.grd_width") - 20
  local text_width = gre.get_string_size(font, size, password).width
  gre.set_value(PASSWORDPOPUP_LAYER..".password_text.hAlign", (text_width < width) and ALIGN_CENTER or ALIGN_RIGHT)
end

--
-- custom ssid - center if text fits, right align if text does not fit
--
local function UpdateCustomSsidAlignment()
  local ssid = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.text")
  local font = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.font")
  local size = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.size")
  local width = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.grd_width") - 20
  local text_width = gre.get_string_size(font, size, ssid).width
  gre.set_value(SSIDPOPUP_LAYER..".ssid_text.hAlign", (text_width < width) and ALIGN_CENTER or ALIGN_RIGHT)
end

--
-- update scrollbar slider location
--
function CBWifi_SyncScrollbar()
  gScrollbar:scroll()
end

--
-- leave slider area -> cancel slider
--
function CBWifi_OnSliderOutbound(mapargs)
  local data = {}
  data[SLIDER_OVERLAY..".grd_hidden"] = true
  data[SSID_TABLE..".text_color."..gSliderIndex..".1"] = NORMAL_TEXT_COLOR
  data[SSID_TABLE..".text_color."..gSelectedIndex..".1"] = ACTIVE_TEXT_COLOR
  data[SSID_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = ACTIVE_IMAGE_ALPHA
  gre.set_data(data)
end

--
-- release slider -> accept choice if within bounds, otherwise cancel
--
function CBWifi_OnSliderRelease(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(SSID_TABLE, "rows").rows

  gSliderIndex = calculate_table_cell_index(SSID_TABLE, y)

  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBWifi_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[SLIDER_OVERLAY..".grd_hidden"] = true
    data[SSID_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[SSID_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[SSID_TABLE..".active_image_alpha."..gSliderIndex..".1"] = ACTIVE_IMAGE_ALPHA
    gre.set_data(data)

    if gSelectedIndex ~= gSliderIndex then
      if gSelectedIndex == (count - 1) then
        gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = true }) -- hide keyboard
        CBWifi_OnKeyboardConfirm(mapargs) -- cancel editing
      end
      if gSliderIndex == (count - 1) then
        gEnteringCustomSsid = true
        -- show keyboard
        gre.set_layer_attrs(SSIDPOPUP_LAYER, { hidden = false })
        gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
        -- overwrite existing notify text
        gre.set_value(SSIDPOPUP_LAYER..".ssid_text.text", "|")
        UpdateCustomSsidAlignment()
      end
    end

    gSelectedIndex = gSliderIndex
  end
end

--
-- slider motion -> update highlighted item if within bounds, otherwise cancel
--
function CBWifi_OnSliderMotion(mapargs)
  local y = mapargs.context_event_data.y
  local old_slider_index = gSliderIndex
  local count = gre.get_table_attrs(SSID_TABLE, "rows").rows

  gSliderIndex = calculate_table_cell_index(SSID_TABLE, y)

  if gSliderIndex == -1 or gSliderIndex == 1 or gSliderIndex == count then
    gSliderIndex = old_slider_index
    CBWifi_OnSliderOutbound(mapargs)
  else
    local data = {}
    data[SSID_TABLE..".text_color."..old_slider_index..".1"] = NORMAL_TEXT_COLOR
    data[SSID_TABLE..".text_color."..gSliderIndex..".1"] = ACTIVE_TEXT_COLOR
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- grab slider button to start sliding
--
function CBWifi_OnStartSlider(mapargs)
  local i = mapargs.context_row
  local x = mapargs.context_event_data.x
  local y = mapargs.context_event_data.y

  local slider_info = gre.get_control_attrs(SLIDER_OVERLAY, "x", "width")

  if i == gSelectedIndex and x >= slider_info.x and x <= (slider_info.x + slider_info.width) then
    gSliderIndex = i

    local data = {}
    data[SSID_TABLE..".active_image_alpha."..i..".1"] = NORMAL_IMAGE_ALPHA
    data[SLIDER_OVERLAY..".grd_hidden"] = false
    data[SLIDER_OVERLAY..".button_y"] = y - (SLIDER_IMAGE_HEIGHT / 2) - gSliderY
    gre.set_data(data)
  end
end

--
-- select a new wifi SSID
--
function CBWifi_OnSelect(mapargs)
  local i = mapargs.context_row
  local count = gre.get_table_attrs(SSID_TABLE, "rows").rows

  gre.set_value(SLIDER_OVERLAY..".grd_hidden", true)

  if i > 1 and i < count and i ~= gSelectedIndex then
    local data = {}

    -- cancel editing with keyboard
    if gSelectedIndex == (count - 1) then
      -- hide keyboard
      gre.set_layer_attrs(SSIDPOPUP_LAYER, { hidden = true })
      gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = true })
      -- cancel editing
      CBWifi_OnKeyboardConfirm()
    end

    data[SSID_TABLE..".text_color."..gSelectedIndex..".1"] = NORMAL_TEXT_COLOR
    data[SSID_TABLE..".active_image_alpha."..gSelectedIndex..".1"] = NORMAL_IMAGE_ALPHA
    data[SSID_TABLE..".text_color."..i..".1"] = ACTIVE_TEXT_COLOR
    data[SSID_TABLE..".active_image_alpha."..i..".1"] = ACTIVE_IMAGE_ALPHA

    gSelectedIndex = i

    gre.set_data(data)
  end

  if gSelectedIndex == (count - 1) and gre.get_layer_attrs(SSIDPOPUP_LAYER, "hidden").hidden then
    gEnteringCustomSsid = true
    -- show keyboard
    gre.set_layer_attrs(SSIDPOPUP_LAYER, { hidden = false })
    gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
    -- overwrite existing custom ssid
    gre.set_value(SSIDPOPUP_LAYER..".ssid_text.text", "|")
    UpdateCustomSsidAlignment()
  end
end

--
-- password - key pressed
--
local function PasswordInputKeyEvent(mapargs)
  local password = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.text")
  password = (string.sub(password, #password, #password) == "|") and string.sub(password, 1, #password - 1) or password

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    password = Utf8_RemoveLastCharacter(password)
  else
    -- append char
    if Utf8_StrLen(password) < 64 then -- allow a maximum of 64 characters in the custom wifi password string.
      password = password..Utf8_FromUcs2(mapargs.context_event_data.code)
    end
  end

  password = password.."|"
  gre.set_value(PASSWORDPOPUP_LAYER..".password_text.text", password)

  UpdatePasswordAlignment()
end

--
-- custom ssid - key pressed
--
local function CustomSsidInputKeyEvent(mapargs)
  local ssid = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.text")
  ssid = (string.sub(ssid, #ssid, #ssid) == "|") and string.sub(ssid, 1, #ssid - 1) or ssid

  if mapargs.context_event_data.code == 8 then
    -- backspace, remove last char
    ssid = Utf8_RemoveLastCharacter(ssid)
  else
    -- append char
    if Utf8_StrLen(ssid) < 32 then -- allow a maximum of 32 characters in the custom wifi SSID string.
      ssid = ssid..Utf8_FromUcs2(mapargs.context_event_data.code)
    end
  end

  ssid = ssid.."|"
  gre.set_value(SSIDPOPUP_LAYER..".ssid_text.text", ssid)

  UpdateCustomSsidAlignment()
end

--
-- key pressed
--
function CBWifi_InputKeyEvent(mapargs)
  if gEnteringPassword then
    PasswordInputKeyEvent(mapargs)
  elseif gEnteringCustomSsid then
    CustomSsidInputKeyEvent(mapargs)
  end
end

--
-- confirm wifi password
--
local function ConfirmPassword(mapargs)
  -- hide enter password layer
  gre.set_layer_attrs(PASSWORDPOPUP_LAYER, { hidden = true })
  -- remove | from end of password
  local password = gre.get_value(PASSWORDPOPUP_LAYER..".password_text.text")
  password = (string.sub(password, #password, #password) == "|") and string.sub(password, 1, #password - 1) or password
  -- send connect to wifi event
  Event:wifi_connect(gre.get_value(SSID_TABLE..".ssid."..gSelectedIndex..".1"), password)
  gAttemptedToConnect = true
  -- exit screen
  CB_OnBackPressed(mapargs)
end

--
--  confirm custom ssid
--
local function ConfirmCustomSsid()
  -- hide edit custom ssid overlay
  gre.set_layer_attrs(SSIDPOPUP_LAYER, { hidden = true })
  -- remove | from end of custom name
  local ssid = gre.get_value(SSIDPOPUP_LAYER..".ssid_text.text")
  ssid = (string.sub(ssid, #ssid, #ssid) == "|") and string.sub(ssid, 1, #ssid - 1) or ssid
  -- Show the literal custom ssid, not the NLS_CUSTOM_SSID_KEY translation!
  -- Even when it is an empty string (to support fallback on a default SSID when this string is empty.)

  --if ssid ~= "" then
  --  gCustomSsid = ssid
  --else
  --  ssid = gCustomSsid
  --end
  gre.set_value(SSID_TABLE..".ssid."..gSelectedIndex..".1", ssid)
end

--
-- confirm editing
--
function CBWifi_OnKeyboardConfirm(mapargs)
  if gEnteringPassword then
    ConfirmPassword(mapargs)
  elseif gEnteringCustomSsid then
    ConfirmCustomSsid()
  end
end

--
-- cancel editing
--
function CBWifi_OnKeyboardCancel()
  gEnteringPassword = false
  gEnteringCustomSsid = false
  gre.set_layer_attrs(SSIDPOPUP_LAYER, { hidden = true })
  gre.set_layer_attrs(PASSWORDPOPUP_LAYER, { hidden = true })
end

--
-- ok button pressed
--
function CBWifi_OnConfirm(mapargs)
  local count = gre.get_table_attrs(SSID_TABLE, "rows").rows
  if gSelectedIndex <= 1 or gSelectedIndex >= count then
    local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
    dlg:set_message(i18n:get(NLS_NO_WIFI_SELECTED_KEY))
    dlg:add_button(i18n:get("NLS_OK"))
    dlg:show()
  else
    gEnteringPassword = true
    gre.set_value(PASSWORDPOPUP_LAYER..".password_text.text", "|")
    UpdatePasswordAlignment()
    gre.set_layer_attrs(PASSWORDPOPUP_LAYER, { hidden = false })
    gre.set_layer_attrs(KEYBOARD_LAYER, { hidden = false })
  end
end

--
-- cancel button pressed
--
function CBWifi_OnCancel(mapargs)
  gAttemptedToConnect = false
  CB_OnBackPressed(mapargs)
end

--
-- create a step name table entry
--
local function Wifi_SsidTableEntry(table, index, properties)
  table[SSID_TABLE..".keyboard_image_alpha."..index..".1"] = properties.keyboard and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
  table[SSID_TABLE..".ssid."..index..".1"] = properties.ssid or ""
  table[SSID_TABLE..".text_color."..index..".1"] = properties.selected and ACTIVE_TEXT_COLOR or NORMAL_TEXT_COLOR
  table[SSID_TABLE..".active_image_alpha."..index..".1"] = properties.selected and ACTIVE_IMAGE_ALPHA or NORMAL_IMAGE_ALPHA
  table[SSID_TABLE..".top_segment_alpha."..index..".1"] = properties.top_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
  table[SSID_TABLE..".body_segment_alpha."..index..".1"] = properties.body_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
  table[SSID_TABLE..".bottom_segment_alpha."..index..".1"] = properties.bottom_segment and ACTIVE_SEGMENT_ALPHA or HIDDEN_SEGMENT_ALPHA
end

--
-- complete list of SSIDs received
--
local function Wifi_ShowSsidList(mapargs)
  gre.send_event("gra.screen.hold")

  gSliderY = gre.get_control_attrs(SLIDER_OVERLAY, "y").y

  local data = {}

  -- top segment of slider
  local index = 1
  Wifi_SsidTableEntry(data, index, { top_segment = true })

  -- slider content: selectable SSIDs
  for _, ssid in ipairs(gSsidList) do
    index = index + 1
    Wifi_SsidTableEntry(data, index, { ssid = ssid.name, selected = ssid.connected, body_segment = true })
    gSelectedIndex = ssid.connected and index or gSelectedIndex
  end

  -- slider content: custom SSID
  index = index + 1
  gCustomSsid = i18n:get(NLS_CUSTOM_SSID_KEY)
  Wifi_SsidTableEntry(data, index, { keyboard = true, ssid = gCustomSsid, body_segment = true })

  -- bottom segment of slider
  index = index + 1
  Wifi_SsidTableEntry(data, index, { bottom_segment = true })

  gre.set_data(data)
  gre.set_table_attrs(SSID_TABLE, { yoffset = 0, rows = index })
  gre.set_layer_attrs("WIFI_Screen.WifiNetworkSelection_Layer", {hidden = false})
  gre.set_group_attrs("WifiScreenBackground_Layer.loading_group", {hidden = true})

  gScrollbar:init()

  gre.send_event("gra.screen.release")
end

--
-- request list of available wifi networks
--
local function ListWifiNetworks()
  gre.set_layer_attrs("WIFI_Screen.WifiNetworkSelection_Layer", {hidden = true})
  gre.set_group_attrs("WifiScreenBackground_Layer.loading_group", {hidden = false})

  gSelectedIndex = 0
  gCustomSsid = ""
  gEnteringCustomSsid = false
  gEnteringPassword = false
  gAttemptedToConnect = false
  gSsidCount = 0
  gSsidList = {}

  Event:wifi_list()

  -- TODO [EventGenerator]
  -- event_generator.wifi_list(20, 0)
end

--
-- prepare wifi network selection screen
--
function CBWifi_OnScreenShow(mapargs)
  gEnteringCustomSsid = false
  gEnteringPassword = false
  gAttemptedToConnect = false

  -- menu buttons
  Control_SetButtons(true, true, true, true, true, true, false)

  -- check whether user is allowed to modify wifi details
  if RoleManager:can_access(MANAGER_ROLE) then
    ListWifiNetworks()
  else
    LoginDialog_Init("WIFI_Screen", MANAGER_ROLE)
  end

  -- initialize loading animation
  gLoadingTextIndex = 1
  gre.set_data("WifiScreenBackground_Layer.loading_group.loading_text.text", string.format(LOADING_TEXT_LIST[gLoadingTextIndex], i18n:get(NLS_SCANNING_WIFI_KEY)))
end

--
-- leaving wifi network selection screen
--
function CBWifi_OnScreenHide(mapargs)
  if not gAttemptedToConnect then
    Event:wifi_abort_connect()
  end
end

--
-- login required, but close button pressed --> return to previous screen
--
function CBWifi_OnLoginCanceled(mapargs)
  gre.send_event_target("gre.touch", "Icon_Back_Layer.CR_Icon_Back_Img")
end

--
-- role changed, if role lowered, check if new role is allowed to change wifi network
--
function CBWifi_OnRoleChanged(mapargs)
  if RoleManager:can_access(MANAGER_ROLE) then
    ListWifiNetworks()
  else
    LoginDialog_Init("WIFI_Screen", MANAGER_ROLE)
  end
end

--
-- let application know that we are still active (every minute)
--
function CBWifi_OnRefreshElevationTimer(mapargs)
  if RoleManager:can_access(MANAGER_ROLE) then
    RoleManager:kick()
  end
end

--
-- timer that triggers loading animation
--
function CBWifi_LoadingAnimationTimer(mapargs)
  if gre.get_group_attrs("WifiScreenBackground_Layer.loading_group", "hidden").hidden == 0 then
    gLoadingTextIndex = gLoadingTextIndex + 1
    if gLoadingTextIndex > #LOADING_TEXT_LIST then gLoadingTextIndex = 1 end
    gre.set_value("WifiScreenBackground_Layer.loading_group.loading_text.text", string.format(LOADING_TEXT_LIST[gLoadingTextIndex], i18n:get(NLS_SCANNING_WIFI_KEY)))
  end
end

--[[ wifi ssid list events ]]--

--
-- Event handler for: io_wifi_list_reply "1u1 ssid_count"
--
function CBWifi_OnWifiListReply(mapargs)
  gSsidCount = mapargs.context_event_data.ssid_count
end

--
-- Event handler for: io_wifi_list_reply_ssid "1u1 connected 1s0 ssid"
--
function CBWifi_OnSsidReceived(mapargs)
  table.insert(gSsidList, {
    name = mapargs.context_event_data.ssid,
    connected = mapargs.context_event_data.connected == 1
  })

  if table_count(gSsidList) == gSsidCount then
    Wifi_ShowSsidList(mapargs)
  end
end

--
-- Event handler for: io_wifi_data_stored
--
function CBWifi_OnWifiDataStored(mapargs)
  local dlg = DialogBox.new(DIALOG_TYPE_INFORMATION)
  dlg:set_message(i18n:get(NLS_WIFI_STORED_KEY))
  dlg:add_button(i18n:get("NLS_OK"))
  dlg:show()
end
