------------------------------------------------------
-- communication with the backend is done using events
------------------------------------------------------

require("Config")
require("Util")

-- event singleton
Event = {}
Event.__index = Event

-- channel the backend is listening on (set in Config.lua)
Event.channel = EVENT_CHANNEL

-- events we as GUI send to backend.
local evt_ready                 = { name = "io_ready"                                                                                            }
local evt_startupdelay          = { name = "io_startupdelay"                                                                                     }
local evt_getrecipes            = { name = "io_get_recipes"                                                                                      }
local evt_getparameters         = { name = "io_get_parameters"                                                                                   }
local evt_listdir               = { name = "io_list_dir"             , format = "1u1 listdirs 1s0 dir"                                           }
local evt_startrecipe           = { name = "io_start_recipe"         , format = "4u1 starttime 1u1 reference_cook 1s0 name"                                    }
local evt_settemperature        = { name = "io_set_temperature"      , format = "2u1 temp"                                                       }
local evt_deleterecipe          = { name = "io_delete_recipe"        , format = "1s0 name"                                                       }
local evt_createrecipe          = { name = "io_create_recipe"        , format = "1u1 steps 1s0 data"                                             } -- with data = name+icon
local evt_recipestep            = { name = "io_recipe_step"          , format = "2u1 temp 2u1 time 1u%d params 1s0 name"                         }
local evt_updaterecipe          = { name = "io_update_recipe"        , format = "1u1 steps 1u1 rename 1s0 data"                                  } -- with data = rename == 1 ? oldname+newname+icon : name+icon
local evt_sortindexcount        = { name = "io_sort_index_count"     , format = "1u1 count"                                                      }
local evt_updatesortindex       = { name = "io_update_sort_index"    , format = "1u1 sort_index 1s0 name"                                        }
local evt_setparameter          = { name = "io_set_parameter"        , format = "2s1 value 1s0 id"                                               }
local evt_setparameter_str      = { name = "io_set_parameter_str"    , format = "1s0 data"                                                       } -- with data = id+value
local evt_wifireply             = { name = "io_wifi_reply"           , format = "1u1 approved 1s0 device"                                        }
local evt_wifilist              = { name = "io_wifi_list"                                                                                        }
local evt_wificonnect           = { name = "io_wifi_connect"         , format = "1u1 ssid_size 1s0 ssid_password"                                } -- with ssid_size = number of bytes in ssid, ssid_password = ssid+password
local evt_wifinoconnect         = { name = "io_wifi_no_connect"                                                                                  }
local evt_pauserecipe           = { name = "io_recipe_pause"                                                                                     }
local evt_cancelrecipe          = { name = "io_recipe_cancel"                                                                                    }
local evt_msg_reply             = { name = "io_msg_reply"            , format = "2u1 id 1u1 approved"                                            }
local evt_action                = { name = "io_action"               , format = "1s0 payload"                                                    } -- with payload = id+value
local evt_elevationrequest      = { name = "io_elevationrequest"     , format = "2u1 pin 1s0 role"                                               }
local evt_kickrolemanager       = { name = "io_rolemanager_kick"                                                                                 }
local evt_disabletouch          = { name = "io_touch_disable"                                                                                    }
local evt_enabletouch           = { name = "io_touch_enable"                                                                                     }
local evt_errorlog              = { name = "io_errorlog"                                                                                         }
local evt_standby               = { name = "io_standby"                                                                                          }
local evt_wakeup                = { name = "io_wakeup"                                                                                           }
local evt_live_get              = { name = "io_live_get"             , format = "1s0 ids"                                                        }
local evt_live_set              = { name = "io_live_set"             , format = "4u1 value 1s0 id"                                               }
local evt_reset_burner          = { name = "io_reset_burner"                                                                                     }
local evt_iotest_start          = { name = "io_iotest_start"                                                                                     }
local evt_iotest_end            = { name = "io_iotest_end"                                                                                       }
local evt_rotor_start           = { name = "io_rotor_start"                                                                                      }
local evt_rotor_stop            = { name = "io_rotor_stop"                                                                                       }
local evt_light_on              = { name = "io_light_start"                                                                                      }
local evt_light_off             = { name = "io_light_stop"                                                                                       }
local evt_audioprog             = { name = "io_audioprog"            , format = "1u1 prognr"                                                     }
local evt_inited                = { name = "io_inited"                                                                                           }
local evt_canstart_recipe       = { name = "io_canstart_recipe"      , format = "1s0 name"                                                       }
local evt_canstart_clean        = { name = "io_canstart_clean"       , format = "1u1 type"                                                       }
local evt_bittable_get          = { name = "io_bittable_get"         , format = "1s0 id"                                                         }
local evt_paramoptions          = { name = "io_parameter_options"    , format = "1s0 id"                                                         }
local evt_makescreenshot        = { name = "io_screenshot"                                                                                       }
local evt_sendscreenshot        = { name = "io_sendscreenshot"       , format = "1s0 note"                                                       }
local evt_message_list          = { name = "io_msglist"                                                                                          }
local evt_message_read          = { name = "io_msgread"              , format = "1u1 id"                                                         }
local evt_clean_totaltimes      = { name = "io_clean_totaltimes"                                                                                 }
local evt_commissioning_done    = { name = "io_commissioning_done"                                                                               }

-- events we as GUI receive from backend.
--[[
local evt_startupdelayreply     = { name = "io_startupdelay"         , format = "2u1 delay"                                                      }
local evt_listdirreply          = { name = "io_listdir_reply"        , format = "1s0 files"                                                      } -- with files = list of files, first entry is the directory
local evt_recipelist            = { name = "io_recipelist"           , format = "2u1 count"                                                      }
local evt_recipe                = { name = "io_recipe"               , format = "1u1 first_use 1u1 sort_index 1u1 calibrated 1u1 steps 1s0 data" } -- with data = name+icon
local evt_recipestep            = { name = "io_recipe_step"          , format = "2u1 temp 2u1 time 1u%d params 1s0 name"                         }
local evt_recipestate           = { name = "io_recipe_state"         , format = "1u1 state 1u1 step"                                             }
local evt_tempupdate            = { name = "io_temperature"          , format = "2u1 temp"                                                       }
local evt_wifirequest           = { name = "io_wifi_request"         , format = "1s0 device"                                                     }
local evt_wifilistreply         = { name = "io_wifi_list_reply"      , format = "1u1 ssid_count"                                                 } -- with ssid_count = number of ssids discovered
local evt_wifilistreply_ssid    = { name = "io_wifi_list_reply_ssid" , format = "1u1 connected 1s0 ssid"                                         }
local evt_wifidatastored        = { name = "io_wifi_data_stored"                                                                                 }
local evt_parameterreply        = { name = "io_parameter_reply"      , format = "2u1 last 2u%d values 1s0 ids"                                   } -- with ids = list of parameter names
local evt_parameterreply_str    = { name = "io_parameter_reply_str"  , format = "1s0 data"                                                       } -- with data = id+value
local evt_pauserecipe           = { name = "io_recipe_pause"                                                                                     }
local evt_resumerecipe          = { name = "io_recipe_resume"        , format = "2u1 time 1u1 state 1u1 step 1s0 name"                           }
local evt_cancelrecipe          = { name = "io_recipe_cancel"                                                                                    }
local evt_error                 = { name = "io_error"                , format = "2u1 device 2u1 code"                                            }
local evt_dialog                = { name = "io_dialog"               , format = "4u1 id 1u1 type 1s0 text"                                       }
local evt_dialog_close          = { name = "io_dialog_close"         , format = "4u1 id"                                                         }
local evt_standby               = { name = "io_standby"                                                                                          }
local evt_proximityalert        = { name = "io_proximity_alert"                                                                                  }
local evt_autocorrect           = { name = "io_autocorrect"          , format = "2s1 delta"                                                      }
local evt_msgreceived           = { name = "io_msg_received"         , format = "2u1 id 2u1 type 1s0 message"                                    }
local evt_elevationgranted      = { name = "io_elevation_granted"    , format = "1s0 role"                                                       }
local evt_elevationdenied       = { name = "io_elevation_denied"                                                                                 }
local evt_errorlogreply         = { name = "io_errorlog_reply"       , format = "4u%d start 4u%d stop 1u%d code 1u%d active 1u%d device"         } -- with %d the number of log entries
local evt_refresh_parameter     = { name = "io_refresh_parameter"                                                                                }
local evt_refresh_recipe        = { name = "io_refresh_recipe"                                                                                   }
local evt_debug_show            = { name = "io_debug_show"                                                                                       }
local evt_live_value            = { name = "io_live_value"           , format = "4u1 value 1s0 id"                                               }
local evt_clean_cancel          = { name = "io_clean_cancel"                                                                                     }
local evt_clean_done            = { name = "io_clean_done"                                                                                       }
local evt_clean_info            = { name = "io_clean_info"           , format = "4u1 totaltime"                                                  }
local evt_clean_door_show       = { name = "io_clean_door_show"      , format = "1u1 flag"                                                       } -- with flag 0(add) or 1(remove)
local evt_clean_door_hide       = { name = "io_clean_door_hide"                                                                                  }
local evt_clean_door_show_rotor = { name = "io_clean_door_show_rotor"                                                                            }
local evt_clean_passed          = { name = "io_clean_passed"         , format = "4u1 totalpassed"                                                }
local evt_clean_step            = { name = "io_clean_step"           , format = "1u1 stepno 1s0 stepdesc"                                        }
local evt_clean_target_temp     = { name = "io_clean_target_temp"    , format = "2u1 temp"                                                       }
local evt_clean_resume          = { name = "io_clean_resume"         , format = "4u1 timestamp 1s0 name"                                         }
local evt_time                  = { name = "io_time"                 , format = "4u1 epoch"                                                      }
local evt_rotor_start           = { name = "io_rotor_start"                                                                                      }
local evt_rotor_stop            = { name = "io_rotor_stop"                                                                                       }
local evt_canstart_recipe_reply = { name = "io_canstart_recipe_reply", format = "1s1 reply 1s0 info"                                             } -- with info = name+reason nls id
local evt_canstart_clean_reply  = { name = "io_canstart_clean_reply" , format = "1u1 type 1s1 reply 1s0 info"                                    } -- with info = reason nls id
local evt_bittable_reply        = { name = "io_bittable_reply"       , format = "1s0 reply"                                                      }
local evt_criterror             = { name = "io_criterror"            , format = "4u1 moduleid 4u1 extrainfo 4u1 linenumber"                      }
local evt_heatup_start          = { name = "io_heatup_start"         , format = "2u1 heatuptemp"                                                 }
local evt_heatup_done           = { name = "io_heatup_done"          , format = "2u1 heatuptime"                                                 }
local evt_paramoptionsreply     = { name = "io_parameter_options"    , format = "1s0 options"                                                    } -- with options = list of options, first option is parameter
local evt_screenshotreply       = { name = "io_screenshot_reply"     , format = "1u1 ok 1s0 file"                                                }
local evt_sendscreenshotreply   = { name = "io_sendscreenshot_reply" , format = "1u1 ok"                                                         }
local evt_action_state          = { name = "io_action_state"         , format = "1u1 state 1s0 data"                                             } -- with data = action id + optional message nls id
local evt_msglist_reply         = { name = "io_msglist_reply"        , format = "1u%d ids 1s0 dir"                                               }
local evt_clean_totaltimesreply = { name = "io_clean_totaltimes_reply" , format = "2u%u totaltimes 1s0 names" }
local evt_demo_mode_changed     = { name = "io_demo_mode_changed" }
]]--

--
-- error handler
--
-- Note : returns always true (bug Storyboard engine )
--
local function handle_error(success, error)

-- commented out, as 'success' is always nil on the ARM target
--
--  if not success then
--    print(error)
--  end

--  print ("handle_error " .. tostring(success) .. " " .. tostring(error))

  success = true
  return success
end

--
-- send an event without payload
--
function Event:send_event(event)
  return handle_error(gre.send_event(event.name, self.channel))
end

--
-- send an event with payload
--
function Event:send_event_data(event, data)
  return handle_error(gre.send_event_data(event.name, event.format, data, self.channel))
end

--
-- send a ready event
--
function Event:ready()
  return self:send_event(evt_ready)
end

--
-- send an event to request the fr-jado logo startup screen display time
--
function Event:startup_delay()
  return self:send_event(evt_startupdelay)
end

--
-- send an event to request the contents of a directory
--
function Event:list_dir(directory, list_dirs)
  return self:send_event_data(evt_listdir, { listdirs = list_dirs or 0, dir = directory })
end

--
-- send an event to request the recipes database
--
function Event:get_recipes()
  return self:send_event(evt_getrecipes)
end

--
-- send an event to start cooking a recipe
--
function Event:start_recipe(name, reference_cook, starttime)
  return self:send_event_data(evt_startrecipe, { starttime = starttime, reference_cook = (reference_cook and 1 or 0), name = name })
end


--
-- send an event to manually update the temperature
--
function Event:set_temperature(temp)
  return self:send_event_data(evt_settemperature, { temp = (temp * 10.0) })
end

--
-- send a recipe step event (used by create_recipe and update_recipe)
--
function Event:recipe_step(step)
  local params = {}

  table.insert(params, step.preheat and 1               or 0) -- is this the preheating step? [0-1]
  table.insert(params, step.hold    and 1               or 0) -- is this the hold step at the end to keep the oven warm? [0-1]
  table.insert(params, step.probe   and 1               or 0) -- core probe enabled [0-1]
  table.insert(params, step.fan                         or 0) -- fan speed [0-2] or [0-4] or [0-5]
  table.insert(params, step.steam                       or 0) -- steam [0-3] off or *, **, ***
  table.insert(params, step.vent                        or 0) -- vent/exhaust valve: 0 (closed), 1 (open), 5 (open for last 5 minutes)
  table.insert(params, step.drying  and step.drying * 2 or 0) -- drying: 0-9.5 minutes in steps of 0.5, so [0-19]
  table.insert(params, step.custom  and 1               or 0) -- 0 = default step name, 1 = custom step name
  table.insert(params, step.notify  and 1               or 0) -- is this a notification step?

  return gre.send_event_data(evt_recipestep.name, string.format(evt_recipestep.format, table_count(params)), {
    temp = (step.temp * 10.0), time = step.time or 0, params = params, name = step.name
  }, self.channel)
end

--
-- send the events to create a new recipe
--
function Event:create_recipe(name, recipe)
  local success = false
  success = self:send_event_data(evt_createrecipe, {
    steps = #recipe.steps, data = string.format("%s%s%s", name, DELIMITER_TOKEN, recipe.icon or "")
  })
  if not success then return false end
  for _, step in ipairs(recipe.steps) do
    success = self:recipe_step(step)
    if not success then return false end
  end
  return success
end

--
-- send the events to modify an existing recipe
--
function Event:update_recipe(oldname, newname, recipe)
  local success = false
  local data = oldname
  if oldname ~= newname then data = string.format("%s%s%s", data, DELIMITER_TOKEN, newname) end
  data = string.format("%s%s%s", data, DELIMITER_TOKEN, recipe.icon or "")
  success = self:send_event_data(evt_updaterecipe, {
    steps = #recipe.steps, rename = ((oldname ~= newname) and 1 or 0), data = data
  })
  if not success then return false end
  for _, step in ipairs(recipe.steps) do
    success = self:recipe_step(step)
    if not success then return false end
  end
  return success
end

--
-- update the sort order of recipes in the database
--
function Event:update_recipe_sort_order(sorted_names)
  local success = self:send_event_data(evt_sortindexcount, { count = #sorted_names })
  if not success then return false end
  for index, name in ipairs(sorted_names) do
    success = self:send_event_data(evt_updatesortindex, { sort_index = index, name = name })
    if not success then return false end
  end
  return success
end

--
-- send an event to delete a recipe
--
function Event:delete_recipe(name)
  return self:send_event_data(evt_deleterecipe, { name = name })
end

--
-- send an event to update a configuration parameter
--
function Event:change_parameter(id, value)
  return self:send_event_data(evt_setparameter, { value = value, id = id })
end

--
-- send en avent to update a text string configuration parameter
--
function Event:change_parameter_string(id, value)
  return self:send_event_data(evt_setparameter_str, { data = string.format("%s%s%s", id, DELIMITER_TOKEN, value) })
end

--
-- send an event to accept/deny an incoming wifi connections
--
function Event:wifi_reply(device, approved)
  return self:send_event_data(evt_wifireply, { approved = (approved and 1 or 0), device = device })
end

--
-- request list of available wifi networks
--
function Event:wifi_list()
  return self:send_event(evt_wifilist)
end

--
-- connect to a wifi network
--
function Event:wifi_connect(ssid, password)
  return self:send_event_data(evt_wificonnect, { ssid_size = #ssid, ssid_password = ssid..password})
end

--
-- notify the application that we don't want to connect to a new wifi network
--
function Event:wifi_abort_connect()
  return self:send_event(evt_wifinoconnect)
end

--
-- send an event to request a list of the configuration parameters and their values
--
function Event:get_parameters()
  return self:send_event(evt_getparameters)
end

--
-- send an event to pause the currently cooking recipe or cleaning program
--
function Event:pause_recipe()
  return self:send_event(evt_pauserecipe)
end

--
-- send an event to cancel the currently cooking recipe or cleaning program
--
function Event:cancel_recipe()
  return self:send_event(evt_cancelrecipe)
end

-- send an event to perform an action from the settings menu
--
function Event:action(id, value)
  return self:send_event_data(evt_action, { payload = id..DELIMITER_TOKEN..((value == nil) and "." or value) })
end

--
-- send an event to request elevation to a different role
--
function Event:request_elevation(role, pin)
  return self:send_event_data(evt_elevationrequest, { pin = pin, role = role })
end

--
-- let the role manager know we are still active, so it doesn't auto log off
--
function Event:kick_rolemanager()
  return self:send_event(evt_kickrolemanager)
end

--
-- send an event to disable the touch screen (ie polling touch events)
--
function Event:disable_touch()
  return self:send_event(evt_disabletouch)
end

--
-- send an event to enable the touch screen (ie start polling touch events again)
--
function Event:enable_touch()
  return self:send_event(evt_enabletouch)
end

--
-- send an event to request a list of the error log entries
--
function Event:get_errorlog()
  return self:send_event(evt_errorlog)
end

--
-- send an event to request the machine to enter standby mode
--
function Event:request_standby()
  return self:send_event(evt_standby)
end

--
-- tell the device it's time to wake up
--
function Event:wakeup()
  return self:send_event(evt_wakeup)
end

--
-- send an event to request the values of the live vars in the ids list
--
function Event:live_get(ids)
  return self:send_event_data(evt_live_get, { ids = ids })
end

--
-- send an event to set the value of a live var
--
function Event:live_set(id, value)
  return self:send_event_data(evt_live_set, { id = id, value = value })
end

--
-- send an event to get the bittable format
--
function Event:get_bittable_format(id)
  return self:send_event_data(evt_bittable_get, { id = id })
end

--
-- start an I/O Test session
-- we will receive 'io_live_value' events
--
function Event:iotest_start()
  return self:send_event(evt_iotest_start)
end

--
-- end the I/O Test session
--
function Event:iotest_end()
  return self:send_event(evt_iotest_end)
end

--
-- start reset sequence on gas burner
--
function Event:reset_burner()
  return self:send_event(evt_reset_burner)
end

--
-- start the rotor
--
function Event:start_rotor()
  return self:send_event(evt_rotor_start)
end

--
-- stop the rotor
--
function Event:stop_rotor()
  return self:send_event(evt_rotor_stop)
end

--
-- indicate to backend that commissioning via "First Use Wizard" was completed
--
function Event:commissioning_done()
  return self:send_event(evt_commissioning_done)
end

--
-- turn on the light
--
function Event:light_on()
  return self:send_event(evt_light_on)
end

--
-- turn off the light
--
function Event:light_off()
  return self:send_event(evt_light_off)
end

--
-- start a audio program
-- prognr one of 6(cAudio_Prog_TOUCH), 7(cAudio_Prog_GESTURE),  8(cAudio_Prog_IN_MSG)
--               9(cAudio_Prog_SCREENDUMP) or 10 (cAudio_Prog_KEYINCORR)
--
function Event:audioprog(prognr)
  return self:send_event_data(evt_audioprog, { prognr = prognr })
end

function Event:audioprog_touch()
  return Event:audioprog(6)
end

function Event:audioprog_screendump()
  return Event:audioprog(9)
end

function Event:audioprog_keyincorr()
  return Event:audioprog(10)
end

--
-- send a inited event
--
function Event:inited()
  return self:send_event(evt_inited)
end

--
-- ask if recipe is allowed to start
--
function Event:can_start_recipe(name)
  return self:send_event_data(evt_canstart_recipe, { name = name })
end

--
-- ask if cleaning program is allowed to start
--
function Event:can_start_clean(type)
  return self:send_event_data(evt_canstart_clean, { type = type })
end

--
-- get the options for a parameter
--
function Event:parameter_options(id)
  return self:send_event_data(evt_paramoptions, { id = id })
end

--
-- tell the application to make a screenshot
--
function Event:make_screenshot()
  return self:send_event(evt_makescreenshot)
end

--
-- tell the application to send the screenshot to the service
--
function Event:send_screenshot(note)
  return self:send_event_data(evt_sendscreenshot, { note = note })
end

--
-- get the list of messages
--
function Event:get_messages()
  return self:send_event(evt_message_list)
end

--
-- notify the application that a message has been read
--
function Event:mark_message_read(id)
  return self:send_event_data(evt_message_read, { id = id })
end

--
-- get the list clean total times
--
function Event:get_cleantotaltimes()
  return self:send_event(evt_clean_totaltimes)
end


