-----------------------------------------
-- implementation of our virtual keyboard
-----------------------------------------

require("Utf8")

local ALPHA_GROUP = ".qwerty_group"     -- alpha keyboard
local NUMERIC_GROUP = ".numeric_group"  -- numeric keyboard
local SHIFT_KEY = ".qwerty_group.Shift" -- shift (caps lock) key

local gShifted = false                  -- is shift (caps lock) active
local gSwitched = false                 -- is switch (special characters view) active

--
-- explode a string into its parts
--
local function SplitString(inputstr, sep)
  if (sep == nil) then
    sep = "%s"
  end
    
  local t = {}
  local i = 1
  for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
    t[i] = str
    i = i + 1
  end
  return t
end

--
-- when a virtual key is pressed, send an actual keydown event
--
local function TriggerKey(code)
  gre.send_event_data("gre.keydown", "4u1 code 4u1 modifiers", {code = Utf8_ToUcs2(code), modifiers = 0})
end

--
-- when a virtual key is pressed, send an actual keydown event
--
local function TriggerRaw(code)
  gre.send_event_data("gre.keydown", "4u1 code 4u1 modifiers", {code = code, modifiers = 0})
end

--
-- on shift button pressed (works as a caps toggle, like caps lock)
--
--- @param gre#context mapargs
local function KeyboardShift(mapargs)
  gShifted = not gShifted
  gre.set_value(mapargs.context_layer..SHIFT_KEY..".alpha", gShifted and 255 or 0)
end

--
-- switch between alpha and numeric keyboards
--
--- @param gre#context mapargs
local function SwitchAlphaNumeric(mapargs) -- support for old keyboard
  local alpha_group_hidden = gre.get_group_attrs(mapargs.context_layer..ALPHA_GROUP, "hidden").hidden
  gre.set_group_attrs(mapargs.context_layer..ALPHA_GROUP, { hidden = 1 - alpha_group_hidden })
  gre.set_group_attrs(mapargs.context_layer..NUMERIC_GROUP, { hidden = alpha_group_hidden })
end

local gSymbolMap1 = { -- main special symbol set
  string.char(0x2B)          , -- +
  string.char(0xC3,0x97)     , -- ×
  string.char(0xC3,0xB7)     , -- ÷
  string.char(0x3D)          , -- =
  string.char(0x2F)          , -- /
  string.char(0x5F)          , -- _
  string.char(0xE2,0x82,0xAC), -- €
  string.char(0xC2,0xA3)     , -- £
  string.char(0xC2,0xA5)     , -- ¥
--string.char(0xE2,0x82,0xA9), -- ₩, this character is not supported by our font
  string.char(0xC6,0x92)     , -- ƒ, replacement for ₩
  string.char(0x21)          , -- !
  string.char(0x40)          , -- @
  string.char(0x23)          , -- #
  string.char(0x24)          , -- $
  string.char(0x25)          , -- %
  string.char(0x5E)          , -- ^
  string.char(0x26)          , -- &
  string.char(0x2A)          , -- *
  string.char(0x28)          , -- (
  string.char(0x29)          , -- )
  string.char(0x2D)          , -- -
  string.char(0x27)          , -- '
  string.char(0x22)          , -- "
  string.char(0x3A)          , -- :
  string.char(0x3B)          , -- ;
  string.char(0x2C)          , -- ,
  string.char(0x3F)          , -- ?
}

local gSymbolMap2 = { -- alternative special symbol set
  string.char(0x60)          , -- `
  string.char(0x7E)          , -- ~
  string.char(0x5C)          , -- \
  string.char(0x7C)          , -- |
  string.char(0x3C)          , -- <
  string.char(0x3E)          , -- >
  string.char(0x7B)          , -- {
  string.char(0x7D)          , -- }
  string.char(0x5B)          , -- [
  string.char(0x5D)          , -- ]
  string.char(0xC2,0xB0)     , -- °
--string.char(0xE2,0x97,0x8C), -- ◌, this character is not supported by our font
--string.char(0xE2,0x97,0x8B), -- ○, this character is not supported by our font
--string.char(0xE1,0x97,0x8F), -- ●, this character is not supported by our font
--string.char(0xE2,0x96,0xA1), -- □, this character is not supported by our font
--string.char(0xE2,0x96,0xA0), -- ■, this character is not supported by our font
--string.char(0xE2,0x99,0xA4), -- ♤, this character is not supported by our font
--string.char(0xE2,0x99,0xA1), -- ♡, this character is not supported by our font
--string.char(0xE2,0x88,0xA2), -- ♢, this character is not supported by our font
--string.char(0xE2,0x88,0xA7), -- ♧, this character is not supported by our font
--string.char(0xE2,0x98,0x86), -- ☆, this character is not supported by our font
--string.char(0xE2,0x96,0xAA), -- ▪, this character is not supported by our font
  string.char(0xE2,0x80,0xA2), -- •, replacement for ◌
  string.char(0xC3,0xA5)     , -- å, replacement for ○
  string.char(0xC3,0x85)     , -- Å, replacement for ●
  string.char(0xC3,0xA6)     , -- æ, replacement for □
  string.char(0xC3,0x86)     , -- Æ, replacement for ■
  string.char(0xC3,0xB8)     , -- ø, replacement for ♤
  string.char(0xC3,0x98)     , -- Ø, replacement for ♡
  string.char(0xC4,0x91)     , -- đ, replacement for ♢
  string.char(0xC4,0x90)     , -- Đ, replacement for ♧
  string.char(0xE2,0x88,0x9E), -- ∞, replacement for ☆
  string.char(0xE2,0x89,0x88), -- ≈, replacement for ▪
  string.char(0xC2,0xA4)     , -- ¤
  string.char(0xC2,0xAB)     , -- «
  string.char(0xC2,0xBB)     , -- »
  string.char(0xC2,0xA1)     , -- ¡
  string.char(0xC2,0xBF)     , -- ¿
}

--
-- enable or disable caps lock
--
local function toggle_capslock()
  gShifted = not gShifted
  
  local data = {}
  
  local alpha = "abcdefghijklmnopqrstuvwxyz"
  alpha:gsub(".", function(c)
    local key = gre.get_value("Keyboard_Layer_v2.alpha_group.key_"..c..".text")
    data["Keyboard_Layer_v2.alpha_group.key_"..c..".text"] = gShifted and string.upper(key) or string.lower(key)
  end)
  
  for index, symbol in ipairs(gSymbolMap1) do
    data["Keyboard_Layer_v2.special_keys_group.key_s"..index..".text"] = gShifted and gSymbolMap2[index] or symbol
  end
  
  data["Keyboard_Layer_v2.control_group.key_shift.image"] = gShifted and image_path("toets-cap-pressed-x0-y614.png") or image_path("toets-cap-x0-y614.png")
  
  gre.set_data(data)
end

--
-- switch between alpha numeric view and special character view
--
local function toggle_special_characters()
  gSwitched = not gSwitched
  
  local data = {}
  
  data["Keyboard_Layer_v2.alpha_group.grd_hidden"] = gSwitched
  data["Keyboard_Layer_v2.special_keys_group.grd_hidden"] = not gSwitched
  data["Keyboard_Layer_v2.control_group.key_switch.image"] = gSwitched and image_path("toets-generic-pressed.png") or image_path("toets-generic.png")
  data["Keyboard_Layer_v2.control_group.key_switch.x"] = gSwitched and 2 or 0
  data["Keyboard_Layer_v2.control_group.key_switch.y"] = gSwitched and 2 or 0
  
  gre.set_data(data)
end

-- map of actions for special keys
local gKeyboardActionMap = {
  ["Switch"   ] = SwitchAlphaNumeric, -- support for old keyboard
  ["Shift"    ] = KeyboardShift,      -- support for old keyboard
  ["Backspace"] = function(mapargs) TriggerRaw(8) end,
  ["Space"    ] = function(mapargs) TriggerRaw(32) end,
  ["Period"   ] = function(mapargs) TriggerKey(".") end,
  ["Comma"    ] = function(mapargs) TriggerKey(",") end,
  ["Colon"    ] = function(mapargs) TriggerKey(":") end,
  ["Plus"     ] = function(mapargs) gre.send_event("keyboard_plus") end,
  ["Minus"    ] = function(mapargs) gre.send_event("keyboard_minus") end,
  ["TAB"      ] = function(mapargs) gre.send_event("keyboard_tab") end,
  ["SPACE"    ] = function(mapargs) TriggerRaw(32) end,
  ["BACKSPACE"] = function(mapargs) TriggerRaw(8) end,
  ["SHIFT"    ] = toggle_capslock,
  ["SWITCH"   ] = toggle_special_characters
}

--
-- handle a virtual keyboard button press event
--
--- @param gre#context mapargs
function CBKeyboardPress(mapargs) -- support for old keyboard
  local data = {}

  if (mapargs.context_control == nil) then
    return
  end
  
  data = SplitString(mapargs.context_control, ".")
  local key_name = data[3]
  
  local action = gKeyboardActionMap[key_name] or function()
    TriggerKey(gShifted and key_name or string.lower(key_name))
  end
  action(mapargs)
end

function CBKeyboard_OnKeyPressed(mapargs)
  if Settings:get("KEYBOARD_BEEP") ~= 0 then
    Event:audioprog_touch()
  end
 
  local key = mapargs.key 
  local action = gKeyboardActionMap[key] or function()
    TriggerKey(gShifted and string.upper(key) or string.lower(key))
  end
  action(key)
end

function CBKeyboard_OnShiftOutbound(mapargs)
  gre.set_value("Keyboard_Layer_v2.control_group.key_shift.image", gShifted and image_path("toets-cap-pressed-x0-y614.png") or image_path("toets-cap-x0-y614.png"))
end

function CBKeyboard_OnSwitchOutbound(mapargs)
  local data = {}
  data["Keyboard_Layer_v2.control_group.key_switch.image"] = gSwitched and image_path("toets-generic-pressed.png") or image_path("toets-generic.png")
  data["Keyboard_Layer_v2.control_group.key_switch.x"] = gSwitched and 2 or 0
  data["Keyboard_Layer_v2.control_group.key_switch.y"] = gSwitched and 2 or 0
  gre.set_data(data)
end

function CBKeyboard_OnCancel(mapargs)
  gre.send_event("keyboard_cancel")
end

function CBKeyboard_OnOk(mapargs)
  gre.send_event("keyboard_confirm")
end
