-------------------------------------------------------
-- internationalization, support for multiple languages
-------------------------------------------------------

local json = require("json")

-- internationalization singleton class
i18n = {}
i18n.__index = i18n

i18n.available_translations = nil -- map of available translations
i18n.active_language = nil        -- currently active language
i18n.active_translation = nil     -- database with translated strings in currently active language
i18n.english = nil                -- english always available as fallback

--
-- read and return the entire contents of a file
--
local function read_file(file)
  local f = assert(io.open(file, "r"))
  local t = f:read("*all")
  f:close()
  return t
end

--
-- parse the json file with a list of available languages
--
local function initialize_translations(file)
  local json_data = json.decode(read_file(translation_path("languages.json")))

  local translations = {}
  for key, value in pairs(json_data) do
    translations[value.display_name] = {
      key = key,
      id = value.id,
      icon = translation_path(value.icon),
      file = translation_path(value.file)
    }
  end
  return translations
end

--
-- parse the json file with the requested translation
--
local function load_translation(translation_file)
  local json_data = json.decode(read_file(translation_file))

  -- strings that are directly set in the gui
  local gdata = {}
  for key, value in pairs(json_data.ui) do
    gdata[key] = value
  end
  gre.set_data(gdata)

  -- manual lookup table
  local tdata = {}
  for key, value in pairs(json_data.messages or {}) do
    tdata[key] = value
  end
  return tdata
end

--
-- initialize the translations database; make a list of available languages
--
function i18n:init()
  self.available_translations = initialize_translations()
  self.english = load_translation(translation_path("English_UK.json"))
end

--
-- get an ordered list of available languages
--
function i18n:translations()
  local translations = {}
  for translation, _ in pairs(self.available_translations) do
    table.insert(translations, translation)
  end
  table.sort(translations, function(a, b) return self.available_translations[a].key < self.available_translations[b].key end)
  return translations
end

--
-- get the meta data for a language
--
function i18n:meta_data(language)
  return self.available_translations[language]
end

--
-- change the currently active language
--
function i18n:load(language)
  load_translation(translation_path("English_UK.json"))
  self.active_translation = load_translation(self.available_translations[language].file)
  self.active_language = language
end

--
-- get currently active language
--
function i18n:language()
  return self.active_language
end

--
-- get the correct translation text for key
-- return the translated string or key if it's not found
--
local gReportedMissing = {}
function i18n:get(key)
  if (not key or not self.active_translation[key]) and not gReportedMissing[key] then
    print("No text for key:[" .. tostring(key) .. "],using UK text.")
    gReportedMissing[tostring(key)] = true
  end
  return self.active_translation[key] or self.english[key] or key or "[?]"
end

--
-- convert a language id to a language name
-- returns the language for the id or nil of it's not found
--
function i18n:id_to_name(id)
  for name, value in pairs(self.available_translations) do
    if id == value.id then
      return name
    end
  end
  return nil
end

--
-- convert a language name to a language id
-- returns the id for the language or nil of it's not found
--
function i18n:name_to_id(lang)
  return self.available_translations[lang].id
end
